/* STRING.CPP */

/****************************************************************************/
/* General purpose string class to replace C char arrays.
/* Includes operators =, +, +=, ==, !=, <, <=, >, >=, [], length,
/* upper/lower case conv., substring. All operations work with C-strings,
/* as well. Case-sensitivity in comparisons may be toggled.
/*
/* Steven L. Pearson
/* 11/30/93
/****************************************************************************/

#include <string.h>		// Borland C-string header
#include <process.h>
#include "string.hpp"	// string class header

// Case sensitivity in comparisons defaults to TRUE;
bool string::caseSensitive = T;

// Length of string, not including terminator
int string::length() const {
	return ::strlen(cString);
}

// Default constructor
string::string() {
	cString = new char[1];
	if(cString==NULL) exit(-1);
	cString[0] = '\0';			// Init. to ""
}

// Copy constructor
string::string(const string& s)
{
	cString = new char[s.length()+1];
	if(cString==NULL) exit(-1);
	::strcpy(cString, s.cString);
}

// Assignment operator
// Works for C-strings by automatically calling C-string constructor.
string& string::operator=(const string& s)
{
	delete[] cString;
	cString = new char[s.length()+1];
	if(cString==NULL) exit(-1);
	::strcpy(cString, s.cString);
	return *this;
}

// Destructor
string::~string() {
		delete[] cString;
}

// C-string constructor
string::string(const char* s) {
	cString = new char[::strlen(s)+1];
	if(cString==NULL) exit(-1);
	::strcpy(cString, s);
}

// Convert to uppercase
string string::toUpper(void) {
	return strupr( cString );
}

// Convert to lowercase
string string::toLower(void) {
	return strlwr( cString );
}

// Substring
// Indexing starts at 1 not 0. Easily changed for you digit-heads. :-)
string string::subString(int start, int end) {
	int subStrLen = end-start+1+1;	// # of chars plus terminator
	char *buf = new char[subStrLen];
	buf[0] = '\0';
	if( start >= 1 && end <= length() && start <= end ) {	/* Bounds check */
		strncpy( buf, cString+start-1, subStrLen-1 );
		buf[subStrLen-1] = '\0';
	} // Otherwise return ""
	string retVal(buf);
	delete[] buf;
	return retVal;
}

// Catenation with another string
string string::operator+(const string& s) const {
	char *buf = new char[s.length()+length()+1];
	if(buf==NULL) exit(-1);
	::strcpy(buf, cString);
	::strcat(buf, s.cString);
	string retval(buf);
	delete[] buf;
	return retval;
}

// Catenation with a C-string
string string::operator+(const char* s) const {
	return *this + string(s);	// Invokes string::operator+(const string&)
}

// Catenation to self
// Works for C-strings by automatically invoking C-string constructor
string& string::operator+=(const string& s)
{
	*this = *this + s;
	return *this;
}

// Test for equality with another string
bool string::operator==(const string& s) {
	if( caseSensitive )
		return (bool) !strcmp( (const char*) *this, (const char*) s );
	else
		return (bool) !strcmpi( (const char*) *this, (const char*) s );
}

// Test for equality with a C-string
bool string::operator==(const char* s) {
	return (*this==string(s));	// Invokes string::operator(const string&)
}

// Test for inequality with another string
bool string::operator!=(const string& s) {
	if( caseSensitive )
		return (bool) strcmp( (const char*) *this, (const char*) s );
	else
		return (bool) strcmpi( (const char*) *this, (const char*) s );
}

// Test for equality with a C-string
bool string::operator!=(const char* s) {
	return (*this!=string(s));
}

// Test for < with another string
bool string::operator<(const string& s) {
	if( caseSensitive )
		return (bool) ( strcmp( (const char*) *this, (const char*) s ) < 0 );
	else
		return (bool) ( strcmpi( (const char*) *this, (const char*) s ) < 0 );
}

// Test for < with a C-string
bool string::operator<(const char* s) {
	return (*this<string(s));
}

// Test for <= with another string
bool string::operator<=(const string& s) {
	return (bool) (*this < s || *this == s);
}

// Test for <= with a C-string
bool string::operator<=(const char* s) {
	return (*this<=string(s));
}

// Test for > with another string
bool string::operator>(const string& s) {
	if( caseSensitive )
		return (bool) ( strcmp( (const char*) *this, (const char*) s ) > 0 );
	else
		return (bool) ( strcmpi( (const char*) *this, (const char*) s ) > 0 );
}

// Test for > with a C-string
bool string::operator>(const char* s) {
	return (*this>string(s));
}

// Test for >= with another string
bool string::operator>=(const string& s) {
	return (bool) (*this > s || *this == s);
}

// Test for >= with a C-string
bool string::operator>=(const char* s) {
	return (*this>=string(s));
}

// Subscript like C-string
// Can be used as l-value or r-value. Index starts at 1, not 0!
char& string::operator[](int i) const {
	// Return first char if out-of-bounds
	// This is obviously not satisfactory, but I need to return a ref
   // to char.  What else can I use?
	return ( ( i>=1 && i<=length() ) ? cString[i-1] : cString[0] );
}

// Publish as C-string
string::operator const char *() const {
	return cString;
}


// Globally overloaded operators (but not friends)

// Catenation when leftmost operand is a C-string
string operator+(const char* s1, const string& s2) {
	return string(s1) + s2; // Invokes string::operator+(const string&)
}

// Compare for equality when leftmost operand is a C-string
bool operator==(const char* s1, const string& s2) {
	return string(s1) == s2; // Invokes string::operator==(const string&)
}

// Compare for inequality when leftmost operand is a C-string
bool operator!=(const char* s1, const string& s2) {
	return string(s1) != s2; // Invokes string::operator!=(const string&)
}

// Compare for < when leftmost operand is a C-string
bool operator<(const char* s1, const string& s2) {
	return string(s1) < s2;
}

// Compare for <= when leftmost operand is a C-string
bool operator<=(const char* s1, const string& s2) {
	return string(s1) <= s2;
}

// Compare for > when leftmost operand is a C-string
bool operator>(const char* s1, const string& s2) {
	return string(s1) > s2;
}

// Compare for >= when leftmost operand is a C-string
bool operator>=(const char* s1, const string& s2) {
	return string(s1) >= s2;
}

