//
// CPORT.HPP
//
// C++ Extention Header file for Cport Communications Library
//
// Copyright (c) 1993 Bri Productions
//
//


#ifndef _CPORT_HPP_
#define _CPORT_HPP_


//-------------------------------------
//
// Cport class
//
//-------------------------------------
class Cport
{
      // Default values
      //
   enum   {
            DEF_RXQ  = 1024,
            DEF_TXQ  = 512,
            DEF_MODE = W8|S1|NONE
         };

   int   open_err;   // open errors
   COM   c;          // serial port handle
   int*  ref_cnt;    // reference count

public:

      // Constructors and destructor
      //
   Cport(unsigned id, 
         int baud, 
         byte mode = DEF_MODE, 
         unsigned rxQ = DEF_RXQ, 
         unsigned txQ = DEF_TXQ,
         byte htype = OFF
         );
   Cport(unsigned id, 
         int baud, 
         byte mode, 
         unsigned rxQ, 
         unsigned txQ,
         byte htype,
         unsigned thresh
         );
   Cport  (const CPARAM& param);
   ~Cport ();


      // Copy constructor and assignment
      //
   Cport            (Cport& cport);
   Cport& operator= (Cport& cport);


      // Typecasts
      //
   operator int()    { return(open_err); }
   operator COM()    { return(c);        }
   int ref()         { return(*ref_cnt); } // debug


      // Control
      //
   void     Handshake (byte htype, unsigned thresh);
   void     Param     (CPARAM& param);
   void     Baud      (int baud);     
   void     Mode      (byte mode);    
   int      Tx        (int cmnd);  
   void     NS550     (int trigger);  
   static void Turbo  (int options);
   unsigned RxQ       (unsigned size);
   unsigned TxQ       (unsigned size);


      // Input 
      // 
   char     Get     ();
   unsigned LenRx   ();
   char*    Get     (char *str, int maxc, char termc = '\n');
   unsigned In      (void *abyte, unsigned nbyte);
   void     FlushRx ();
   char     Peek    ();
   unsigned RxScan  (char ch);


      // Output 
      //
   int      Put     (char _c);
   unsigned LenTx   ();
   int      Put     (const char *str);
   unsigned Out     (const void *abyte, unsigned nbyte);
   void     FlushTx ();
   void     TxWait  ();


      // Status 
      //
   unsigned Error  ();
   unsigned Status ();
   void     Out1   (byte on_off);
   void     Rts    (byte on_off);
   void     Dtr    (byte on_off);
   int      Uart   ();


      // Misc fuctions
      //
   void SetBreak ();
   void ClrBreak ();
   void Scratch  (byte abyte);
   byte Scratch  ();

   static byte          Checksum (const void *abyte, unsigned nbyte);
   static unsigned      Crc16    (const void *abyte, unsigned nbyte);
   static unsigned long Crc32    (const void *abyte, unsigned nbyte);

};


//-------------------------------------
//
// Constructors and destructor
//
//-------------------------------------

inline Cport::Cport(unsigned id, int baud, byte mode, 
                                       unsigned rxQ, unsigned txQ, byte htype)
{
   *(ref_cnt = new int) = 1;
   c = ComOpen(id, baud, mode, rxQ, txQ);
   if((open_err = comopen_errno) == NO_ERR && htype != OFF)
      ComHandshake(c, htype, (rxQ * 3) / 4);
}

inline Cport::Cport(unsigned id, int baud, byte mode, 
                     unsigned rxQ, unsigned txQ, byte htype, unsigned thresh)
{
   *(ref_cnt = new int) = 1;
   c = ComOpen(id, baud, mode, rxQ, txQ);
   if((open_err = comopen_errno) == NO_ERR && htype != OFF)
      ComHandshake(c, htype, thresh);
}

inline Cport::Cport(const CPARAM& param)
{
   *(ref_cnt = new int) = 1;
   c = ComOpenS(&param);
   open_err = comopen_errno;
}

inline Cport::~Cport()
{
   if(!(--(*ref_cnt)))
   {
      if(open_err == NO_ERR)
         ComClose(c);
      delete ref_cnt;
   }
}


//-------------------------------------
//
// Copy constructor and assignment
//
//-------------------------------------

inline Cport::Cport(Cport& cport)
{
   open_err = cport.open_err;
   c = cport.c;
   (*(ref_cnt = cport.ref_cnt))++;
}

inline Cport& Cport::operator=(Cport& cport)
{
   if(!(--(*ref_cnt)))
	{
      if(open_err == NO_ERR)
    	  ComClose(c);
      delete ref_cnt;
	}

   open_err = cport.open_err;
   c = cport.c;
   (*(ref_cnt = cport.ref_cnt))++;
   return(*this);
}


//-------------------------------------
//
// Control
//
//-------------------------------------

inline void     Cport::Handshake (byte htype, unsigned thresh)
                                       { ComHandshake(c, htype, thresh); }
inline void     Cport::Param (CPARAM& param) { ComParam(c, &param);      }
inline void     Cport::Baud  (int baud)      { ComBaud(c, baud);         }
inline void     Cport::Mode  (byte mode)     { ComMode(c, mode);         }
inline int      Cport::Tx    (int on_off)    { return(ComTx(c, on_off)); }
inline void     Cport::NS550 (int trigger)   { ComNS550(c, trigger);     }
inline void     Cport::Turbo (int options)   { ComTurbo(options);        }
inline unsigned Cport::RxQ   (unsigned size) { return(ComRxQ(c,size));    }
inline unsigned Cport::TxQ   (unsigned size) { return(ComTxQ(c,size));    }


//-------------------------------------
//
// Input 
//
//-------------------------------------

inline char     Cport::Get     ()                { return(ComGetc(c));       }
inline unsigned Cport::LenRx   ()                { return(ComLenRx(c));      }
inline char*    Cport::Get     (char *str, int max_c, char term_c) 
                               { return(ComGets(c, str, max_c, term_c));     }
inline unsigned Cport::In      (void *abyte, unsigned n_byte) 
                                      { return(ComIn(c, abyte, n_byte));     }
inline void     Cport::FlushRx ()                { ComFlushRx(c);            }
inline char     Cport::Peek    ()                { return(ComPeek(c));       }
inline unsigned Cport::RxScan  (char ch)         { return(ComRxScan(c, ch)); }


//-------------------------------------
//
// Output
//
//-------------------------------------

inline int      Cport::Put     (char _c)         { return(ComPutc(c,_c ));  }
inline unsigned Cport::LenTx   ()                { return(ComLenTx(c));     }
inline int      Cport::Put     (const char *str) { return(ComPuts(c, str)); }
inline unsigned Cport::Out     (const void *abyte, unsigned n_byte)
                                   { return(ComOut(c, abyte, n_byte));      }
inline void     Cport::FlushTx ()                { ComFlushTx(c);           }
inline void     Cport::TxWait  ()                { ComTxWait (c);           }


//-------------------------------------
//
// Status functions 
//
//-------------------------------------

inline unsigned Cport::Error   ()            { return(ComError(c));     }  
inline unsigned Cport::Status  ()            { return(ComStatus(c));    }
inline void     Cport::Out1    (byte on_off) { ComMcr(c, on_off, 0x04); }
inline void     Cport::Rts     (byte on_off) { ComMcr(c, on_off, 0x02); }
inline void     Cport::Dtr     (byte on_off) { ComMcr(c, on_off, 0x01); }
inline int      Cport::Uart    ()            { return(ComUart(c));      }


//-------------------------------------
//
// Misc fuctions
//
//-------------------------------------

inline void     Cport::SetBreak ()         { ComSetBreak (c);         }
inline void     Cport::ClrBreak ()         { ComClrBreak (c);         }
inline void     Cport::Scratch(byte abyte) { ComPutScrtch(c, abyte);  }
inline byte     Cport::Scratch()           { return(ComGetScrtch(c)); }


inline byte          Cport::Checksum (const void *abyte, unsigned nbyte)
		 			               { return(ComChecksum(abyte, nbyte)); }
inline unsigned      Cport::Crc16    (const void *abyte, unsigned nbyte)
		 			               { return(ComCrc16(abyte, nbyte));    }
inline unsigned long Cport::Crc32    (const void *abyte, unsigned nbyte)
								               { return(ComCrc32(abyte, nbyte));    }

#endif   // CPORT.HPP

