//JDHEAP.CPP
// part of JDHEAP class by John M. Dlugosz
// This is posted for academic pursuits, and may
// be used freely, with attribution.

// this is the main part of my heap class.  This is
// code to allocate, free, and manage a heap.

#include "usual.h"
#include "except.h"
#include "jdheap.h"

unsigned short jdheap::current_counter= 0;
static const char FNAME[]= "jdheap\\"__FILE__;

#define error(x,y) error (memory_failure::x,y)

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

inline ushort jdheap::next_count()
{
if (++current_counter == 0)  current_counter= 1;
return current_counter;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

static void NORETURN error (memory_failure::code_t code, int line)
{
throw memory_failure (code, FNAME, line);
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

void jdheap::prepare_unused_node_list()
{
// this links all nodes into a freelist
const max= capacity-1;
for (int loop= 1;  loop < max;  loop++) {
   list[loop].next= loop+1;
   list[loop].prev= -1;
   }
list[max].next= -1;
list[max].prev= -1;
freehead= 1;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

jdheap::jdheap (record* list, int listcap, void* block, unsigned long blocksize)
: list(list), capacity(listcap)
{
Start= to_linear(block);
End= Start + blocksize;
CRC_mode= FALSE;
min_free_size= 4;
prepare_unused_node_list();
// set up the list with 1 free item holding all space
listsize= 1;
record* p= list+0;
p->pointer= Start;
p->size= blocksize;
p->mark_string= 0;  //empty
p->next= p->prev= 0;  //point to itself, circular both ways
// counter is 0 to indicate a free item
p->counter= 0;
rover= 0;  //point to the one and only item
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */
/*     allocate memory                                */
/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

static void NORETURN memerror (ulong size, int line)
{
throw memory_error (size, FNAME, line);
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */


#if FALSE

static int seek_for_fit (jdheap::record* list, int listsize, int x, ulong size)
{
// this is isolated, for eventual replacement with hand optomized asm
// also, it generates better code if put in its own function!
const jdheap::record* p;
for (;listsize--;x=p->next) {
   p= list+x;
   if (p->counter != 0)  continue;  //not a free item
   if (p->size >= size)  return x;  //found one!
   }
return -1;  //did not find one
}

#else
extern "C" int far __cdecl seek_for_fit (jdheap::record* list, int listsize, int x, ulong size);
#endif

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

int jdheap::first_fit (ulong size)
{
int retval= seek_for_fit (list, listsize, rover, size);
if (retval == -1)  memerror (size, __LINE__);
rover= retval;
return retval;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

static void* stamp (int x, jdheap::record* rec)
{
ushort* p= (ushort*)jdheap::from_linear(rec->pointer);
*p= x;
return p+1;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

void* jdheap::split (int x, record* p, ulong size)
{
// this will split the specified node and finish alloc'ing on the split off one.
// first step is to allocate the other node
listsize++;
int x2= freehead;
if (x2 == -1)  error (EmptyList, __LINE__);
record* p2= list+x2;
freehead= p2->next;
// make this node take the specified part of the original
p2->pointer= p->pointer;
p->pointer += size;
p->size -= size;
p2->size= size;
p2->mark_string= 0;
p2->counter= next_count();
// do the linked list stuff
p2->prev= p->prev;
p2->next= x;
p->prev= x2;
list[p2->prev].next=x2;
return stamp (x2,p2);
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

void* jdheap::alloc (ulong size)
{
// add 2, and round up to even size at the same time
size += 3;
size &= ~1;
int x= first_fit (size);  //could change strategy later
record* p= list+x;
if (p->size  <= size + min_free_size) {  //take the whole thing
   // transform this node from free to used
   p->counter= next_count();
   p->mark_string= 0;
   return stamp (x,p);
   }
// take part of it
return split (x,p,size);
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */
/*   free memory                                      */
/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

void jdheap::remove_node (record& r, int x2)
{
// unlink from its surroundings
int prev= r.prev;
int next= r.next;
list[prev].next= next;
list[next].prev= prev;
// add to unused node list
r.next= freehead;
r.prev= -1;  //show as unused node
freehead= x2;
listsize--;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

void jdheap::free (void* mempointer_)
{
if (!mempointer_)  return;
int* memptr= ((int*)mempointer_)-1;
int x= *memptr;
record& r= list[x];
if (r.pointer != to_linear(memptr))  error (BadPtr1, __LINE__);
if (r.counter == 0)  error (BadPtr2, __LINE__);
// turn into a free node
r.counter= 0;
//check for merging nodes
int x2= r.prev;
record* p2= list+x2;    //previous node
if (p2->isfree())  {
   // check for correct region.  I have to watch for the "seam", but this
   // will also suppress bad frees if the heap got messed up somehow.
   if (p2->pointer+p2->size == r.pointer) {
      r.pointer= p2->pointer;
      r.size += p2->size;
      remove_node (*p2,x2);
      }
   }
x2= r.next;
p2= list+x2;  //next node
if (p2->isfree()) {
   if (r.pointer+r.size == p2->pointer) {
      r.size += p2->size;  //enlarge to cover both
      remove_node (*p2,x2);
      }
   }
rover= x;  //reposition to a known good point, because the merging might
           //have invalidated its value.
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */
/*     utility                                        */
/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

int jdheap::get_node (const void* mempointer) const
{
int* memptr= ((int*)mempointer)-1;
int x= *memptr;
record& r= list[x];
if (r.pointer != to_linear(memptr))  error (BadPtr1, __LINE__);
if (r.counter == 0)  error (BadPtr2, __LINE__);
return x;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

unsigned long jdheap::size (const void* p) const
{
record& r= list[get_node(p)];
return r.size-2;  //subtract out its own header
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

void jdheap::mark (const void* p, const char* s)
{
record& r= list[get_node(p)];
r.mark_string= s;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */

const char* jdheap::mark (const void* p) const
{
record& r= list[get_node(p)];
return r.mark_string;
}

/* /\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\/\ */
