// ===================================================================
// RBLURRY.CPP
//	Illustrate blurry reflection.
// Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================


#include "oort.h"
#include "world.h"
#include "colors.h"

Texture *
RedMarble()
{
    ColorMap *cmap = new ColorMap;
    cmap->AddEntry(new LinearColorMapEntry(0, 0.8,
					    RGBColor(0.8, 0.8, 0.6),
					    RGBColor(0.8, 0.4, 0.4)));
    cmap->AddEntry(new LinearColorMapEntry(0.8, 1.0,
					    RGBColor(0.8, 0.4, 0.4),
                                            RGBColor(0.8, 0.2, 0.2)));
    return new Marble(IdentityMatrix(), 1.0, cmap);
}

Texture *
BlueMarble()
{
    ColorMap *cmap = new ColorMap;
    cmap->AddEntry(new OneColorMapEntry(0, 0.5, RGBColor(0.3, 0.3, 0.5)));
    cmap->AddEntry(new LinearColorMapEntry(0.5, 0.55, RGBColor(0.3, 0.3, 0.5), RGBColor(0.2, 0.2, 0.3)));
    cmap->AddEntry(new LinearColorMapEntry(0.55, 0.6, RGBColor(0.2, 0.2, 0.3), RGBColor(0.25, 0.25, 0.35)));
    cmap->AddEntry(new LinearColorMapEntry(0.6, 0.7, RGBColor(0.25, 0.25, 0.35), RGBColor(0.15, 0.15, 0.26)));
    cmap->AddEntry(new LinearColorMapEntry(0.7, 0.8, RGBColor(0.15, 0.15, 0.26), RGBColor(0.1, 0.1, 0.2)));
    cmap->AddEntry(new LinearColorMapEntry(0.8, 0.9, RGBColor(0.1, 0.1, 0.2), RGBColor(0.3, 0.3, 0.5)));
    cmap->AddEntry(new LinearColorMapEntry(0.9, 1.0, RGBColor(0.3, 0.3, 0.5), RGBColor(0.1, 0.1, 0.2)));
    return new Marble(ScaleMatrix(0.3, 0.3, 0.3), 0.3, cmap);
}

Texture *
WhiteMarble()
{
    ColorMap *cmap = new ColorMap;
    cmap->AddEntry(new OneColorMapEntry(0, 0.3, White));
    cmap->AddEntry(new LinearColorMapEntry(0.3, 0.7, White, RGBColor(0.6)));
    cmap->AddEntry(new LinearColorMapEntry(0.7, 0.9, RGBColor(0.6), RGBColor(0.45)));
    cmap->AddEntry(new LinearColorMapEntry(0.9, 1.0, RGBColor(0.45), RGBColor(0.3)));
    return new Marble(ScaleMatrix(0.2, 0.2, 0.2), 1.0, cmap);  
}

extern void printhist();

void
PopulateWorld(World& world)
{
    world.SetDepthLimit(3);
    world.SetOutputFile("rblurry.raw");

    // The World is lit by ambient light only, so no shadow rays are cast.

    world.SetViewerParameters(Vector3D(0), Vector3D(0, 1, 12), Vector3D(0, 1, 0));
    world.SetScreenWidth(3);
    world.SetScreenHeight(1.5);
    world.SetScreenDistance(2);
    world.SetAmbientLight(RGBColor(1));
    world.SetBackgroundColor(RGBColor(0, 1, 0));

    float angs[] = {0.04, 0.08, 0.12, 0.16, 0.2};
    for (int i = 0; i < 5; i++) {
	DistributedHall *surf = new DistributedHall;
	surf->SetReflect(new PureColor(RGBColor(0.8)));
	surf->SetReflectParms(DistributedHall::DistribParms(4, 8, 10, angs[i]));
	world.AddObject(new Sphere(Vector3D(3 * (i - 2), 0, 0), 1.0, surf));
    }

    Texture *ChessBoard = new Checkerboard(0.1, 0.1,
					   new PureColor(Aquamarine),
                                           new PureColor(CornflowerBlue),
					   IdentityMatrix(),
					   new SphericalMapping(Vector3D(0, 0, 12)));
    HallSurface *checks = new HallSurface;
    checks->SetBumpMap(new ReverseNormal);
    checks->SetDiffuse(ChessBoard);
    checks->SetAmbient(ChessBoard);
    world.AddObject(Sphere(Vector3D(0, 0, 12), 15, checks));
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}
