// ===================================================================
// TEXTS.CPP
//	Illustrates a number of textures.
// Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================

#include "oort.h"
#include "world.h"
#include "colors.h"
#include "utils.h"

void
PopulateWorld(World& world)
{
    world.SetDepthLimit(3);
    world.SetOutputFile("texts.raw");

    world.SetViewerParameters(Vector3D(0), Vector3D(0, 3.5, 15), Vector3D(0, 1, 0));
    world.SetScreenWidth(8);
    world.SetScreenHeight(4);
    world.SetScreenDistance(4);

    world.SetAmbientLight(RGBColor(1));
    world.SetBackgroundColor(RGBColor(0));

    // Three light sources.
    world.AddLight(new PointLight(Vector3D(-20, 40, -80), RGBColor(1./3), 1, 0, 0));
    world.AddLight(new DirectLight(Vector3D(0, 1, 0), RGBColor(1./3)));
    world.AddLight(new PointLight(Vector3D(30, 40, 80), RGBColor(1./3), 1, 0, 0));

    // Create a table textured with WoodUpstill, then put a white ring on it.
    {
	float min = -20;
        float max = 20;
	float legdiff = 2, legwid = 3;

	legwid += legdiff;
 
	// Table surface.
	HallSurface *newsurf = new HallSurface;
	newsurf->SetDiffuse(new WoodUpstill(RotationYMatrix(45*M_PI/180), 0.2));
	newsurf->SetSpecular(new PureColor(RGBColor(0.6)), 40); 

	// Create the table
	Aggregate *table = new Aggregate;
	table->AddObject(MakeBox(Vector3D(min, -6, min), Vector3D(max, -5, max), newsurf));
        newsurf = new HallSurface;
	newsurf->SetDiffuse(new WoodUpstill(RotationXMatrix(M_PI/2) * RotationYMatrix(45*M_PI/180), 0.2));
	newsurf->SetSpecular(new PureColor(RGBColor(0.6)), 10);
	table->AddObject(MakeBox(Vector3D(min + legdiff, -20, min + legdiff), Vector3D(min + legwid, -5, min + legwid), newsurf));
	table->AddObject(MakeBox(Vector3D(min + legdiff, -20, max - legwid), Vector3D(min + legwid, -5, max - legdiff), newsurf));
	table->AddObject(MakeBox(Vector3D(max - legwid, -20, min + legdiff), Vector3D(max - legdiff, -5, min + legwid), newsurf));
	table->AddObject(MakeBox(Vector3D(max - legwid, -20, max - legwid), Vector3D(max - legdiff, -5, max - legdiff), newsurf));
	table->ApplyTransform(RotationYMatrix(45*M_PI/180) * TranslationMatrix(Vector3D(0, 0, -15)));
	world.AddObject(table);

	// Ring surface: off-white diffuse reflector.
	HallSurface *surf = new HallSurface;
	surf->SetDiffuse(new PureColor(0.8));

	// Add ring to world.
        world.AddObject(new Ring(Vector3D(0, 1, 0), 4.9, Vector3D(0, 0, -15), 17, 18, surf));
    }

    // Use the random number generator to place NumSpheres spheres
    // in the ring.
    const int NumSpheres = 12;
    Sphere *sphs[NumSpheres];
    Vector3D centers[NumSpheres];
    {
	srand(19);
	for (int i = 0; i < NumSpheres; i++) {
	    int goti = 0;
	    do {
		float mag = RandomValue(0, 15);
		float ang = RandomValue(0, 2 * M_PI);
		centers[i] = RotationYMatrix(ang) * Vector3D(1, 0, 0) * mag;
                centers[i] -= Vector3D(0, 4, 15);
		goti = (! i) || (MinDist(centers[i], centers, i - 1) > 5);
            } while (! goti);
	}
	HallSurface *surf = new HallSurface;
        surf->SetDiffuse(new PureColor(1, 0, 0)); 
	for (i = 0; i < NumSpheres; i++) {
	    world.AddObject(sphs[i] = new Sphere(centers[i], 1, surf));
	}
    }
    // Now override the surfaces of the various spheres.
    {
	HallSurface *surf = new HallSurface;
	surf->SetDiffuse(RedMarble());
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100); 
	sphs[0]->OverrideColor(surf);

	surf = new HallSurface;
	surf->SetDiffuse(BlueMarble());
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[1]->OverrideColor(surf);

        surf = new HallSurface;
	surf->SetDiffuse(WhiteMarble());
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[2]->OverrideColor(surf);

	surf = new HallSurface;
	surf->SetDiffuse(BlackMarble());
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[3]->OverrideColor(surf);

	surf = new HallSurface;
	surf->SetDiffuse(CherryWood(IdentityMatrix(), 0.3));
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[4]->OverrideColor(surf);

	surf = new HallSurface;
	surf->SetDiffuse(PineWood(IdentityMatrix(), 0.3));
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[5]->OverrideColor(surf);

	surf = new HallSurface;
	surf->SetDiffuse(DarkWood(IdentityMatrix(), 0.3));
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[6]->OverrideColor(surf);

	surf = new HallSurface;
	surf->SetDiffuse(Clouds(IdentityMatrix(), 0.5));
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[7]->OverrideColor(surf);
        sphs[8]->OverrideColor(MakeReflective());
	sphs[9]->OverrideColor(MakeGranite());
	surf = new HallSurface;
	surf->SetDiffuse(new PureColor(Red));
	surf->SetSpecular(new PureColor(RGBColor(0.6)), 100);
	sphs[10]->OverrideColor(surf);
	sphs[11]->OverrideColor(MakeGlass());
    }
    {
	Texture *check1 = WhiteMarble(ScaleMatrix(0.1, 0.1, 0.1));
	Texture *check2 = BlueMarble(ScaleMatrix(0.35, 0.35, 0.35) * RotationYMatrix(M_PI/2) * TranslationMatrix(Vector3D(1000, 0, 0)));
	Texture *checks = new Checkerboard(20, 20, check1, check2, RotationXMatrix(M_PI/2));
	HallSurface *newsurf = new HallSurface;
	newsurf->SetDiffuse(checks);
	Object3D *backplane = new Plane(0, 1, 0, 20, newsurf);
	world.AddObject(backplane);
    }
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1993 by Nicholas Wilt\n";
    cout << "See COPYRGHT.DOC for details on copyright restrictions.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}

