// ===================================================================
// bump.h
//	Header file for bump-mapping implementations.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#ifndef __BUMP__

#define __BUMP__

// Forward-declare ShadingInfo class.
class ShadingInfo;

// =========================================================
// BumpMap
//	Takes the ShadingInfo for a point--direction of the 
//	incident ray, intersection point, normal and other
//	information--and modifies the intersection point or 
//	the normal, or both.
// =========================================================

class BumpMap {
public:
    virtual ~BumpMap() { }
    virtual void PerturbNormal(ShadingInfo& shade) = 0;
    virtual BumpMap *Dup() const = 0;
    virtual void ApplyTransform(const Matrix& tform) { }
    virtual void PreMulTransform(const Matrix& tform) { }
};

// =========================================================
// BumpMapReference
//	Bump map with a reference matrix attached.
//	Many bump maps, such as ReverseNormal or TwoSided,
//	do not need a reference matrix at all.
// =========================================================
class BumpMapReference : public BumpMap {
public:
    BumpMapReference(const Matrix& _reference = IdentityMatrix());
    virtual ~BumpMapReference() { }

    virtual void PerturbNormal(ShadingInfo& shade) = 0;
    virtual BumpMap *Dup() const = 0;
    virtual void ApplyTransform(const Matrix& tform);
    virtual void PreMulTransform(const Matrix& tform);
protected:
    Matrix reference;

    virtual Vector3D PreprocessVector(const Vector3D& v) const;
};

// =========================================================
// ReverseNormal
//	Reverses the direction of the normal.
// =========================================================
class ReverseNormal : public BumpMap {
public:
    ReverseNormal(BumpMap *a = 0) { bump = a; }
    void PerturbNormal(ShadingInfo& shade);
    virtual BumpMap *Dup() const { return new ReverseNormal(bump); }
private:
    BumpMap *bump;
};


// =========================================================
// TwoSided bump map
//	Ensures the normal is facing the incident direction.
//	If necessary, the normal is negated.
// =========================================================
class TwoSided : public BumpMap {
public:
    TwoSided(BumpMap *a = 0) { bump = a; }
    void PerturbNormal(ShadingInfo& shade);
    virtual BumpMap *Dup() const { return new TwoSided(bump); }
private:
    BumpMap *bump;
};

// =========================================================
// WaveSource
//	Precomputes a set of noise vector values.
// =========================================================
class WaveSource {
public:
    WaveSource(int, NoiseMaker *);

    int N() const { return n; }
    Vector3D GetWave(int inx) const { return waves[inx % n]; }
private:
    int n;
    Vector3D *waves;
};

// =========================================================
// FreqSource
//	Precomputes a set of noise values.
// =========================================================
class FreqSource {
public:
    FreqSource(int, NoiseMaker *);

    int N() const { return n; }
    float GetFreq(int inx) const { return freqs[inx % n]; }
private:
    int n;
    float *freqs;
};


// =========================================================
// Ripples
//	Gives the surface a rippled appearance.
// =========================================================
class Ripples : public BumpMapReference {
public:
    Ripples(WaveSource *WS,
	    FreqSource *FS,
	    float Freq,
	    float Phase,
	    float BumpAmt,
	    const Matrix& _reference): BumpMapReference(_reference) {
	  ws = WS;
	  fs = FS;
	  freq = Freq;
	  phase = Phase;
	  bumpamt = BumpAmt;
      }
    void PerturbNormal(ShadingInfo& shade);
    virtual BumpMap *Dup() const;
private:
    WaveSource *ws;
    FreqSource *fs;
    float freq, phase, bumpamt;
};

// =========================================================
// Dented
//	Gives the surface a dented appearance by
//	modifying the intersection point.
//	An experiment in displacement mapping.
//	From Upstill's _The Renderman Companion_.
// =========================================================
class Dented : public BumpMapReference {
public:
    Dented(float, const Matrix&);
    void PerturbNormal(ShadingInfo& shade);
    virtual BumpMap *Dup() const;
private:
    float Km;
};

#endif

