// ===================================================================
// surface.h
//	Header file for Surface classes.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================


// Forward-declare classes
class World;

// ---------------------------------------------------------
// ShadingInfo 
//	Encapsulates all the information needed to shade a
//	point.  All members are public, so they can be easily 
//	accessed by the various classes that help do
//	shading in OORT.
// ---------------------------------------------------------
class ShadingInfo {
public:
    Vector3D p;			// Point being shaded
    Vector3D normal;		// Normal at the intersection point
    Vector3D incident;		// Incident direction
    Vector3D reflected;		// Direction of perfect reflection

    Object3D *obj;		// Object being shaded
    Ray *ray;			// Incident ray
    World *world;		// World the shading is occurring in
};

// ---------------------------------------------------------
// Surface
//	Surface computes the color at an intersection point
//	using the information in a ShadingInfo structure.
// ---------------------------------------------------------

class Surface {
public:
    virtual ~Surface() { }
    virtual void ApplyTransform(const Matrix&) = 0;
    virtual void PreMulTransform(const Matrix&) = 0;
    virtual Surface *Dup() const = 0;

    virtual int Transparent() = 0;
    virtual RGBColor Transmitted(Vector3D&) = 0;
    virtual void BeginShadeLight(Vector3D& loc) = 0;
    virtual RGBColor ShadeLight(ShadingInfo& shade, Vector3D& dir) = 0; 

    virtual RGBColor ComputeColor(ShadingInfo& shade, int depth) = 0;
};

class HallSurface : public Surface {
public:
    HallSurface() {
	NewNormal = 0;
	Ambient = Diffuse = Specular = Reflect = Transmit = 0;
    }

    virtual void SetBumpMap(BumpMap *);
    virtual void SetAmbient(Texture *);
    virtual void SetDiffuse(Texture *);
    virtual void SetSpecular(Texture *, float);
    virtual void SetTransmit(Texture *, float);
    virtual void SetReflect(Texture *);

    virtual void ApplyTransform(const Matrix&);
    virtual void PreMulTransform(const Matrix&);
    virtual Surface *Dup() const;
    virtual int Transparent();
    virtual RGBColor Transmitted(Vector3D&);
    virtual RGBColor ComputeColor(ShadingInfo& shade, int depth);

    virtual void BeginShadeLight(Vector3D& loc);
    virtual RGBColor ShadeLight(ShadingInfo& shade, Vector3D& dir); 
protected:
    BumpMap *NewNormal; 	// Bump map

    Texture *Ambient;		// Ambient-reflected light
    Texture *Diffuse;		// Diffuse color of the object

    Texture *Specular;		// Index of specular reflection
    float specnum;		// Specularity factor

    Texture *Transmit;		// How much light is transmitted
    float index;		// Index of refraction for object

    Texture *Reflect;		// How much light is reflected

    RGBColor diffuse, specular;	// Computed when BeginShadeLight() called
};

class DistributedHall : public HallSurface {
public:
    class DistribParms {
    public:

	DistribParms(int min = 4, int max = 16, int cand = 10, 
		     float ang = 0, 
		     RGBColor thresh = RGBColor(0.4, 0.3, 0.6)) 
	{
	    MinSamples = min;  
	    MaxSamples = max;
	    Candidates = cand;
	    Angle = ang;
	    Threshold = thresh;
	}

	// Routines for read-only access to parameters.
	int GetMinSamples() const { return MinSamples; }
	int GetMaxSamples() const { return MaxSamples; }
	int GetCandidates() const { return Candidates; }
	float GetAngle() const { return Angle; }
	RGBColor GetThreshold() const { return Threshold; }

	// Routines to write to parameters.
	void SetMinSamples(int n) { MinSamples = n; }
	void SetMaxSamples(int n) { MaxSamples = n; }
	void SetCandidates(int n) { Candidates = n; }
	void SetAngle(float ang) { Angle = ang; }
	void SetThreshold(const RGBColor& x) { Threshold = x; }
    private:
	// Number of samples permitted (default = 4, 16).
	int MinSamples, MaxSamples;

	// Number of candidates to consider when computing next sample.
	int Candidates;

	// Angle of disparity permitted distributed rays.  Range [0, pi].
	// Default = 0.
	float Angle;

	// Threshold for contrast--default is (0.4, 0.3, 0.6).
	RGBColor Threshold;
    };

    // Constructor starts us out with no textures or bump maps at all.
    DistributedHall() {
	NewNormal = 0;
	Ambient = Diffuse = Specular = Transmit = Reflect = 0;
    }

    // Routines to set up 
    virtual void SetTransmitParms(const DistribParms&);
    virtual void SetReflectParms(const DistribParms&);
    virtual RGBColor ComputeColor(ShadingInfo& shade, int depth);
    virtual Surface *Dup() const;
protected:
    virtual RGBColor SampleSpace(ShadingInfo&, Vector3D&, 
				 const DistribParms&, int, long *);
    virtual Vector3D NewSample(const DistribParms&, const Vector3D *, int);
private:
    DistribParms TransParms;	// Ray distribution parameters
				// for transmitted light
    DistribParms ReflectParms;	// Ray distribution parameters
				// for reflected light
};


