// ===================================================================
// utils.h
//	Header for a set of utilities included with OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================


// The utilities declared in this header file use OORT primitives 
// to construct more complicated instances of objects, according to 
// parameters given by the user.  For example, OORT provides a
// HallSurface class that encapsulates surfaces used to shade objects.
// This header file declares a number of functions that help you
// create useful surfaces, such as Glass().
// 
// The utilities declared in this header can be split into a number
// of subclasses:
//	- Object functions, e.g. MakeEllipsoid, MakeCone.
//	- Surface functions, e.g. Matte, Glass
//	- Texture functions, e.g. Wood, BlueMarble.
//	- Aggregate classes, e.g. SphereFlake, STetrahedron.

// =====================================
// Object-making functions.
// =====================================

// Creates an axis-aligned box with the given min and max vectors.
Aggregate *MakeBox(const Vector3D& min, const Vector3D& max, Surface *surf);

// Functions to manufacture different types of quadrics.  All are
// aligned along the Y axis.  The a and c radii for elliptical
// primitives (e.g. elliptical cylinder, elliptical cone) are
// the X and Z axes, respectively.

// a and c are the X and Z radii.
Object3D *MakeCone(float a, float c, Surface *surf);
Object3D *MakeCylinder(float a, float c, Surface *surf);

// a, b and c are the X, Y and Z radii.
// This function creates a hyperboloid of one sheet.
Object3D *MakeHyperboloid(float a, float b, float c, Surface *surf);

// a and c are the X and Z radii.  f is the focal distance.
Object3D *MakeParaboloid(float a, float c, float f, Surface *surf);

// Functions to manufacture different types of algebraic surface.
Algebraic *MakeCubicCusp(Surface *surf);	// Cubic cusp catastrophe
Algebraic *MakeFolium(Surface *surf);		// Folium
Algebraic *MakeKummer(Surface *surf);		// Kummer's surface
Algebraic *MakeLemniscate(Surface *surf);	// Lemniscate of Gerano
Algebraic *MakePiriform(Surface *surf);		// Piriform
Algebraic *MakeSteiner(Surface *surf);		// Steiner's surface
Algebraic *MakeQuarticCylinder(Surface *surf);	// Quartic cylinder

// =====================================
// Surface-making functions.
// =====================================
Surface *MakeClear();
Surface *MakeShiny(Texture *ambient = 0,
		   Texture *diffuse = 0,
		   Texture *specular = 0,
		   float specularity = 20);
Surface *MakeReflective(Texture *reflect = 0);
Surface *MakeGlass(Texture *transmit = 0,
		   Texture *reflect = 0,
		   Texture *specular = 0,
		   float specularity = 100,
		   Texture *diffuse = 0);
Surface *MakeGranite(const Matrix& reference = IdentityMatrix(),
		     const RGBColor& Ka = RGBColor(0.2),
		     const RGBColor& Kd = RGBColor(0.8));

// =====================================
// Texture-making functions.
// =====================================
Texture *Clouds(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *RedMarble(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *BlueMarble(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *WhiteMarble(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *BlackMarble(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *CherryWood(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *PineWood(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);
Texture *DarkWood(const Matrix& mat = IdentityMatrix(), float turbulence = 1.0);

// =====================================
// Aggregate objects
// =====================================

// SphereFlake: a recursively defined object composed of spheres.
class SphereFlake : public Aggregate {
public:
    SphereFlake(const Vector3D& center, 
    		float rad, 
		const Vector3D& dir, 
		int depth, 
		Surface *surf);
private:
    static Vector3D objset[9];
    static int objset_computed;
    void AddSphere(const Vector3D& center, float rad, const Vector3D& dir, int depth, Surface *surf);
    void CreateObjSet(Vector3D passbk[9]);
};

// Sierpinski's Tetrahedron, a recursively defined object composed
// of triangles.  The number of triangles is 4^N, where N is the
// depth passed to STetrahedron.
class STetrahedron : public Aggregate {
public:
    STetrahedron(int depth, float mag, Surface *surf);
private:
    void CreateTetra(int depth, 
		     const Vector3D& center, 
		     float mag, 
		     Surface *surf);
};

