// ================================================================
// world.h
//	Header file for World class.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ================================================================

// ---------------------------------------------------------
// World 
//	Contains all the information required to perform 
//	ray tracing on a set of objects.
// ---------------------------------------------------------
class World {

public:
    // For now, the available anti-aliasing modes are none (one ray
    // per pixel) and adaptive (as described in the book).
    enum AntiAliasModes { None, Adaptive };

protected:
    // Viewer parameters
    Vector3D ViewerLoc;		  // Location of viewer
    Vector3D LookAt;		  // Point that viewer is looking at
    Vector3D UpVector;		  // Up vector

    // Global list of objects
    Matrix ViewTransform;	  // View transform
    ObjectList Objects; 	  // Object tree
    ObjectList Unboundeds;	  // Unbounded objects
    LightList lights;		  // List of lights

    // Default colors
    RGBColor Background;	  // Background color
    RGBColor Ambient;		  // Ambient lighting

    // Depth limit.  Limits recursion due to spawned rays.
    // Default is 8.
    int DepthLimit;     	  // Depth limit

    // Image parameters
    int HorzRes, VertRes;	  // Horizontal and vertical resolution

    float ScreenHeight; 	  // Height of screen
    float ScreenWidth;		  // Width of screen
    float ScreenDist;		  // Screen distance

    int StartScan, EndScan;	  // Starting and ending scanlines
            			  // to put out

    long NumPixels;

    // Sampling parameters
    enum AntiAliasModes mode;	// Naive or Adaptive (default = Naive)
    int MinSamples, MaxSamples;	// Minimum and maximum samples to take
				//	(default = 4 and 16)
    int NumCandidates;		// Number of candidates to consider
				//	(default = 10)
    RGBColor ContrastThreshold;	// Contrast threshold (higher contrasts
				//	cause more rays).  Default =
				// 	(0.4, 0.3, 0.6) per Mitchell.

    // Output file
    char *Filename;		// Filename to put the chars to

    // State variables
    int printhierarchy; 	// True if hierarchy is to be
				// printed before 'tracing begins
    int printpers;		// True if a period is printed for
				// every scan line computed
    int interactive;		// True if user can hit a key and
				// get an update during the 'trace
    int hierarchyseed;		// Seed to give srand() before
				// before hierarchy is generated
    int numtries;		// Number of hierarchies to try

    // Goes through list of objects and breaks all aggregates into
    // their component objects.  Returns the resulting list.
    // Destroys the parameter list.  Thus, to disband all the
    // aggregates in the list of bounded objects in the World,
    // this function is called as follows:
    //		Objects = DisbandAggregate(Objects);
    ObjectList DisbandAggregates(ObjectList&);

    // Goes through objects and culls the ones that are unbounded,
    // like planes and quadrics.  Those get tested always.
    void RemoveUnboundeds();

    // Builds a hierarchy in the given list out of the given.
    static void BuildOneHierarchy(ObjectList& list, Object3D **objs, int n);

    // Assumes ObjectList contains the list of all objects in the World;
    // replaces ObjectList with a Goldsmith-Salmon-manufactured BVH.
    // Parameter is the number of hierarchies to consider before deciding
    // on one.
    void BuildHierarchy(int tries);

    // Culls those bounding boxes that only contain one object
    void CullSingletons(Object3D *bv, ObjectList *ParentList);

    Object3D *FindAtten(ObjectList &optr, Ray& ray,
			RGBColor& Atten, float maxt);
    Object3D *Find_Nearest(Ray& ray, float& t);

    Vector3D NewSample(Vector3D *, int);

    void DrawScreen(long, int, int); // Draws screen for interactive ray tracing
    void UpdateScreen();  // Updates screen for interactive ray tracing
    void CountObjectsRec(ObjectList& list);
    static void HierarchyCostRec(BoundingVolume *, float&, float);

public:
    // -------------------------------------------------------
    // Constructor sets some default values.
    // -------------------------------------------------------
    World();

    // -------------------------------------------------------
    // Functions to set parameters of the World.
    // -------------------------------------------------------

    // Sets the name of the output file.
    void SetOutputFile(char *NewFilename);

    // Sets horizontal and vertical resolution of output image.
    void SetHorzRes(int NewHorzRes);
    void SetVertRes(int NewVertRes);

    // Sets width and height of the viewing frustrum.
    void SetScreenWidth(float NewScreenWidth);
    void SetScreenHeight(float NewScreenHeight);
    void SetScreenDistance(float NewScreenDist);

    // Sets background color.  Pixels where no object is
    // intersected will be assigned this color.
    void SetBackgroundColor(RGBColor& NewBackground);

    // Set ambient light present in World.
    void SetAmbientLight(RGBColor& NewAmbient);

    // Set starting/ending scanlines to ray trace.
    // The ending scanline is not inclusive, i.e. to ray-trace
    // scanlines 0-479 for a 640x480 image you set Start to
    // 0 and End to 480.
    void SetStart(int Start);
    void SetEnd(int End);

    // Set depth limit of recursion.
    void SetDepthLimit(int NewDepthLimit);

    // Set viewer parameters
    //	LookAt will be in the middle of the rendered image.
    //	ViewerLoc will be the location of the viewer.
    //	UpVector will point toward the top of the rendered
    //	    image (this vector should be normalized).
    void SetViewerParameters(Vector3D& NewLookAt,
			     Vector3D& NewViewerLoc,
			     Vector3D& UpVector);

    // Set number of hierarchies to generate before deciding
    // on one.  Half a dozen is more than enough.
    void SetNumTries(int _numtries);

    // Anti-aliasing parameters.
    void SetAntiAliasMode(enum AntiAliasModes _mode);
    void SetSamples(int min, int max);
    void SetNumCandidates(int);
    void SetContrastThreshold(const RGBColor& thresh);

    // Returns the ambient light in the World.
    RGBColor GetAmbientLight() { return Ambient; }

    // Returns nonzero if the depth given has exceeded the
    // World's depth limit of recursion.
    int HitDepthLimit(int depth) { return depth >= DepthLimit; }

    // Add objects and light sources to World.
    void AddObject(Object3D& Obj);
    void AddObject(Object3D *Obj);
    void AddLight(Light *newlight);

    // Parse an NFF file, add the objects to the World and
    // reset the viewer parameters according to it.
    void ParseNFFFile(const char *filename);

    // Parse the command line given by the argc/argv parameters
    // and modify the world accordingly.
    void ParseCommandLine(int argc, char *argv[]);

    // Count the objects in the World.
    void CountObjects();

    // Returns the cost of the World's hierarchy.
    static float HierarchyCost(const ObjectList&);

    // Finds the nearest intersection along the ray, shades
    // that point.  Call initially with depth=0.
    RGBColor FindColor(Ray& ray, int depth);

    // Computes the color of a pixel.  See world.cpp for how
    // to use.
    RGBColor PixelColor(float xtarg, float ytarg);

    // Computes the color due to light sources using the
    // intersection point and other information in 
    RGBColor DiffuseLight(ShadingInfo& shade, Surface *surf);
    Object3D *ShadowAtten(Ray& ray, RGBColor& Atten, float maxt);

    // Using the parameters in the World so far, generate a
    // bounding volume hierarchy, ray-trace the image and
    // write it to the output file.
    void RayTrace();
};

// Routine that adds an object to a bounding box hierarchy using 
// the Goldsmith-Salmon heuristic.
void AddNewObject(ObjectList *root, Object3D *addme);

