// =================================================================
// bbox.cpp
//	Bounding box-related routines.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// =================================================================

#include "oort.h"

// AxisAlignedBox implementations
Vector3D
AxisAlignedBox::Min() const
{
    return min;
}

Vector3D
AxisAlignedBox::Max() const
{
    return max;
}

int
AxisAlignedBox::Unbounded() const
{
    return min == Vector3D(-MAXFLOAT) || max == Vector3D(-MAXFLOAT);
}

void
AxisAlignedBox::Include(const Vector3D& newpt)
{
    Minimize(min, newpt);
    Maximize(max, newpt);
}

void
AxisAlignedBox::Include(const AxisAlignedBox& bbox)
{
    Minimize(min, bbox.min);
    Maximize(max, bbox.max);
}

extern "C" int bboxint(const Vector3D *minmax, 
		       const Vector3D *locdir, 
		       float *tmin, float *tmax);

// ComputeMinMaxT computes the minimum and maximum parameters
// of intersection with the ray; it returns 1 if the ray hits
// the bounding box and 0 if it does not.
int
AxisAlignedBox::ComputeMinMaxT(Ray& ray, float *tmin, float *tmax) const
{
#ifndef ASM
    float t1, t2;
    float minx, maxx, miny, maxy, minz, maxz;
    if (fabs(ray.dir.x) < 0.001) {
	if (min.x < ray.loc.x && max.x > ray.loc.x) {
	    minx = -MAXFLOAT;
	    maxx = MAXFLOAT;
	}
	else
	    return 0;
    }
    else {
	t1 = (min.x - ray.loc.x) / ray.dir.x;
	t2 = (max.x - ray.loc.x) / ray.dir.x;
	if (t1 < t2) {
	    minx = t1;
	    maxx = t2;
	}
	else {
	    minx = t2;
	    maxx = t1;
	}
	if (maxx < 0)
	    return 0;
    }

    if (fabs(ray.dir.y) < 0.001) {
	if (min.y < ray.loc.y && max.y > ray.loc.y) {
	    miny = -MAXFLOAT;
	    maxy = MAXFLOAT;
	}
	else
	    return 0;
    }
    else {
	t1 = (min.y - ray.loc.y) / ray.dir.y;
	t2 = (max.y - ray.loc.y) / ray.dir.y;
	if (t1 < t2) {
	    miny = t1;
	    maxy = t2;
	}
	else {
	    miny = t2;
	    maxy = t1;
	}
	if (maxy < 0)
	    return 0;
    }

    if (fabs(ray.dir.z) < 0.001) {
	if (min.z < ray.loc.z && max.z > ray.loc.z) {
	    minz = -MAXFLOAT;
	    maxz = MAXFLOAT;
	}
	else
	    return 0;
    }
    else {
	t1 = (min.z - ray.loc.z) / ray.dir.z;
	t2 = (max.z - ray.loc.z) / ray.dir.z;
	if (t1 < t2) {
	    minz = t1;
	    maxz = t2;
	}
	else {
	    minz = t2;
	    maxz = t1;
	}
	if (maxz < 0)
	    return 0;
    }

    *tmin = minx;
    if (miny > *tmin)
	*tmin = miny;
    if (minz > *tmin)
	*tmin = minz;

    *tmax = maxx;
    if (maxy < *tmax)
	*tmax = maxy;
    if (maxz < *tmax)
	*tmax = maxz;
#else
    if (! bboxint(&min, &ray.loc, tmin, tmax))
	return 0;
#endif
    return 1;
}

int
BoundingBox::NearestInt(Ray& ray, float& t, float maxt)
{
    float tmin, tmax;

    Statistics::Intersections::BBox++;

    if (! AxisAlignedBox::ComputeMinMaxT(ray, &tmin, &tmax))
	return 0;
    if (tmax < tmin)
	return 0;
    if (tmin < 0) {
	// No intersection if both parameters are negative
	if (tmax < 0)
	    return 0;
	// The ray hit the bounding box
	// tmax is parameter of intersection
	t = (tmax < maxt) ? tmax : maxt;
    }
    else {
	// The ray hit the bounding box
	// with parameter tmin.
	t = (tmin < maxt) ? tmin : maxt;
    }
    return 1;
}

SpanList *
BoundingBox::FindAllIntersections(Ray& ray)
{
    cerr << "BoundingBox::FindAllIntersections called\n";
    exit(1);
    return 0;
}

void
BoundingBox::ApplyTransform(const Matrix &tform)
{
    AxisAlignedBox bbox(Vector3D(MAXFLOAT), Vector3D(-MAXFLOAT));

    for (ObjectList::Iterator sc(contents); sc.Valid(); sc.GotoNext()) {
	sc.Contents()->ApplyTransform(tform);
	bbox = Union(bbox, sc.Contents()->BBox());
    }
    min = bbox.Min();
    max = bbox.Max();
}

Vector3D
BoundingBox::FindNormal(const Vector3D& intersection)
{
    cerr << "BoundingBox::FindNormal called\n";
    return Vector3D(0);
}

Object3D *
BoundingBox::Dup() const
{
    BoundingBox *ret = new BoundingBox;

    for (ObjectList::Iterator sc(contents); sc.Valid(); sc.GotoNext())
	ret->AddObject(sc.Contents()->Dup());

    return ret;
}

int
BoundingBox::IsInside(Vector3D& v)
{
    return ! (v.x < min.x ||
	      v.x > max.x ||
	      v.y < min.y ||
	      v.y > max.y ||
	      v.z < min.z ||
	      v.z > max.z);
}

float
BoundingBox::PtDistance(Vector3D& v)
{
    cerr << "BoundingBox::PtDistance called\n";
    exit(2);
    return 0;
}

void 
BoundingBox::Describe(int ind) const
{
    ObjectList::Iterator sc(contents);

    indent(ind);
    cout << setprecision(2) << "Bounding box (" << min.x << ", ";
    cout << min.y << ", " << min.z << ")-(" << max.x << ", ";
    cout << max.y << ", " << max.z << ") contains:\n";
    while (sc.Valid()) {
	sc.Contents()->Describe(ind + 2);
	sc.GotoNext();
    }
}

void
BoundingVolume::AddObject(const Object3D& NewObj)
{
    contents.AddToList(NewObj.Dup());
}

void
BoundingVolume::AddObject(Object3D *NewObj)
{
    contents.AddToList(NewObj);
}

void
BoundingVolume::TransformSurface(SurfaceList& clist, const Matrix& tform)
{
    ObjectList::Iterator sc(contents);

    while (sc.Valid()) {
	sc.Contents()->TransformSurface(clist, tform);
	sc.GotoNext();
    }
}

float
AxisAlignedBox::SurfaceArea() const
{
    Vector3D ext = max - min;
    return 2 * (ext.x * ext.y +
		ext.x * ext.z +
		ext.y * ext.z);
}

float
BoundingBox::SurfaceArea() const
{
    return AxisAlignedBox::SurfaceArea();
}

AxisAlignedBox
BoundingBox::BBox() const
{
    return AxisAlignedBox(Min(), Max());
}

void
BoundingBox::AddObject(const Object3D& NewObj)
{
    BoundingVolume::AddObject(NewObj);
    Include(NewObj.BBox());
}

void
BoundingBox::AddObject(Object3D *NewObj)
{
    BoundingVolume::AddObject(NewObj);
    Include(NewObj->BBox());
}

