// ===================================================================
// csg.cpp
//	Constructive Solid Geometry implementation.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include <iostream.h>
#include "oort.h"

// -------------------------------------------------------------------
//	CSGObject routines.
//	These provide united support for the three CSG primitives,
//	union, intersection and difference.
//	Many functions are the same for all three (thus in CSGObject).
// -------------------------------------------------------------------

CSGObject::CSGObject(Object3D *Left, Object3D *Right) : Object3D(CSG)
{
    left = Left;
    right = Right;
    if ((left->ReturnColor() && left->ReturnColor()->Transparent()) || 
	(right->ReturnColor() && right->ReturnColor()->Transparent())) {
	HallSurface *newsurf = new HallSurface;
	newsurf->SetTransmit(new PureColor(RGBColor(1)), 1.3);
	OverrideColor(newsurf);
    }
}

int
CSGObject::NearestInt(Ray& ray, float& t, float maxt)
{
    SpanList *tmp = FindAllIntersections(ray);
    if (tmp) {
	Span min = tmp->ExtractMinSpan();
	int ret;
	t = 0;

	// Find first intersection greater than Limits::Threshold
	while (t < Limits::Threshold) {
	    if (min.omin) {
		ret = 1;
		nearest = min.omin;
		OverrideColor(min.omin->ReturnColor());
		min.omin = 0;
		t = min.tmin;
	    }
	    else {
		ret = 1;
		nearest = min.omax;
		OverrideColor(min.omax->ReturnColor());
		t = min.tmax;
		if (! tmp->NumSpans())
		    break;
		min = tmp->ExtractMinSpan();
	    }
	}

	// If nearest int is too close, return no-intersection.
	if (t < Limits::Threshold || t >= maxt) {
	    nearest = 0;
	    ret = 0;
	}

	delete tmp;
	return ret;
    }
    return 0;
}

Vector3D
CSGObject::FindNormal(const Vector3D& Intersection)
{
    if (nearest)
	return nearest->FindNormal(Intersection);
    cerr << "CSGObject::FindNormal called\n";
    exit(1);
    return Vector3D(0);
}

void
CSGObject::ApplyTransform(const Matrix& tform)
{
    left->ApplyTransform(tform);
    right->ApplyTransform(tform);
}

AxisAlignedBox
CSGObject::BBox() const
{
    AxisAlignedBox ret(Vector3D(MAXFLOAT), Vector3D(-MAXFLOAT));

    ret.Include(left->BBox());
    ret.Include(right->BBox());
    return ret;
}

void
CSGObject::CountMe() const
{
    left->CountMe();
    right->CountMe();
}

// -------------------------------------------------------------------
//	CSGUnion routines.
// -------------------------------------------------------------------

SpanList *
CSGUnion::FindAllIntersections(Ray& ray)
{
    SpanList *leftint = left->FindAllIntersections(ray);
    SpanList *rightint = right->FindAllIntersections(ray);
    return MergeUnion(leftint, rightint);
}

Object3D *
CSGUnion::Dup() const
{
    return new CSGUnion(left->Dup(), right->Dup());
}

int
CSGUnion::IsInside(Vector3D& v)
{
    return left->IsInside(v) && right->IsInside(v);
}

int
CSGUnion::HitAtAll(Ray& ray, float maxt)
{
    return left->HitAtAll(ray, maxt) || right->HitAtAll(ray, maxt);
}

float
CSGUnion::PtDistance(Vector3D& v)
{
    float d1 = left->PtDistance(v);
    float d2 = right->PtDistance(v);
    return (d1 < d2) ? d1 : d2;
}

void
CSGUnion::Describe(int ind) const
{
    indent(ind);  cout << "CSGUnion:\n";
    indent(ind+2);  cout << "Left:\n";
    left->Describe(ind+4);
    indent(ind+2);  cout << "Right:\n";
    right->Describe(ind+4);
}

void
CSGUnion::CountMe() const
{
    Statistics::Objects::CSGUnion += 1;
    CSGObject::CountMe();
}

// -------------------------------------------------------------------
//	CSGIntersection routines.
// -------------------------------------------------------------------

SpanList *
CSGIntersection::FindAllIntersections(Ray& ray)
{
    SpanList *leftint = left->FindAllIntersections(ray);

    if (leftint) {
	SpanList *rightint = right->FindAllIntersections(ray);
	return MergeIntersection(leftint, rightint);
    }
    return 0;
}

Object3D *
CSGIntersection::Dup() const
{
    return new CSGIntersection(left->Dup(), right->Dup());
}

int
CSGIntersection::IsInside(Vector3D& v)
{
    return left->IsInside(v) && right->IsInside(v);
}

float
CSGIntersection::PtDistance(Vector3D& v)
{
    float d1 = left->PtDistance(v);
    float d2 = right->PtDistance(v);
    if (d1 < 0 && d2 < 0)		  // If inside both, return min
	return (d1 < d2) ? d2 : d1;	  // distance estimate
    else {
	if (d1 < 0)			  // If inside left, return that
	    return d1;			  // distance estimate
	else if (d2 < 0)		  // If inside right, return that
	    return d2;			  // distance estimate
	else
	    return (d1 < d2) ? d1 : d2;   // Else return min estimate
    }
}

void
CSGIntersection::Describe(int ind) const
{
    indent(ind);    cout << "CSGIntersection:\n";
    indent(ind+2);  cout << "Left:\n";
    left->Describe(ind+4);
    indent(ind+2);  cout << "Right:\n";
    right->Describe(ind+4);
}

AxisAlignedBox
CSGIntersection::BBox() const
{
    return Intersect(left->BBox(), right->BBox());
}

void
CSGIntersection::CountMe() const
{
    Statistics::Objects::CSGIntersection += 1;
    CSGObject::CountMe();
}

// -------------------------------------------------------------------
//	CSGDifference routines.
// -------------------------------------------------------------------

SpanList *
CSGDifference::FindAllIntersections(Ray& ray)
{
    SpanList *leftint = left->FindAllIntersections(ray);

    if (leftint) {
	SpanList *rightint = right->FindAllIntersections(ray);
	return MergeDifference(leftint, rightint);
    }
    return 0;
}

Object3D *
CSGDifference::Dup() const
{
    return new CSGDifference(left->Dup(), right->Dup());
}

int
CSGDifference::IsInside(Vector3D& v)
{
    return left->IsInside(v) && ! right->IsInside(v);
}

float
CSGDifference::PtDistance(Vector3D& v)
{
    float d1 = left->PtDistance(v);
    float d2 = right->PtDistance(v);
    if (d2 < 0) 	  // Inside right node--so not inside object
	return d2;
    if (d1 < 0) 	  // Inside left node--so inside object
	return (-d1 < d2) ? d1 : -d2;

    // Not inside either of the constituent objects, return nearest
    // distance estimate.
    return (d1 < d2) ? d1 : d2;
}

void
CSGDifference::Describe(int ind) const
{
    indent(ind);  cout << "CSGDifference:\n";
    indent(ind+2);  cout << "Left:\n";
    left->Describe(ind+4);
    indent(ind+2);  cout << "Right:\n";
    right->Describe(ind+4);
}

void
CSGDifference::CountMe() const
{
    Statistics::Objects::CSGDifference += 1;
    CSGObject::CountMe();
}


