// ===================================================================
// inv.cpp
//	Implementation file for inverse-ray objects (objects with a
//	canonical form that are transformed by transforming the ray
//	into the same space as the canonical form).
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

// Transform the ray so that it is in canonical space.
Ray
InverseRayObject::InvertRay(Ray& ray)
{
    Ray ret = ray;
    ret.ApplyTransform(inv);
    return ret;
}

// Transform a point so that it is in canonical space.
Vector3D
InverseRayObject::InvertPt(Vector3D& v)
{
    return v * inv;
}

// Rotate a vector so that it is in canonical space.
Vector3D
InverseRayObject::InvRotatePt(Vector3D& v)
{
    return RotateOnly(v, inv);
}

// Transform a ray from canonical space back to regular space.
Ray
InverseRayObject::TransformRay(Ray& ray)
{
    Ray ret = ray;
    ret.ApplyTransform(mat);
    return ret;
}

// Transform a point from canonical space back to regular space.
Vector3D
InverseRayObject::TransformPt(Vector3D& v)
{
    return v * mat;
}

// Rotate a point from canonical space back to regular space.
Vector3D
InverseRayObject::RotatePt(Vector3D& v)
{
    return RotateOnly(v, mat);
}

// Compute the nearest intersection along the ray, if any.
// This function transforms the ray to canonical space before
// passing it along to the Intersect routine for the object.
int
InverseRayObject::NearestInt(Ray& cpyme, float& t, float maxt)
{
    float tempt;
    Ray ray = InvertRay(cpyme);

    int ret = Intersect(ray, tempt, maxt);
    if (ret != 0) {
	t = cpyme.Interp(ray.Extrap(tempt) * mat);
	ret = (t > Limits::Threshold && t < maxt);
    }
    return ret;
}

// Apply a transform to the object.  This function affects only
// the transformations to and from the canonical space of the
// object.
void
InverseRayObject::ApplyTransform(const Matrix& tform)
{
    mat = tform * mat;
    inv *= Invert(tform);
}


