// ===================================================================
// light.cpp
//	Implementation file for various light sources.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "world.h"

// -------------------------------------------------------------------
// Light
//	This class encapsulates the various types of light source
//	available in OORT.  Shadow rays are cast at light sources;
//	they emit light of a given color, which is attenuated by
//	distance.
// -------------------------------------------------------------------

Light::Light(const RGBColor& clr, float C1, float C2, float C3)
{
    Color = clr;
    c1 = C1;
    c2 = C2;
    c3 = C3;
    LastOpaque = 0;
}

// -------------------------------------------------------------------
// Light::DistAtten.
//	Computes the attenuation due to distance of the light source,
//	clamped to 1.0.
// -------------------------------------------------------------------
float
Light::DistAtten(float t)
{
    float ret = 1 / (c1 + t * (c2 + c3 * t));
    return (ret < 1) ? ret : 1;
}

// -------------------------------------------------------------------
// Check for a cached object and, if there is one, check if it is
// intersected by the given ray within the given maximum parameter of
// intersection.  If so, return 1; otherwise clear the cached object
// and return 0.
// -------------------------------------------------------------------
int
Light::HitCachedObject(Ray& ray, float maxt)
{
    if (LastOpaque) {

	Statistics::ShadowTotal++;

	if (LastOpaque->HitAtAll(ray, maxt)) {
	    Statistics::ShadowHits++;
	    return 1;
	}
	else {
	    LastOpaque = 0;
	    return 0;
	}
    }
    return 0;
}

RGBColor
Light::ShadowRay(ShadingInfo& shade, 
		 Vector3D& target,
		 Surface *surf)
{
    RGBColor ret(0);

    Vector3D RayDirection = target - shade.p;	// Find direction of ray
    float dist = Magnitude(RayDirection);	// Find distance to light source
    RayDirection /= dist;			// Normalize ray direction
    Ray ShadowRay(shade.p, RayDirection);

    // If hit the cached opaque object, return black.
    if (HitCachedObject(ShadowRay, dist))
	return RGBColor(0);

    // We're going to have to fire a shadow ray.
    Statistics::ShadowRays++;

    RGBColor Atten(1, 1, 1);
    Object3D *obj = shade.world->ShadowAtten(ShadowRay, Atten, dist);
    if (obj) {
	LastOpaque = obj;
	return RGBColor(0);
    }
    return surf->ShadeLight(shade, RayDirection) *
    	   Atten * Color * DistAtten(dist);
}

// -------------------------------------------------------------------
// PointLight
//	This class encapsulates the ubiquitous point light sources in
//	ray tracing.  Only one shadow ray is cast at point light sources,
//	even if distribution is requested by the user.
// -------------------------------------------------------------------

PointLight::PointLight(const Vector3D& loc,
		       const RGBColor& clr,
		       float C1, float C2, float C3) : 
    Light(clr, C1, C2, C3)
{
    Location = loc;
}

// -------------------------------------------------------------------
// Find the color of the light source as seen from the given
// intersection point.
// -------------------------------------------------------------------

RGBColor
PointLight::ShadowLight(ShadingInfo& shade, Surface *surf)
{
    return ShadowRay(shade, Location, surf);
}

void
PointLight::ApplyTransform(const Matrix& tform)
{
    Location = tform * Location;
}

// -------------------------------------------------------------------
// DirectLight
//	This class encapsulates the directional light source.
//	They are located at infinity; only one shadow ray is cast at
//	directional light sources, even if distribution is requested
//	by the user.
// -------------------------------------------------------------------

// --------------------------
// Constructor
// --------------------------
DirectLight::DirectLight(const Vector3D& dir, const RGBColor& clr) :
    Light(clr, 0, 0, 0), Direction(Normalize(dir)) { }

// -------------------------------------
// Find the color of the light source as
// seen from the given intersection 
// point.
// -------------------------------------

RGBColor
DirectLight::ShadowLight(ShadingInfo& shade, Surface *surf)
{
    RGBColor ret(0);

    Ray ShadowRay(shade.p, Direction);

    // If hit the cached opaque object, return black.
    if (HitCachedObject(ShadowRay, Limits::Infinity))
	return RGBColor(0);

    // We're going to have to fire a shadow ray.
    Statistics::ShadowRays++;

    RGBColor Atten(1, 1, 1);
    Object3D *obj = shade.world->ShadowAtten(ShadowRay, Atten, Limits::Infinity);
    if (obj) {
	LastOpaque = obj;
	return RGBColor(0);
    }
    return surf->ShadeLight(shade, Direction) * Atten * Color;
}

void
DirectLight::ApplyTransform(const Matrix& tform)
{
    Direction = RotateOnly(tform, Direction);
}

// -------------------------------------------------------------------
// RectangularLight
//	Extended light source.
// -------------------------------------------------------------------

RectangularLight::RectangularLight(float _width, float _height,
    const RGBColor& _clr, int _min, int _max, int _numcandidates,
    const RGBColor& _threshold, float _c1, float _c2, float _c3):
    width(_width), height(_height), MinSamples(_min), MaxSamples(_max),
    NumCandidates(_numcandidates), ContrastThreshold(_threshold),
    Light(_clr, _c1, _c2, _c3)
{
    applied = IdentityMatrix();
}

Vector3D
RectangularLight::NewSample(Vector3D *samples, int n)
{
    Vector3D ret(RandomValue(0, 1), RandomValue(0, 1), 0);
    if (! n)
	return ret;

    float mindist = MinDist(ret, samples, n);
    for (int i = 0; i < NumCandidates; i++) {
	Vector3D candidate(RandomValue(0, 1), RandomValue(0, 1), 0);
	float dist = MinDist(candidate, samples, n);
	if (dist > mindist) {
	    ret = candidate;
	    mindist = dist;
	}
    }
    return ret;
}

RGBColor
RectangularLight::ShadowLight(ShadingInfo& shade, Surface *surf)
{
    RGBColor Imin(MAXFLOAT), Imax(-MAXFLOAT), Intens(0);
    Vector3D *pts = new Vector3D[MaxSamples];

    for (int i = 0; i < MaxSamples; i++) {
	pts[i] = NewSample(pts, i);

	Vector3D target = pts[i];
	target *= Vector3D(width, height, 0);
	target -= Vector3D(width / 2, height / 2, 0);
	target = applied * target;
	RGBColor clr = ShadowRay(shade, target, surf);
	Intens += clr;
	clr = NormalizeColor(clr);
	Minimize(Imin, clr);
	Maximize(Imax, clr);
	if (i >= MinSamples) {
	    if (i >= MaxSamples)
		break;
	    if (! MoreRays(Imin, Imax, ContrastThreshold))
		break;
	}
    }
    delete[] pts;
    Statistics::RayPixHist[i - 1] += 1;
    return Intens / (i + 1);
}

void
RectangularLight::ApplyTransform(const Matrix& tform)
{
    applied *= tform;
}

