// ===================================================================
// nff.cpp
//	Parser for NFF files for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "world.h"

static void
prerror(const char *err)
{
    cerr << err << '\n';
    exit(3);
}

void
World::ParseNFFFile(const char *filename)
{
    char nexttok[256];
    char cmd;
    ifstream inpf(filename);
    HallSurface *clr = 0;

    if (! inpf) {
	sprintf(nexttok, "Could not open %s", filename);
	prerror(nexttok);
    }
    nexttok[0] = '\0';
    while (! inpf.eof()) {
	if (nexttok[0]) {
	    cmd = nexttok[0];
	    nexttok[0] = '\0';
	}
	else {
	    inpf >> cmd;
	    if (! inpf)
		break;
	}
	switch (cmd) {
	    case 'v':
		{
		    float fromx, fromy, fromz;
		    float atx, aty, atz;
		    float upx, upy, upz;
		    float ang, hither;
		    int xres, yres;

		    inpf >> nexttok;
		    if (strcmp(nexttok, "from"))
			prerror("\"from\" expected");
		    inpf >> fromx >> fromy >> fromz >> nexttok;
		    if (strcmp(nexttok, "at"))
			prerror("\"at\" expected");
		    inpf >> atx >> aty >> atz >> nexttok;
		    if (strcmp(nexttok, "up"))
			prerror("\"up\" expected");
		    inpf >> upx >> upy >> upz >> nexttok;
		    if (strcmp(nexttok, "angle"))
			prerror("\"angle\" expected");
		    inpf >> ang >> nexttok;
		    if (strcmp(nexttok, "hither"))
			prerror("\"angle\" expected");
		    inpf >> hither >> nexttok;

		    SetViewerParameters(Vector3D(atx, aty, atz),
					Vector3D(fromx, fromy, fromz),
					Vector3D(upx, upy, upz));
		    SetScreenWidth(asin(ang * M_PI / 180));
		    SetScreenHeight(asin(ang * M_PI / 180));
		    SetScreenDistance(1);

		    if (strcmp(nexttok, "resolution"))
			prerror("\"resolution\" expected");
		    inpf >> xres >> yres;
		    SetHorzRes(xres);
		    SetVertRes(yres);
		    nexttok[0] = '\0';
		}
		break;
	    case 'b':
		{
		    float x, y, z;

		    inpf >> x >> y >> z;
		    SetBackgroundColor(RGBColor(x, y, z));
		}
		break;
	    case 'l':
		{
		    float lx, ly, lz;
		    float cx, cy, cz;
		    char *sc;

		    inpf >> lx >> ly >> lz;
		    inpf >> nexttok;
		    cx = strtod(nexttok, &sc);
		    if (sc)
			AddLight(new PointLight(Vector3D(lx, ly, lz),
						RGBColor(1, 1, 1), 1, 0, 0));
		    else {
			inpf >> cy >> cz;
			AddLight(new PointLight(Vector3D(lx, ly, lz),
						RGBColor(cx, cy, cz), 1, 0, 0));
			nexttok[0] = '\0';
		    }
		}
		break;
	    case 'f':
	      {
		  float r, g, b, kd, ks, shine, t, inx;
		  inpf >> r >> g >> b >> kd >> ks >> shine >> t >> inx;
		  clr = new HallSurface;
		  clr->SetDiffuse(new PureColor(RGBColor(kd*r, kd*g, kd*b)));
		  clr->SetSpecular(new PureColor(RGBColor(ks*r, ks*g, ks*b)), shine);
		  if (t > 0)
		      clr->SetTransmit(new PureColor(RGBColor(t)), inx);
	      }
	      break;
	    case 'c':
	      {

	      }
	      break;
	    case 's':
	      {
		  float x, y, z, r;
		  inpf >> x >> y >> z >> r;
		  AddObject(new Sphere(Vector3D(x, y, z), r, clr));
	      }
	      break;
	    case 'p':
		if (inpf.peek() == 'p') {
		    int n;
		    Vector3D *verts;
		    inpf >> cmd;
		    inpf >> n;
		    verts = new Vector3D[n];
		    for (int i = 0; i < n; i++) {
			float x, y, z, x1, y1, z1;
			inpf >> x >> y >> z >> x1 >> y1 >> z1;
			verts[i] = Vector3D(x, y, z);
		    }
		    AddObject(new Polygon(n, verts, clr));
		    delete verts;
		}
		else {
		    int n;
		    Vector3D *verts;
		    inpf >> n;
		    verts = new Vector3D[n];
		    for (int i = 0; i < n; i++) {
			float x, y, z;
			inpf >> x >> y >> z;
			verts[i] = Vector3D(x, y, z);
		    }
		    AddObject(new Polygon(n, verts, clr));
		    delete verts;
		}
		break;
	    case '#':
		inpf.getline(nexttok, 256);
		nexttok[0] = '\0';
		break;
	}
    }
}

