// ===================================================================
// object3d.cpp
//	Support routines for Object3D class.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "world.h"

// =========================================================
// Object3D::ShadeObject
//	Called by World when it is determined that the 
//	object was the closest along the given ray.
// =========================================================
RGBColor
Object3D::ShadeObject(World& world, Ray& ray, Vector3D& Intersection, int depth)
{
    ShadingInfo shade;

    shade.p = Intersection;
    shade.normal = FindNormal(Intersection);
    shade.incident = ray.dir;
    shade.reflected = ray.ReflectRay(shade.normal);
    shade.obj = this;
    shade.ray = &ray;
    shade.world = &world;
    return surf->ComputeColor(shade, depth);
}

int
Object3D::HitAtAll(Ray& ray, float maxt)
{
    float tempt;
    return NearestInt(ray, tempt, maxt) != 0;
}

int
Object3D::IsInside(Vector3D& v)
{
    float dist = PtDistance(v);
    return dist < 0 || fabs(dist) < Limits::Small;
}

void
Object3D::TransformSurface(SurfaceList& clist, const Matrix& tform)
{
    if (flags & BOUNDING) {
	((BoundingVolume *) this)->TransformSurface(clist, tform);
    }	
    else if (! (flags & CSG)) {
	if (surf && ! clist.SurfaceThere(surf)) {
	    clist.AddToList(surf);
	    surf->PreMulTransform(tform);	
	}
    }
}

int
SurfaceList::SurfaceThere(Surface *probe)
{
    for (Iterator sc(*this); sc.Valid(); sc.GotoNext()) {
	if (sc.Contents() == probe)
	    return 1;
    }
    return 0;
}

void
ObjectList::TransformList(const Matrix& Transform)
{
    for (Iterator sc(*this); sc.Valid(); sc.GotoNext())
	sc.Contents()->ApplyTransform(Transform);
}

void
ObjectList::TransformSurfaces(SurfaceList& clist, const Matrix& Transform)
{
    for (Iterator sc(*this); sc.Valid(); sc.GotoNext())
	sc.Contents()->TransformSurface(clist, Transform);
}

