// ==================================================================
// poly.cpp
//	Implementation file for Polynomial class.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ==================================================================

#include <stdio.h>
#include <mem.h>
#include <math.h>
#include <iostream.h>
#include <stdlib.h>

#include "poly.h"

int Polynomial::MAXPOW = 32;
double Polynomial::SMALL_ENOUGH = 1e-12;
double Polynomial::RELERROR = 1e-8;
int Polynomial::MAXIT = 800;

Polynomial::Polynomial(int Order, double *Coeff)
{
    ord = Order;
    coeff = new double[ord+1];
    memcpy(coeff, Coeff, (ord+1)*sizeof(double));
}

Polynomial::Polynomial(const Polynomial& a)
{
    ord = a.ord;
    while (ord > 0 && a.coeff[ord] == 0.0)
	ord--;
    coeff = new double[ord+1];
    memcpy(coeff, a.coeff, (ord+1)*sizeof(double));
}

Polynomial&
Polynomial::operator=(const Polynomial& x)
{
    if (coeff)
	delete[] coeff;
    ord = x.ord;
    while (ord > 0 && fabs(x.coeff[ord]) < SMALL_ENOUGH)
	ord--;
    coeff = new double[ord+1];
    memcpy(coeff, x.coeff, (ord+1)*sizeof(double));
    return *this;
}

// ---------------------------------------------------------
// Destructor frees the coefficient array.
// ---------------------------------------------------------
Polynomial::~Polynomial()
{
    delete[] coeff;
    coeff = 0;
}

// ---------------------------------------------------------
// Add a new coefficient (of given order) to the polynomial
// ---------------------------------------------------------
void
Polynomial::AddElm(int Order, double x)
{
    if (! coeff) {
	ord = Order;
	coeff = new double[ord+1];
	memset(coeff, 0, ord*sizeof(double));
	coeff[ord] = x;
    }
    else {
	if (Order > ord) {
	    double *temp = new double[Order+1];
	    memcpy(temp, coeff, (ord+1)*sizeof(double));
	    if (Order-1 > ord)
		memset(temp+ord+1, 0, (Order-ord-1)*sizeof(double));
	    temp[ord = Order] = x;
	    delete[] coeff;
	    coeff = temp;
	}
	else
	    coeff[Order] += x;
    }
}

// ---------------------------------------------------------
// Evaluate polynomial at a given point.
// ---------------------------------------------------------
double
Polynomial::Evaluate(double x) const
{
    int inx = ord+1;
    double ret = coeff[--inx];

    while (inx > 0)
	ret = x * ret + coeff[--inx];
    return ret;
}

// ---------------------------------------------------------
// Negate a polynomial.
// ---------------------------------------------------------
Polynomial
Polynomial::operator-()
{
    for (int i = 0; i <= ord; i++)
	coeff[i] = -coeff[i];
    return *this;
}

// ---------------------------------------------------------
// Transforms a polynomial so the first coefficient
// (coefficient of largest term) is equal to 1.
// ---------------------------------------------------------
Polynomial
Normalize(const Polynomial& x)
{
    double divby = fabs(x.coeff[x.ord]);
    if (divby == 1)
	return Polynomial(x);
    else {
	Polynomial ret(x);
	for (int i = 0; i <= x.ord; i++)
	    ret.coeff[i] /= divby;
	return ret;
    }
}

// ---------------------------------------------------------
// Returns the derivative of a polynomial.
// ---------------------------------------------------------
Polynomial
Derivative(const Polynomial& x)
{
    Polynomial ret;

    for (int i = 1; i <= x.ord; i++)
	ret.AddElm(i-1, i*x.coeff[i]);
    return ret;
}

// ---------------------------------------------------------
// Returns (a+b)^x, 
//	e.g. Triangle(1, 1, 2) returns x^2 + 2x + 1
// ---------------------------------------------------------
Polynomial
Triangle(double a, double b, int x)
{
    int i;
    double *coeff;
    double tempa = 1;
    double tempb = pow((double) b, x);
    double mulby = 1;
    double divby = 1;
    Polynomial ret;

    for (i = 1; i <= x; i++)
	mulby *= i;
    divby = mulby;
    coeff = new double[x+1];
    for (i = 0; i < x; i++) {
	coeff[i] = mulby*tempa*tempb/divby;
	tempa *= a;
	if (b != 0.0)
	    tempb /= b;
	mulby /= i+1;
	divby /= x-i;
    }
    coeff[i] = tempa;
    ret = Polynomial(x, coeff);
    delete[] coeff;
    return ret;
}

// ---------------------------------------------------------
// Add a polynomial to this one.
// ---------------------------------------------------------
Polynomial&
Polynomial::operator+=(const Polynomial& x)
{
    for (int i = 0; i <= x.ord; i++)
	AddElm(i, x.coeff[i]);
    return *this;
}

// ---------------------------------------------------------
// Multiply polynomial by another polynomial.
// ---------------------------------------------------------
Polynomial&
Polynomial::operator*=(const Polynomial& x)
{
    // Still zero if the polynomial has no coefficients yet.
    if (coeff) {
	double *newcoeff = new double[ord + x.ord + 2];
	memset(newcoeff, 0, (ord + x.ord + 2)*sizeof(double));
	for (int i = 0; i <= ord; i++)
	    for (int j = 0; j <= x.ord; j++)
		newcoeff[i + j] += coeff[i] * x.coeff[j];
	delete[] coeff;
	coeff = newcoeff;
	ord += x.ord;
    }
    return *this;
}

// ---------------------------------------------------------
// Multiply polynomial by a constant.
// ---------------------------------------------------------
Polynomial&
Polynomial::operator*=(double x)
{
    for (int i = 0; i <= ord; i++)
	coeff[i] *= x;
    return *this;
}

// ---------------------------------------------------------
// Binary + for polynomials.
// ---------------------------------------------------------
Polynomial
operator+(const Polynomial& A, const Polynomial& B)
{
    Polynomial ret(A);

    for (int i = 0; i <= B.ord; i++)
	ret.AddElm(i, B.coeff[i]);
    return ret;
}

// ---------------------------------------------------------
// Binary - for polynomials.
// ---------------------------------------------------------
Polynomial
operator-(const Polynomial& A, const Polynomial& B)
{
    Polynomial ret(A);

    for (int i = 0; i <= B.ord; i++)
	ret.AddElm(i, -B.coeff[i]);
    return ret;
}

// ---------------------------------------------------------
// Binary * for polynomials.
// ---------------------------------------------------------
Polynomial
operator*(const Polynomial& A, const Polynomial& B)
{
    Polynomial ret = A;
    ret *= B;
    return ret;
}

// ---------------------------------------------------------
// Scale polynomial by constant.
// ---------------------------------------------------------
Polynomial
operator*(double x, const Polynomial& A)
{
    Polynomial ret(A);
    ret *= x;
    return ret;
}

// ---------------------------------------------------------
// Scale polynomial by constant.
// ---------------------------------------------------------
Polynomial
operator*(const Polynomial& A, double& x)
{
    Polynomial ret(A);
    ret *= x;
    return ret;
}

// ---------------------------------------------------------
// Output polynomial to stream.
// ---------------------------------------------------------
ostream&
operator<<(ostream& x, const Polynomial& y)
{
    int cnt = y.ord + 1;
    double *c = y.coeff + y.ord + 1;

    while (c > y.coeff+1)
	x << *--c << "x^" << --cnt << " + ";
    x << *--c;
    return x;
}

// ---------------------------------------------------------
// ModRF
//
//	uses the modified regula-falsi method to evaluate
// the root in interval [a,b] of the polynomial described in
// coef. The root is returned is returned in *val.  The
// routine returns zero if it can't converge.
// ---------------------------------------------------------
int
Polynomial::ModRF(double a, double b, double *val)
{
    int its;
    double fa, fb, x, fx, lfx;

    fa = Evaluate(a);
    fb = Evaluate(b);

    /*
     * if there is no sign difference the method won't work
     */
    if (fa * fb > 0.0)
	return 0;

    if (fabs(fa) < RELERROR) {
	*val = a;
	return 1;
    }

    if (fabs(fb) < RELERROR) {
	*val = b;
	return 1;
    }

    lfx = fa;

    for (its = 0; its < MAXIT; its++) {
	fx = Evaluate( x = ((fb * a - fa * b) / (fb - fa)) );

	if (fabs(x) > RELERROR) {
	    if (fabs(fx / x) < RELERROR) {
		*val = x;
		return 1;
	    }
	} else if (fabs(fx) < RELERROR) {
	    *val = x;
	    return 1;
	}

	if ((fa * fx) < 0) {
	    b = x;
	    fb = fx;
	    if ((lfx * fx) > 0)
		fa /= 2;
	} else {
	    a = x;
	    fa = fx;
	    if ((lfx * fx) > 0)
		fb /= 2;
	}

	lfx = fx;
    }

//  fprintf(stderr, "modrf overflow %f %f %f\n", a, b, fx);

    return 0;
}

// ---------------------------------------------------------
// modp
//
//	calculates the modulus of u(x) / v(x) leaving it in r,
//  it returns 0 if r(x) is a constant.
//  note: this function assumes the leading coefficient of v
//	is 1 or -1
// ---------------------------------------------------------
Polynomial
operator%(const Polynomial& u, const Polynomial& v)
{
    int j, k;
    Polynomial ret = u;
    Polynomial nv = Normalize(v);

    if (nv.coeff[nv.ord] < 0.0) {
	for (k = u.ord - nv.ord - 1; k >= 0; k -= 2)
	    ret.coeff[k] = -ret.coeff[k];

	for (k = u.ord - nv.ord; k >= 0; k--)
	    for (j = nv.ord + k - 1; j >= k; j--)
		ret.coeff[j] = -ret.coeff[j] - ret.coeff[nv.ord + k] * nv.coeff[j - k];
    } else {
	for (k = u.ord - nv.ord; k >= 0; k--)
	    for (j = nv.ord + k - 1; j >= k; j--)
		ret.coeff[j] -= ret.coeff[nv.ord + k] * nv.coeff[j - k];
    }

    k = nv.ord - 1;
    while (k >= 0 && fabs(ret.coeff[k]) < Polynomial::SMALL_ENOUGH) {
	ret.coeff[k] = 0.0;
	k--;
    }

    ret.ord = (k < 0) ? 0 : k;

    return ret;
}

// ---------------------------------------------------------
// Allocates and passes back the Sturm sequence array of
// polynomials for the given polynomial (*this).
// ---------------------------------------------------------
int
Polynomial::SturmSequence(Polynomial **passbk)
{
    Polynomial *ret;
    int i, numarr;

    ret = new Polynomial[ord+1];
    ret[0] = *this;
    ret[1] = Normalize(Derivative(*this));
    for (numarr = 2; ret[numarr - 1].ord > 0; numarr++) {
	ret[numarr] = Normalize(-(ret[numarr - 2] % ret[numarr - 1]));
    }
    *passbk = new Polynomial[numarr];
    for (i = 0; i < numarr - 1; i++)
	(*passbk)[i] = ret[i];
    (*passbk)[i] = -(ret[i - 2] % ret[i - 1]);
    delete[] ret;
    return numarr - 1;
}

// ---------------------------------------------------------
// Returns the number of real roots for the polynomial.
// ---------------------------------------------------------
int
NumRoots(Polynomial *sseq, int n, int *atneg, int *atpos)
{
    int i;
    int atposinf, atneginf;
    double f, lf;

    atposinf = atneginf = 0;

    /*
     * changes at positive infinity
     */
    lf = sseq[0].coeff[sseq[0].ord];

    for (i = 1; i <= n; i++) {
	f = sseq[i].coeff[sseq[i].ord];
	if (lf == 0.0 || lf * f < 0)
	    atposinf++;
	lf = f;
    }

    /*
     * changes at negative infinity
     */
    if (sseq[0].ord & 1)
	lf = -sseq[0].coeff[sseq[0].ord];
    else
	lf = sseq[0].coeff[sseq[0].ord];

    for (i = 1; i <= n; i++) {
	if (sseq[i].ord & 1)
	    f = -sseq[i].coeff[sseq[i].ord];
	else
	    f = sseq[i].coeff[sseq[i].ord];
	if (lf == 0.0 || lf * f < 0)
	    atneginf++;
	lf = f;
    }

    *atneg = atneginf;
    *atpos = atposinf;

    return(atneginf - atposinf);
}

// ---------------------------------------------------------
// Returns the number of sign changes.
// ---------------------------------------------------------
int
NumChanges(Polynomial *sseq, int n, double a)
{
    int i;
    int changes;
    double f, lf;

    changes = 0;

    lf = sseq[0].Evaluate(a);

    for (i = 1; i <= n; i++) {
	f = sseq[i].Evaluate(a);
	if (fabs(f) > Polynomial::SMALL_ENOUGH && (lf == 0.0 || lf * f < 0))
	    changes++;
	lf = f;
    }

    return changes;
}

// ---------------------------------------------------------
// Finds roots in the given interval.
// ---------------------------------------------------------
void
SBisect(Polynomial *sseq, int np,
	double min, double max,
	int atmin, int atmax,
	double *roots)
{
    double mid;
    int n1, n2, its, atmid, nroot;

    if ((nroot = atmin - atmax) == 1) {
	/*
	 * first try a less expensive technique.
	 */
	if (sseq[0].ModRF(min, max, roots))
	    return;

	/*
	 * if we get here we have to evaluate the root the hard
	 * way by using the Sturm sequence.
	 */
	for (its = 0; its < Polynomial::MAXIT; its++) {
	    mid = (min + max) / 2;

	    atmid = NumChanges(sseq, np, mid);

	    if (fabs(mid) > Polynomial::RELERROR) {
		if (fabs((max - min) / mid) < Polynomial::RELERROR) {
		    roots[0] = mid;
		    return;
		}
	    }
	    else if (fabs(max - min) < Polynomial::RELERROR) {
		roots[0] = mid;
		return;
	    }

	    if ((atmin - atmid) == 0)
		min = mid;
	    else
		max = mid;
	}

	if (its == Polynomial::MAXIT) {
	    cerr << "SBisect: overflow min " << min << " max " << max;
	    cerr << " diff " << max-min << " nroot " << nroot;
	    cerr << " n1 " << n1 << " n2 " << n2 << '\n';
	    roots[0] = mid;
	}

	return;
    }

    /*
     * more than one root in the interval, we have to bisect...
     */
    for (its = 0; its < Polynomial::MAXIT; its++) {
	mid = (min + max) / 2;

	atmid = NumChanges(sseq, np, mid);

	n1 = atmin - atmid;
	n2 = atmid - atmax;

	if (n1 != 0 && n2 != 0) {
	    SBisect(sseq, np, min, mid, atmin, atmid, roots);
	    SBisect(sseq, np, mid, max, atmid, atmax, &roots[n1]);
	    break;
	}

	if (n1 == 0)
	    min = mid;
	else
	    max = mid;
    }

    if (its == Polynomial::MAXIT) {
	fprintf(stderr, "sbisect: roots too close together\n");
	fprintf(stderr, "sbisect: overflow min %f max %f diff %e\
		nroot %d n1 %d n2 %d\n",
		min, max, max - min, nroot, n1, n2);
	for (n1 = atmax; n1 < atmin; n1++)
	roots[n1 - atmax] = mid;
    }
}

// --------------------------------------------------------
// Find roots of the polynomial.  This routine allocates an
// array of doubles for you and returns the number of roots.
// --------------------------------------------------------
int
Polynomial::FindRealRoots(double **passbk)
{
    int ret = 0;
    double roots[4];
    switch (ord) {
	case 1:     // Linear
	    *passbk = new double[1];
	    **passbk = -coeff[0] / coeff[1];
	    return 1;
	case 2:     // Quadratic
	    ret = SolveQuadric(coeff, roots);
	    if (ret) {
		*passbk = new double[ret];
		memcpy(*passbk, roots, ret*sizeof(double));
	    }
	    return ret;
	case 3:     // Cubic
	    ret = SolveCubic(coeff, roots);
	    if (ret) {
		*passbk = new double[ret];
		memcpy(*passbk, roots, ret*sizeof(double));
	    }
	    return ret;
	case 4:     // Quartic
	    ret = SolveQuartic(coeff, roots);
	    if (ret) {
		*passbk = new double[ret];
		memcpy(*passbk, roots, ret*sizeof(double));
	    }
	    return ret;
	default:    // Higher order
	    if (ord <= 0)
		return 0;
	    break;
    }
    // Find Sturm sequence
    Polynomial norm = Normalize(*this);
    Polynomial *sseq;
    int sn = SturmSequence(&sseq);

    // Find number of roots
    int atmin, atmax;
    int numroots = NumRoots(sseq, sn, &atmin, &atmax);

    if (! numroots) {
	delete[] sseq;
	return 0;
    }

    // Bracket roots
    double min = -1;
    int nchanges = NumChanges(sseq, sn, min);
    for (int i = 0; nchanges != atmin && i != MAXPOW; i++) {
	min *= 10;
	nchanges = NumChanges(sseq, sn, min);
    }
    if (nchanges != atmin) {
	cerr << "Polynomial::FindRealRoots: Unable to bracket all negative roots\n";
	exit(1);
    }
    double max = 1;
    nchanges = NumChanges(sseq, sn, max);
    for (i = 0; nchanges != atmax && i != MAXPOW; i++) {
	max *= 10;
	nchanges = NumChanges(sseq, sn, max);
    }
    if (nchanges != atmax) {
	cerr << "Polynomial::FindRealRoots: Unable to bracket all positive roots\n";
	exit(1);
    }
    *passbk = new double[numroots];
    SBisect(sseq, sn, min, max, atmin, atmax, *passbk);

    delete[] sseq;

    return numroots;
}

#if 0
int
main()
{
    Polynomial p = Triangle(2, 1, 2);
    Polynomial q = Triangle(1, 1, 2);
    Polynomial r = Triangle(2, 2, 1);
    cout << p << " evaluated at 5 is " << p.Evaluate(5) << '\n';
    cout << q << " evaluated at 5 is " << q.Evaluate(5) << '\n';
    cout << p << " * " << q << " =\n" << p*q << "\n\n";
    cout << p << " - " << 4*q << " =\n" << p-4*q << "\n\n";
    cout << "Derivative(" << p*q << ") = " << Derivative(p*q) << "\n\n";
    cout << p % q << "\n\n";
    cout << "   " << r << "\n % " << q << "\n = " << r%q << "\n\n";

    double *roots;
    Polynomial s = p*q*r;
    cout << s << '\n';

    Polynomial *sseq;
    int i, numarr;
    numarr = s.SturmSequence(&sseq);
    for (i = 0; i < numarr; i++)
	cout << sseq[i] << "\n\n";

    int numroots = s.FindRealRoots(&roots);
    for (i = 0; i < numroots; i++)
	cout << roots[i] << '\n';

    return 0;
}
#endif

