// ===================================================================
// texture.cpp
//	Texture-mapping support functions for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "world.h"

// -------------------------------------------------------------------
// ColorMap classes.
//	ColorMapEntry - encapsulates a single entry in the color map.
//	ColorMap - maps a number in the range [0, 1] to an interpolated
//		   vector given a set of ColorMapEntries.
// -------------------------------------------------------------------

ColorMap::~ColorMap()
{
    for (int i = 0; i < num_entries; i++)
        delete entries[i];
    delete[] entries;
}

void
ColorMap::AddEntry(ColorMapEntry *entry)
{
    if (num_entries >= max_entries) {
        ColorMapEntry **newentries = new ColorMapEntry*[max_entries <<= 1];
        for (int i = 0; i < num_entries; i++)
            newentries[i] = entries[i];
        delete[] entries;
        entries = newentries;
    }
    entries[num_entries++] = entry->Dup();
}

RGBColor
ColorMap::ComputeColor(float value)
{
    for (int i = 0; i < num_entries; i++)
	if (entries[i]->IsInside(value))
	    return entries[i]->ComputeColor(value);
    return RGBColor(0);
}

// ---------------------------------------------------------
// Textures
//	These convert points in 3-space into colors.
// ---------------------------------------------------------

// ---------------------------------------------------------
// Checkerboard texture.
// ---------------------------------------------------------

Checkerboard::Checkerboard(float XSize, float YSize,
			   Texture *Text1, Texture *Text2,
			   const Matrix& ref, Mapping *map):
	TextureMapping(ref, map), xsize(XSize), ysize(YSize),
	text1(Text1), text2(Text2)
{
}

RGBColor
Checkerboard::GetColor(const Vector3D& Int) const
{
    Vector3D v = PreprocessVector(Int);
    int U = posmod(v.x, xsize) < 0.5;
    int V = posmod(v.y, ysize) < 0.5;
    if (U ^ V)
    	return text1->GetColor(Int);
    else
    	return text2->GetColor(Int);
}

Texture *
Checkerboard::Dup() const
{
    return new Checkerboard(xsize, ysize, 
    			    text1->Dup(), text2->Dup(), 
			    reference, mapping->Dup());
}

// ---------------------------------------------------------
// RAWFile class
//	Provides a way to access .RAW files as generated by 
//	OORT, QRT, DKB, and other ray tracers.
// ---------------------------------------------------------

RAWFile::RAWFile(char *filename)
{
    unsigned char c, d;
    fptr = fopen(filename, "rb");
    if (! fptr) {
	fprintf(stderr, "Could not open %s\n", filename);
	exit(1);
    }
    c = fgetc(fptr);
    d = fgetc(fptr);
    width = (d << 8) | c;
    c = fgetc(fptr);
    d = fgetc(fptr);
    height = (d << 8) | c;
    cachedscanl = 0;
    scanl[0] = new unsigned char[3*width];
    scanl[1] = new unsigned char[3*width];
    fread(scanl[0], sizeof(unsigned char), 3*width, fptr);
    fread(scanl[1], sizeof(unsigned char), 3*width, fptr);
}

RAWFile::~RAWFile()
{
    delete[] scanl[0];
    delete[] scanl[1];
    fclose(fptr);
}

RGBColor
RAWFile::GetPixel(int x, int y)
{
    unsigned char *ptr, *temp;

    if (x < 0 || y < 0 || x >= width || y >= height)
	return RGBColor(0);

    switch (y - cachedscanl) {
	case -1:
	    temp = scanl[0];
	    scanl[0] = scanl[1];
	    scanl[1] = temp;
	    cachedscanl--;
	    Seek(y);
	    fread(scanl[0], sizeof(unsigned char), 3*width, fptr);
	    ptr = scanl[0];
	    break;
	case 0: 		    // (x, y) is in scanl[0]
	    ptr = scanl[0];
	    break;
	case 1: 		    // (x, y) is in scanl[1]
	    ptr = scanl[1];
	    break;
	case 2: 		    // (x, y) is just below scanl[1]
	    temp = scanl[0];
	    scanl[0] = scanl[1];
	    scanl[1] = temp;
	    cachedscanl++;
	    Seek(y);
	    fread(scanl[1], sizeof(unsigned char), 3*width, fptr);
	    ptr = scanl[1];
	    break;
	default:
	    cachedscanl = y;
	    if (y == height - 1)
	    	cachedscanl--;
	    Seek(cachedscanl);
	    fread(scanl[0], sizeof(unsigned char), 3*width, fptr);
	    Seek(cachedscanl+1);
	    fread(scanl[1], sizeof(unsigned char), 3*width, fptr);
	    ptr = scanl[(y == height - 1) ? 1 : 0];
	    break;
    }
    ptr += x;
    return RGBColor(ptr[0] / 255.0, ptr[width] / 255.0, ptr[width+width] / 255.0);
}

RAWFileInMemory::RAWFileInMemory(char *filename)
{
    unsigned char c, d;
    FILE *fptr = fopen(filename, "rb");
    if (! fptr) {
	fprintf(stderr, "Could not open %s\n", filename);
	exit(1);
    }
    c = fgetc(fptr);
    d = fgetc(fptr);
    width = (d << 8) | c;
    c = fgetc(fptr);
    d = fgetc(fptr);
    height = (d << 8) | c;
    scanl = new unsigned char *[3*width];
    for (int i = 0; i < height; i++) {
	int iscanl;
	c = fgetc(fptr);
	d = fgetc(fptr);
	iscanl = (d << 8) | c;
	if (iscanl != i) {
	    delete[] scanl;
	    return;
	}
	scanl[i] = new unsigned char[3*width];
	fread(scanl[i], sizeof(unsigned char), 3*width, fptr);
    }
}

RAWFileInMemory::~RAWFileInMemory()
{
    for (int i = 0; i < height; i++)
	delete[] scanl[i];
    delete[] scanl;
}

RGBColor
RAWFileInMemory::GetPixel(int x, int y)
{
    if (! scanl ||
        x < 0 || x >= width || 
	y < 0 || y >= height)
	return RGBColor(0);
    unsigned char *ptr = scanl[y] + x;
    return RGBColor(ptr[0] / 255.0, ptr[width] / 255.0, ptr[width+width] / 255.0);
}

// ---------------------------------------------------------
// OpenFile--given a filename, evaluates the file type and returns a
//	     pointer to the corresponding ImageFile type.
//	     For now, RAW files are the only type supported.
// ---------------------------------------------------------
ImageFile *
OpenFile(char *filename)
{
    FILE *fptr = fopen(filename, "rb");
    if (! fptr)
	return 0;
    fclose(fptr);
    return new RAWFile(filename);
}

// ---------------------------------------------------------
// ImageMap texture
// ---------------------------------------------------------

ImageMap::ImageMap(const Matrix& ref, Mapping *map,
		   int Interp,
		   int Once,
		   char *filename, 
		   float xscale, float yscale) : TextureMapping(ref, map)
{
    interp = (Interp != 0);
    once = (Once != 0);
    file = OpenFile(filename);
    scale = Vector3D(xscale, yscale, 0);
}

ImageMap::ImageMap(const Matrix& ref, Mapping *map,
		   int Interp,
		   int Once,
		   ImageFile *File,
		   float xscale, float yscale) : 
    TextureMapping(ref, map)
{
    interp = (Interp != 0);
    once = (Once != 0);
    file = File;
    scale = Vector3D(xscale, yscale, 0);
}

RGBColor
ImageMap::GetColor(const Vector3D& Intersection) const
{
    Vector3D v = PreprocessVector(Intersection) * scale;

    v *= Vector3D(file->Width(), file->Height(), 0);
    if (! once) {
	v.x = fmod(v.x, file->Width());
	if (v.x < 0)
	    v.x += file->Width();
	v.y = fmod(v.y, file->Height());
	if (v.y < 0)
	    v.y += file->Height();
    }
    long ulx = (long) v.x;
    long uly = (long) v.y;
    if (interp) {
	float horzweight = v.x - ulx;
	float vertweight = v.y - uly;
	float ul = (1 - horzweight) * (1 - vertweight);
	float ur = horzweight * (1 - vertweight);
	float ll = (1 - horzweight) * vertweight;
	float lr = horzweight * vertweight;
	return ul * file->GetPixel(ulx, uly) +
	       ur * file->GetPixel(ulx+1, uly) +
	       ll * file->GetPixel(ulx, uly+1) +
	       lr * file->GetPixel(ulx+1, uly+1);
    }
    else
	return file->GetPixel(ulx, uly);
}

Texture *
ImageMap::Dup() const
{
    if (mapping)
	return new ImageMap(reference, mapping, interp, once, file);
    else
	return new ImageMap(reference, 0, interp, once, file);
}

// ---------------------------------------------------------
// SolidTexture class
//	All solid textures are derived from this guy.
// ---------------------------------------------------------

SolidTexture::SolidTexture(const Matrix& ref,
			   ColorMap *CMap): 
    Texture(ref)
{
    cmap = CMap;
}

// -------------------------------------
// Bozo texture
//	Just maps the solid noise value 
//	at the intersection point into 
//	a color map.
// -------------------------------------

Bozo::Bozo(const Matrix& ref, float Turbulence, ColorMap *CMap):
    SolidTexture(ref, CMap)
{
    turbulence = Turbulence;
}

RGBColor
Bozo::GetColor(const Vector3D& Intersection) const
{
    Vector3D local = PreprocessVector(Intersection);
    float noise;

    if (turbulence != 0.0)
	local += GlobalNoise::Noise->DTurbulence(Intersection) * turbulence;
    noise = GlobalNoise::Noise->Noise(local);
    return cmap->ComputeColor(noise);
}

Texture *
Bozo::Dup() const
{
    return new Bozo(reference, turbulence, cmap);
}

// -------------------------------------
// Wood texture.
// -------------------------------------

Wood::Wood(const Matrix& Ref, float Turbulence, ColorMap *CMap):
    SolidTexture(Ref, CMap)
{
    turbulence = Turbulence;
}

RGBColor
Wood::GetColor(const Vector3D& Intersection) const
{
    Vector3D v = PreprocessVector(Intersection);

    Vector3D turb = GlobalNoise::Noise->DTurbulence(v);

    v += Vector3D(cycloidal(v.x + turb.x),
		  cycloidal(v.y + turb.y),
		  0) * turbulence;
    float len = Magnitude(v);
    float noise = triangle_wave(len);
    return cmap->ComputeColor(noise);
}

Texture *
Wood::Dup() const
{
    return new Wood(reference, turbulence, cmap);
}

// -------------------------------------
// Marble texture.
// -------------------------------------

Marble::Marble(const Matrix& Ref, float Turbulence, ColorMap *CMap):
	SolidTexture(Ref, CMap)
{
    turbulence = Turbulence;
}

RGBColor
Marble::GetColor(const Vector3D& Intersection) const
{
    Vector3D v = PreprocessVector(Intersection);
    float noise = triangle_wave(v.x +
				GlobalNoise::Noise->Turbulence(v) *
				turbulence);
    return cmap->ComputeColor(noise);
}

Texture *
Marble::Dup() const
{
    return new Marble(reference, turbulence, cmap);
}

// -------------------------------------
// Gradient texture.
// -------------------------------------

Gradient::Gradient(const Matrix& Ref, float Turbulence, ColorMap *CMap):
    SolidTexture(Ref, CMap)
{
    turbulence = Turbulence;
}

RGBColor
Gradient::GetColor(const Vector3D& Intersection) const
{
    Vector3D v = PreprocessVector(Intersection);
    if (turbulence != 0.0)
	v += GlobalNoise::Noise->DTurbulence(v) * turbulence;
    return cmap->ComputeColor(posmod(posmod(v.x, 1) +
				     posmod(v.y, 1) +
				     posmod(v.z, 1), 1));
}

Texture *
Gradient::Dup() const
{
    return new Gradient(reference, turbulence, cmap);
}

// ---------------------------------------------------------
// WoodUpstill
//	Wood texture derived from Listing 16.15 in Steve 
//	Upstill's book, _The RenderMan Companion_.
//	(Addison-Wesley, 1990).
// ---------------------------------------------------------

WoodUpstill::WoodUpstill(const Matrix& ref,
			 float _ringscale,
			 const RGBColor& _light,
			 const RGBColor& _dark): 
    SolidTexture(ref, 0)
{
    ringscale = _ringscale;
    light = _light;
    dark = _dark;
}

RGBColor
WoodUpstill::GetColor(const Vector3D& _p) const
{
    Vector3D p = PreprocessVector(_p);
    p += GlobalNoise::Noise->DNoise(p);

    float r = hypot(p.y, p.z) * ringscale;
    r += GlobalNoise::Noise->Noise(r);
    r -= floor(r);
    r = Hermite(0, 0.8, r) - Hermite(0.83, 1.0, r);
    return MixColors(light, dark, r);
}

Texture *
WoodUpstill::Dup() const
{
    return new WoodUpstill(reference, ringscale, light, dark);
}

// ---------------------------------------------------------
// Granite
//	Granite texture derived from Listing 16.15 in Steve 
//	Upstill's book, _The RenderMan Companion_.
//	(Addison-Wesley, 1990).
// ---------------------------------------------------------

Granite::Granite(const RGBColor& _K,
		 const Matrix& _reference):
    SolidTexture(_reference, 0)
{
    K = _K;
}

RGBColor
Granite::GetColor(const Vector3D& _p) const
{
    Vector3D p = PreprocessVector(_p);
    float sum = 0, freq = 1;
    for (int i = 0; i < 6; i++) {
	sum += fabs(0.5 - GlobalNoise::Noise->Noise(4 * freq * p)) / freq;
	freq *= 2;
    }
    return K * sum;
}

Texture *
Granite::Dup() const
{
    return new Granite(K, reference);
}

