// ===================================================================
// world.cpp
//	World-related routines in ray tracer.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include <alloc.h>
#include "oort.h"
#include "world.h"

// World constructor.
World::World()
{
    Ambient = RGBColor(0);
    Background = RGBColor(0.1, 0.1, 1.0);
    ViewTransform = IdentityMatrix();
    Filename = 0;
    HorzRes = 320;
    VertRes = 200;
    StartScan = 0;
    EndScan = 200;
    printpers = interactive = 1;
    printhierarchy = 0;
    numtries = 1;
    hierarchyseed = 0;
    DepthLimit = 8;

    mode = World::None;
    MinSamples = 4;
    MaxSamples = 16;
    NumCandidates = 10;
    ContrastThreshold = RGBColor(0.4, 0.3, 0.6);
}

// First of a whole set of routines for setting World
// parameters.	This one sets the output filename.
// The output filename is initially NULL, so it is
// strdup'd (and free'd if the output file is changed
// by a subsequent call to SetOutputFile).
void
World::SetOutputFile(char *NewFilename)
{
    if (Filename)
	free(Filename);
    Filename = (char *) malloc(strlen(NewFilename) + 1);
    strcpy(Filename, NewFilename);
}

// Set horizontal resolution of the output.  This is
// the number of pixels across.
void
World::SetHorzRes(int NewHorzRes)
{
    HorzRes = NewHorzRes;
}

// Set vertical resolution of the output.  This is
// the number of pixels down.
void
World::SetVertRes(int NewVertRes)
{
    VertRes = NewVertRes;
}

// Set height of screen.  This is the height of the
// virtual screen in 3-space.
void
World::SetScreenHeight(float NewScreenHeight)
{
    ScreenHeight = NewScreenHeight;
}

// Set width of screen.  This is the width of the
// virtual screen in 3-space.
void
World::SetScreenWidth(float NewScreenWidth)
{
    ScreenWidth = NewScreenWidth;
}

// Set screen distance.  This is the distance from
// viewer to screen in 3-space.
void
World::SetScreenDistance(float NewScreenDist)
{
    ScreenDist = NewScreenDist;
}

// Set depth limit on recursion.
void
World::SetDepthLimit(int NewDepthLimit)
{
    DepthLimit = NewDepthLimit;
}

// Set viewer parameters.  ViewerLoc is where the viewer
// looks from; LookAt is the point the viewer looks at;
// and UpVector is the normalized direction of upward
// (traditionally (0.0, 1.0, 0.0).
void
World::SetViewerParameters(Vector3D& NewLookAt,
			   Vector3D& NewViewerLoc,
			   Vector3D& NewUpVector)
{
    ViewerLoc = NewViewerLoc;
    LookAt = NewLookAt;
    UpVector = NewUpVector;
    ViewTransform = ViewMatrix(LookAt, ViewerLoc, UpVector);
}

// Set ambient lighting.  Generally a small gray level such
// as (0.1, 0.1, 0.1).
void
World::SetAmbientLight(RGBColor& NewAmbient)
{
    Ambient = NewAmbient;
}

void
World::SetNumTries(int _numtries)
{
    numtries = _numtries;
}

void
World::SetAntiAliasMode(enum AntiAliasModes _mode)
{
    mode = _mode;
}

void
World::SetSamples(int _MinSamples, int _MaxSamples)
{
    if (_MinSamples == 0) {
	MaxSamples = _MaxSamples;
	return;
    }
    if (_MaxSamples == 0) {
	MinSamples = _MinSamples;
	return;
    }
    if (_MinSamples < _MaxSamples) {
	MinSamples = (_MinSamples < 1) ? 1 : _MinSamples;
	MaxSamples = _MaxSamples;
    }
    else {
	MinSamples = (_MaxSamples < 1) ? 1 : _MaxSamples;
	MaxSamples = _MinSamples;
    }
}

void
World::SetNumCandidates(int _NumCandidates)
{
    NumCandidates = _NumCandidates;
}

void
World::SetContrastThreshold(const RGBColor& _ContrastThreshold)
{
    ContrastThreshold = _ContrastThreshold;
}

// Set background color.  Default is some kind of blue.
void
World::SetBackgroundColor(RGBColor& NewBackground)
{
    Background = NewBackground;
}

// Set scanline to start tracing.  Default is 0.
void
World::SetStart(int Start)
{
    StartScan = Start;
}

// Set scanline to end tracing.  Default is VertRes.
void
World::SetEnd(int End)
{
    EndScan = End;
}

// Add a light to the World.
void
World::AddLight(Light *newlight)
{
    lights.AddToList(newlight);
}

// Add an instance of an object to the list in the world.
void
World::AddObject(Object3D& Obj)
{
    Objects.AddToList(Obj.Dup());
}

// Add an object to the list.
void
World::AddObject(Object3D *Obj)
{
    Objects.AddToList(Obj);
}

Object3D *
World::Find_Nearest(Ray& ray, float& t)
{
    float mint; 	// Minimum parameter of intersection found thus far
    Object3D *nearest;	// Nearest object found thus far
    float tempt;
    Heap<KayKajiyaNode *> TraversalHeap;

    nearest = 0;
    mint = Limits::Infinity;

    for (ObjectList::Iterator sc(Unboundeds); sc.Valid(); sc.GotoNext()) {
	if (sc.Contents()->NearestInt(ray, tempt, mint)) {
	    nearest = sc.Contents();
	    mint = tempt;
	}
    }

    // If intersect with overall bounding volume, place it in heap.
    if (! Objects.Empty()) {
 	Object3D *head = *Objects.Head();
	if (head->NearestInt(ray, tempt, mint)) {
	    if (head->Flags() & BOUNDING) {
	        KayKajiyaNode *NewNode = new KayKajiyaNode(head, tempt);
	    	TraversalHeap.Insert(NewNode);
	    }
	    else {
	    	if (tempt < mint) {
		    mint = tempt;
		    nearest = head;
	    	}
	    }
    	}
    }

    // Loop to find nearest intersection
    KayKajiyaNode *c;
    while (TraversalHeap.N()) {
	c = TraversalHeap.ExtractMin();

	if (! c) {
	    cerr << "Null pointer extracted.  Out of memory.\n";
	    exit(1);
	}

	// If the nearest bounding volume is farther away than
	// our running minimum intersection, the heap contains no
	// bounding volumes or objects nearer than what we already have.
	if ( (mint < c->t && fabs(mint - c->t) > Limits::Small) &&
	      c->t < Limits::Infinity) {
	    delete c;
	    break;
	}

	// Traverse all objects listed in bounding volume
	ObjectList::Iterator sc( ((BoundingVolume *) c->obj)->contents);

	while (sc.Valid()) {
	    if (sc.Contents()->NearestInt(ray, tempt, mint)) {
	    	// If hit bounding volume, insert into heap.
	    	if (sc.Contents()->Flags() & BOUNDING) {
		    KayKajiyaNode *NewNode = new KayKajiyaNode(sc.Contents(), tempt);
		    TraversalHeap.Insert(NewNode);
	        }
		// If hit nonbounding object, check if nearest.
		else {
	    	    if (tempt < mint) {
			mint = tempt;
			nearest = sc.Contents();
		    } // if (tempt < mint)
	        } // else
	    } // if (NearestInt)
	    sc.GotoNext();
    	} // for (each object in bounding volume)
	delete c;
    } // while (heap is not empty)

    // Delete everything remaining in heap
    while (TraversalHeap.N())
	delete TraversalHeap.ExtractMin();

    // Pass back parameter of nearest intersection
    t = mint;

    // Return nearest object intersected, if any
    return nearest;
}

// Function to find the attenuation of a given shadow ray.
// Returns opaque object if one is found (for later caching),
// or 0 if none was found.  The Atten is an RGBColor which
// describes the attenuation (initially (1.0, 1.0, 1.0),
// chipped away by transparent objects on ray).
Object3D *
World::FindAtten(ObjectList &olist,
		 Ray& ray,
		 RGBColor& Atten,
		 float maxt)
{
    ObjectList::Iterator optr(olist);
    Object3D *ret;
    ObjectList trans;		  // List of transparent objects
				  // Used only if no opaque object is found

    // Scan list of objects for opaque or transparent.
    while (optr.Valid()) {
	Object3D *contents = optr.Contents();
	// If object is transparent, save it for later.
	// Intersecting an opaque one will obviate the need to
	// intersect with this one.
	if ((! (contents->flags & BOUNDING)) &&
	    contents->surf &&
	    contents->surf->Transparent()) {
	    trans.AddToList(contents);
	    optr.GotoNext();
	    continue;
	}

	if ((! contents->flags & BOUNDING) && (! contents->surf)) {
	    optr.GotoNext();
	    continue;
	}

	if (contents->HitAtAll(ray, maxt)) {
	    // If intersect with bounding volume, recurse
	    if (contents->flags & BOUNDING) {
		ret = FindAtten( ((BoundingVolume *) contents)->contents,
				     ray, Atten, maxt);
		if (ret)
		    return ret;
	    }
	    // Otherwise, deal with leaf object.
	    else {
		Atten = RGBColor(0);	// Clear Atten
		return contents;
	    }
	}
	// Go on to next object in list
	optr.GotoNext();
    }

    // No opaque object was intersected, go through the list
    // of transparent ones.
    while (trans.ListCount()) {
	Object3D *obj = trans.ExtractHead();
	SpanList *spans = obj->FindAllIntersections(ray);

	if (spans) {
	    // Find first intersection greater than Limits::Threshold
	    while (spans->NumSpans()) {
		Span span = spans->ExtractMinSpan();

		if (span.tmin > Limits::Threshold && span.tmin < maxt) {
		    if (span.omin->surf && span.omin->surf->Transparent())
	        	Atten *= span.omin->surf->Transmitted(ray.Extrap(span.tmin));
		    else {
			Atten = RGBColor(0);
			return span.omin;
		    }
		}
		if (span.tmax > Limits::Threshold && span.tmax < maxt) {
		    if (span.omax->surf && span.omax->surf->Transparent())
			Atten *= span.omax->surf->Transmitted(ray.Extrap(span.tmax));
		    else {
			Atten = RGBColor(0);
			return span.omax;
		    }
		}
	    }
	    delete spans;
	}
    }

    // No opaque object found, return 0
    return 0;
}

Object3D *
World::ShadowAtten(Ray& ray, RGBColor& Atten, float maxt)
{
    Object3D *first = FindAtten(Unboundeds, ray, Atten, maxt);
    if (first)
	return first;
    else
	return FindAtten(Objects, ray, Atten, maxt);
}

// Compute diffuse and specular contributions to an intersection's
// color.  These are computed together because they are both directly
// due to to light sources' light.
RGBColor
World::DiffuseLight(ShadingInfo& shade, Surface *surf)
{
    RGBColor Contrib(0);
    for (LightList::Iterator sc(lights); sc.Valid(); sc.GotoNext()) {
	Contrib += sc.Contents()->ShadowLight(shade, surf);
    }

    return Contrib;
}

// Find color of a given ray.  Take and pass back pointer to
// pointer to nearest object found for subsequent caching.
RGBColor
World::FindColor(Ray& ray, int depth)
{
    Object3D *obj;
    float t;

    if (obj = Find_Nearest(ray, t))
	return obj->ShadeObject(*this, ray, ray.Extrap(t), depth);
    else
	return Background;
}

// Put a scanline in .RAW format out to given ostream.
void
PutScanline(ostream& out, RGBColor *scanl, int num_clrs)
{
    int i;
    unsigned char *r = new unsigned char[num_clrs];
    unsigned char *g = new unsigned char[num_clrs];
    unsigned char *b = new unsigned char[num_clrs];

    for (i = 0; i < num_clrs; i++) {
	RGBColor currval = NormalizeColor(scanl[i]);
	r[i] = (unsigned char) (currval.x * 255.0);
	g[i] = (unsigned char) (currval.y * 255.0);
	b[i] = (unsigned char) (currval.z * 255.0);
    }
    out.write((char *) r, num_clrs * sizeof(unsigned char));
    out.write((char *) g, num_clrs * sizeof(unsigned char));
    out.write((char *) b, num_clrs * sizeof(unsigned char));
    delete[] r;
    delete[] g;
    delete[] b;
}

ObjectList
World::DisbandAggregates(ObjectList& list)
{
    ObjectList ret;

    while (list.ListCount()) {
    	Object3D *obj = list.ExtractHead();
    	if (obj->Flags() & AGGREGATE) {
    	    ret.AddToList(DisbandAggregates(((Aggregate *) obj)->List()));
	}
    	else
    	    ret.AddToList(obj);
    }
    return ret;
}

void
World::DrawScreen(long beg, int Col, int Row)
{
    long now;
    clrscr();
    cout << "OORT v1.0 in progress:\n\n";

//    cout << heapsize() << " bytes left\n";
    cout << "Primary rays traced: " << Statistics::PrimaryRays << '\n';

    long pixels = (long) HorzRes * Row + Col;
    cout << "Pixels computed: " << pixels << " / " << NumPixels << '\n';
    time(&now);
    float avg;
    cout << "Seconds elapsed rendering: " << now - beg << '\n';

    if (Statistics::PrimaryRays)
    	avg = (float) (now - beg) / Statistics::PrimaryRays;
    else
    	avg = 0;
    cout << "Average time per primary ray (s): " << avg << " s\n";

    if (pixels) {
	avg = (float) (now - beg) / pixels;
    }
    else
        avg = 0;
    cout << "Average time per pixel (s): " << avg << " s\n";
    float raysperpix = (pixels) ? (float) Statistics::PrimaryRays / pixels : 0;
    cout << "Average rays per pixel: " << raysperpix << '\n';
    cout << "Estimated remaining time: "
	     << avg * (NumPixels - pixels) << " s\n";
}

Vector3D
World::NewSample(Vector3D *samples, int n)
{
    Vector3D ret(RandomValue(0, 1), RandomValue(0, 1), 0);
    if (! n)
	return ret;

    float mindist = MinDist(ret, samples, n);
    for (int i = 0; i < NumCandidates; i++) {
	Vector3D candidate(RandomValue(0, 1), RandomValue(0, 1), 0);
	float dist = MinDist(candidate, samples, n);
	if (dist > mindist) {
	    ret = candidate;
	    mindist = dist;
	}
    }
    return ret;
}

RGBColor
World::PixelColor(float xtarg, float ytarg)
{
    RGBColor ret;
    Vector3D target;

    switch (mode) {
	case None:
	    target = Vector3D(xtarg + ScreenWidth / HorzRes / 2,
			      ytarg + ScreenHeight / VertRes / 2,
			      -ScreenDist);
	    Statistics::PrimaryRays += 1;
	    ret = FindColor(Ray(Vector3D(0), target), 0);
	    break;
	case Adaptive:
	    {
		RGBColor Imin(MAXFLOAT), Imax(-MAXFLOAT), Intens(0);
		Vector3D *pts = new Vector3D[MaxSamples];
		int i;

		for (i = 0; i < MaxSamples; i++) {
		    pts[i] = NewSample(pts, i);

		    Vector3D target = pts[i];
		    target *= Vector3D(ScreenWidth / HorzRes, ScreenHeight / VertRes, 0);
		    target += Vector3D(xtarg, ytarg, -ScreenDist);

		    Statistics::PrimaryRays += 1;
		    RGBColor clr = FindColor(Ray(Vector3D(0), target), 0);
		    Intens += clr;
		    clr = NormalizeColor(clr);
		    Minimize(Imin, clr);
		    Maximize(Imax, clr);

		    if (i >= MinSamples) {
			if (i >= MaxSamples)
			    break;
			if (! MoreRays(Imin, Imax, ContrastThreshold))
			    break;
		    }

		}
		delete[] pts;
		Statistics::RayPixHist[i - 1] += 1;
		ret = Intens / (i + 1);
	    }
    }
    return ret;
}

// Trace rays, send to output file.
void
World::RayTrace()
{
    float x, y, incx, incy;
    int ix, iy;
    ofstream OutputStream;
    RGBColor *scanl = new RGBColor[HorzRes];    // Colors of scan line
    long beg, end;	  // Times in seconds

    // Get the time before beginning trace so we can report ET
    time(&beg);

    // Transform the object database according to the view transform
    Objects.TransformList(ViewTransform);
    SurfaceList clist;
    Objects.TransformSurfaces(clist, Invert(ViewTransform));

    Objects = DisbandAggregates(Objects);
    RemoveUnboundeds();

    if (hierarchyseed)
	srand(hierarchyseed);
    else
	randomize();
    BuildHierarchy(numtries);

    if (Objects.Head())
        CullSingletons(*Objects.Head(), &Objects);

    for (ObjectList::Iterator sc(Objects); sc.Valid(); sc.GotoNext()) {
        if (sc.Contents()->Flags() & BOUNDING)
	    AxisAlignedBox bbox = sc.Contents()->BBox();
    }

    if (printhierarchy) {
        cout << "Hierarchy built.  It is as follows:\n";
        for (ObjectList::Iterator sc(Objects); sc.Valid(); sc.GotoNext())
	    sc.Contents()->Describe(0);
    }

    CountObjects();

    // Transform the lights according to the view transform
    for (LightList::Iterator lightsc(lights); lightsc.Valid(); lightsc.GotoNext())
        lightsc.Contents()->ApplyTransform(ViewTransform);

    // Open up the output file
    if (! Filename) {
        cerr << "No output file specified.\n";
        exit(2);
    }
    OutputStream.open(Filename, ios::binary);
    if (! OutputStream) {
        cerr << "Cannot open " << Filename << " for output.\n";
        cerr << "Error " << OutputStream.rdstate() << '\n';
        exit(2);
    }

    // Write the width and height to the output file
    OutputStream.write((char *) &HorzRes, sizeof(int));
    OutputStream.write((char *) &VertRes, sizeof(int));
    Statistics::HierarchyHeuristic = HierarchyCost(Objects);

    time(&end);
    Statistics::PrepTime = end - beg;
    beg = end;

    if (interactive)
        DrawScreen(beg, 0, 0);

    // Begin trace.
    y = ScreenHeight / 2;
    incy = - ScreenHeight / VertRes;
    y += incy * StartScan;

    incx = ScreenWidth / HorzRes;

    NumPixels = ((long) EndScan - StartScan) * HorzRes;

    for (iy = StartScan; iy < EndScan; iy++, y += incy) {
	if ((! interactive) && printpers)
	    cerr << '.';
	else
	    DrawScreen(beg, 0, iy - StartScan);
	OutputStream.write((char *) &iy, sizeof(int));
	x = -ScreenWidth / 2;
	for (ix = 0; ix < HorzRes; ix++, x += incx) {
	    scanl[ix] = PixelColor(x, y);
	    if (interactive) {
		if (kbhit()) {
		    // Clear buffer
		    (void) getch();
		    DrawScreen(beg, ix, iy - StartScan);
		}
	    }
	}
	PutScanline(OutputStream, scanl, HorzRes);
    }
    if ((! interactive) && printpers)
    	cerr << '\n';
    else
    	clrscr();

    OutputStream.close();
    time(&end);
    Statistics::RenderTime = end - beg;
    delete scanl;
}

void
World::ParseCommandLine(int argc, char *argv[])
{
    char temp[256];	  // Temporary place to put filename before copying

    argc--;	  // Bump past the invocation string
    argv++;
    while (argc--) {
	if (argv[0][0] != '-') {
	    argv++;
	    continue;
	}
	if (! strncmp(argv[0]+1, "seed=", 5)) {
	    int seed;
	    char *sc;
	    if (GlobalNoise::Noise)
	    	delete GlobalNoise::Noise;
	    seed = strtol(argv[0]+6, &sc, 10);
	    if (*sc != '\0') {
		    cout << "Specify integer for seed.\n";
    		Usage();
	    }
	    GlobalNoise::Noise = new PerlinNoise(seed);
	}
	else if (strncmp(argv[0]+1, "hierseed", 8) == 0) {
	    hierarchyseed = atoi(argv[0]+9);
	}
	else if (strncmp(argv[0]+1, "interactive", 11) == 0) {
	    if (argv[0][12] == '+')
		interactive = 1;
	    else if (argv[0][12] == '-')
		interactive = 0;
	    else {
		cout << "Specify + or - with interactive option\n";
		Usage();
	    }
	}
	else if (strncmp(argv[0]+1, "anti", 4) == 0) {
	    SetAntiAliasMode(Adaptive);
	}
	else if (strncmp(argv[0]+1, "min", 3) == 0) {
	    SetSamples(atoi(argv[0]+4), 0);
	}
	else if (strncmp(argv[0]+1, "max", 3) == 0) {
	    SetSamples(0, atoi(argv[0]+4));
	}
	else if (strncmp(argv[0]+1, "depth", 1) == 0) {
	    DepthLimit = atoi(*argv + 6);
	}
	else if (strncmp(argv[0]+1, "tries", 1) == 0) {
	    numtries = atoi(*argv + 6);
	}
	else if (strncmp(argv[0]+1, "S", 1) == 0) {
	    Statistics::AppendTo = *argv + 2;
	}
	else if (strncmp(argv[0]+1, "h", 1) == 0) {
	    VertRes = atoi(*argv + 2);
	    StartScan = 0;
	    EndScan = VertRes;
	}
	else if (strncmp(argv[0]+1, "o", 1) == 0) {
	    strcpy(temp, argv[0] + 2);
	    if (! strchr(temp, '.'))
		strcat(temp, ".raw");
	    SetOutputFile(temp);
	}
	else if (strncmp(argv[0]+1, "quiet", 5) == 0) {
	    printhierarchy = 0;
	}
	else if (strncmp(argv[0]+1, "s", 1) == 0) {
	    StartScan = atoi(*argv + 2);
	}
	else if (strncmp(argv[0]+1, "e", 1) == 0) {
	    EndScan = atoi(*argv + 2);
	}
	else if (strncmp(argv[0]+1, "t", 1) == 0) {
	    Limits::Threshold = atof(*argv + 2);
	}
	else if (strncmp(argv[0]+1, "w", 1) == 0) {
	    HorzRes = atoi(*argv + 2);
	}
	else
	    cout << "Ignoring unknown option " << argv[0] << '\n';
	argv++;
    }
}

