// int.cpp: main routines for RAW2GIF
// Copyright (C) 1991 by Nicholas Wilt.  All rights reserved.

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <dos.h>
#include <math.h>
#include <limits.h>
#include <alloc.h>
#include <fstream.h>
#include <conio.h>
#include <string.h>
#include <time.h>
#include "hist.h"
#include "lsearch.h"
#include "colorbox.h"
#include "heap.h"
#include "image.h"

#define TURBO

int red[256], green[256], blue[256];
static ColorBoxNode *boxes;
static ColorBox ****whichbox;
static ColorBoxNode ****checklist;
int numboxes = 0;

void
findreps(int y,
	 unsigned char *r,
	 unsigned char *g,
	 unsigned char *b,
	 int width)
{
  while (width--) {
    ColorBox *box = whichbox[*r >> 3][*g >> 3][*b >> 3];
    box->repr += *r++;
    box->repg += *g++;
    box->repb += *b++;
    box->avg_by++;
  }
}

LocalSearch *locsearch;

ColorBoxNode *
Quantize(Histogram& hist, int num_clrs)
{
  ColorVector min, max;
  ColorBox *currbox;
  Heap h(comp_cboxes);
  ColorBoxNode *singles = 0;
  ColorBoxNode *currnode;
  int num_singles = 0;
  ColorBox *new1, *new2;

  min.r = min.g = min.b = 0;
  max.r = max.g = max.b = 32;
  currbox = new ColorBox(min, max, &hist);
  h.Insert(currbox);
  while (h.N() && (h.N() + num_singles < num_clrs)) {
    do {
      currbox = (ColorBox *) h.ExtractMin();
      if (currbox->volume == 1) {
	currnode = new ColorBoxNode;
	currnode->box = currbox;
	currnode->next = singles;
	singles = currnode;
	num_singles++;
      }
    } while (currbox->volume == 1 && h.N());
    if (currbox->volume != 1) {
      currbox->Split(&new1, &new2);
      delete currbox;
      h.Insert(new1);
      h.Insert(new2);
    }
  }
  while (h.N()) {
    currbox = (ColorBox *) h.ExtractMin();
    currnode = new ColorBoxNode;
    currnode->box = currbox;
    currnode->next = singles;
    singles = currnode;
  }
  return singles;
}

void FindWhichBox(Histogram& hist)
{
  int i, j, k;
  ColorBoxNode *sc;

  whichbox = new ColorBox***[32];
  for (i = 0; i < 32; i++)
    whichbox[i] = new ColorBox**[32];
  for (i = 0; i < 32; i++)
    for (j = 0; j < 32; j++) {
      whichbox[i][j] = new ColorBox*[32];
      for (k = 0; k < 32; k++)
	whichbox[i][j][k] = 0;
    }
  for (sc = boxes; sc; sc = sc->next) {
    numboxes++;
    if (sc->box->avg_by)
      cerr << "avg_by already nonzero in " << sc->box << '\n';
    for (i = sc->box->min.r; i < sc->box->max.r; i++)
      for (j = sc->box->min.g; j < sc->box->max.g; j++)
	for (k = sc->box->min.b; k < sc->box->max.b; k++)
	  whichbox[i][j][k] = sc->box;
  }
}

LocalSearch *FindAverages()
{
  int i;
  ColorBoxNode *sc, *prev, *min;
  ColorBox *temp;
  unsigned minint;
  long numreps;
  Color *clrs;
  int numclrs = 0;
  int inx;

  i = 0;
  min = 0;
  minint = 65535;
  for (sc = boxes; sc; sc = sc->next) {
    numclrs++;
    if (sc->box->avg_by) {
      unsigned scint;

      sc->box->repr /= sc->box->avg_by;
      sc->box->repg /= sc->box->avg_by;
      sc->box->repb /= sc->box->avg_by;
      sc->box->inx = i++;

      scint = 77 * (int) sc->box->repr +
	     151 * (int) sc->box->repg +
	      28 * (int) sc->box->repb;

      if (scint < minint) {
	min = sc;
	minint = scint;
      }
    }
    else
      cerr << "avg_by is 0 in FindAverages\n";
  }
  // Ensure that the border around the screen is of
  // minimal intensity.
  if (min != boxes->box) {
    boxes->box->inx = min->box->inx; // Swap indices
    min->box->inx = 0;

    temp = min->box;		// Swap pointers
    min->box = boxes->box;
    boxes->box = temp;
  }
  for (sc = boxes; sc; sc = sc->next) {
    if (sc->box->repr < 0 || sc->box->repr > 255 ||
	sc->box->repg < 0 || sc->box->repg > 255 ||
	sc->box->repb < 0 || sc->box->repb > 255)
      cerr << "Invalid color in FindAverages\n";
    red[sc->box->inx] = sc->box->repr;
    green[sc->box->inx] = sc->box->repg;
    blue[sc->box->inx] = sc->box->repb;
  }
  clrs = new Color[numclrs];
  inx = numclrs;
  for (sc = boxes; sc; sc = sc->next) {
    clrs[--inx] = Color(sc->box->inx,
			sc->box->repr,
			sc->box->repg,
			sc->box->repb);
  }
  return new LocalSearch(clrs, numclrs);
}

void parse_commandline(int argc,
		       char *argv[],
		       char *infile,
		       char *outfile,
		       int *dithering,
		       int *num_clrs)
{
  char filename[256];

  argv++;	// Advance to point to first argument
  argc--;	// Decrement argument count
  *dithering = 0;
  filename[0] = infile[0] = outfile[0] = '\0';
  *num_clrs = 256;
  if (! argc) {
    cerr << "Usage: raw2gif [-N] [-d] [-ifilename] [-ofilename] rootname\n";
    cerr << "\t-N specifies quantize to N colors (must be power of 2)\n";
    cerr << "\t-d specifies Floyd-Steinberg dithering\n";
    cerr << "\t-i[filename] specifies input filename (default extension .RAW)\n";
    cerr << "\t-o[filename] specifies output filename (default extension .GIF)\n";
    cerr << "\trootname specifies root name of .RAW input and .GIF output files.\n\n";
  }
  while (argc--) {
    if (argv[0][0] == '-') {
      switch (argv[0][1]) {
	case '0': case '1':
	case '2': case '3':
	case '4': case '5':
	case '6': case '7':
	case '8': case '9':
	  {
	    int tmp = atoi(argv[0]+1);
	    if (tmp < 0 || tmp > 256) {
	      cerr << "Color count out of range: " << argv[0]+1 << "\n";
	      exit(2);
	    }
	    if (tmp != (tmp & -tmp)) {
	      cerr << "Color count must be power of 2 for GIF format.\n";
	      exit(3);
	    }
	    *num_clrs = tmp;
	    break;
	  }
	case 'd':
	case 'D':
	  if (argv[0][2] == '\0') {
	    *dithering = 1;
	    break;
	  }
	case 'i':
	  if (infile[0])
	    cerr << "Warning: input file " << argv[0]+2 << " subsumes prior input file " << infile << ".\n";
	  strcpy(infile, argv[0]+2);
	  if (! strchr(infile, '.'))
	    strcat(infile, ".RAW");
	  break;
	case 'o':
	  if (outfile[0])
	    cerr << "Warning: output file " << argv[0]+2 << " subsumes prior input file " << outfile << ".\n";
	  strcpy(outfile, argv[0]+2);
	  if (! strchr(outfile, '.'))
	    strcat(outfile, ".GIF");
	  break;
	default:
	  cerr << "Bad command-line option " << argv[0]+1 << '\n';
	  exit(2);
      }
    }
    else {
      if (filename[0])
	cerr << "Warning: filename " << argv[0] << " subsumes prior filename " << filename << ".\n";
      strcpy(filename, argv[0]);
    }
    argv++;
  }

  if (! infile[0]) {
    if (! filename[0]) {
      cerr << "Did not specify input file\n";
      exit(2);
    }
    strcpy(infile, filename);
    if (! strchr(filename, '.'))
      strcat(infile, ".raw");
  }
  if (! outfile[0]) {
    if (! filename[0]) {
      cerr << "Did not specify output file\n";
      exit(2);
    }
    strcpy(outfile, filename);
    if (! strchr(filename, '.'))
      strcat(outfile, ".gif");
  }

}

int
main(int argc, char *argv[])
{
  unsigned char cmd;
  int i, j, k;
  int inx;
  char infile[256], outfile[256];
  int blackwhite = 0;
  Histogram *hist = new Histogram(32);
  ColorBoxNode *sc;
  long before, after;
  FILE *out;
  int num_clrs, dither;
//time_t beg = time(NULL);

#ifdef TURBO
  cout << "RAW2GIF-Turbo: Convert .RAW files to GIF format\n";
  cout << "Copyright (C) 1992 by Nicholas Wilt.\n";
  cout << "GIF encoder portion copyright (C) 1989 by Jef Poskanzer.\n\n";
#else
  cout << "RAW2GIF: Convert .RAW files to GIF format\n";
  cout << "Copyright (C) 1992 by Nicholas Wilt.\n";
  cout << "GIF encoder portion copyright (C) 1989 by Jef Poskanzer.\n\n";
  cout << "Shareware: Register for a faster version using ORDER.DOC.\n\n";
#endif
  infile[0] = outfile[0] = '\0';
  parse_commandline(argc, argv, infile, outfile, &dither, &num_clrs);
  Image image(infile, dither);

  cerr << "Histogramming...\n";
  image.HistImage(*hist);
  cerr << "Quantizing...\n";
  boxes = Quantize(*hist, num_clrs);
  cerr << "Finding representatives...\n";
  FindWhichBox(*hist);
  image.ReadImage(findreps);
  delete hist;
  locsearch = FindAverages();
  cerr << "Writing GIF file...\n";
  image.GearForGetPixel(locsearch, Color(boxes->box->inx,
					 boxes->box->repr,
					 boxes->box->repg,
					 boxes->box->repb));
  out = fopen(outfile, "wb");
  image.MapToGIF(out, num_clrs);
  image.StopGetPixel();
//cout << "Time elapsed: " << time(NULL) - beg << " seconds\n";
  return 0;
}
