/****************************************/
/* ROOMMAZ2.C                           */
/* modified from ROOMMAZE.C             */
/* (C) Copyright 1993 David Bollinger   */
/* send comments to CIS ID# 72510,3623  */
/* compiled with Borland C++ 3.0        */
/* command line: bcc -ms -v- roommaz2.c */
/****************************************/

#undef  DEBUG              // define DEBUG to monitor maze creation
#define DEBUG_DELAY 100    // how much delay to see debug steps

#include <bios.h>
#ifdef DEBUG
#include <dos.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include "roommaze.h"

#define AND        &&
#define OR         ||
#define NOT        !
#define SCREENX    320     // screen x dimension
#define SCREENY    200     // screen y dimension
#define MAZEX      60      // maze x dimension
#define MAZEY      60      // maze y dimension
                           //   reduced maze size only to fit more on screen
#define GRIDX      3       // how coarse are the x coordinates ( > 1 )
#define GRIDY      3       // how coarse are the y coordinates ( > 1 )
                           //   basically defines minimum width and height
                           //   also helps align room boundaries
                           //   (reduced from 4 due to smaller maze size)
#define MAXCANDI   256     // how many wall segments to search for door
#define NUMLEVELS  12      // how many distinct maze levels to create
#define FILENAME   "maze.dat" // where to write maze data to disk

// maze objects, defined by the following weighting map:
// if direction=0:                  if direction=1:            (etc)...
// (default weighting)
//
//              front                               right
//              +---+                               +---+
//              | 1 |                               | 8 |
//          +---+---+---+                       +---+---+---+
//    left  | 2 |X,Y| 8 |  right         front  | 1 |X,Y| 4 |  behind
//          +---+---+---+                       +---+---+---+
//              | 4 |                               | 2 |
//              +---+                               +---+
//              behind                              left
//
// Note that the weighting map is now direction sensitive.
// This weighting map allows trace walls to travel along a wall and
// calculate a relative wall index to identify turns correctly.
// This weighting map is also conceptually more pleasing than the original
// since a left/right turn simply requires adding/subtracting 1 from
// the current direction.
// Absolute directions:  0=up,     1=left,  2=down,    3=right
// Relative directions: +0=front, +1=left, +2=behind, +3=right

#define FL         0       // unused, empty floor
#define NE         1       // north end
#define WE         2       // west end
#define LR         3       // lower right
#define SE         4       // south end
#define VE         5       // vertical wall
#define UR         6       // upper right
#define ET         7       // east T
#define EE         8       // east end
#define LL         9       // lower left
#define HO         10      // horizontal wall
#define ST         11      // south T
#define UL         12      // upper left
#define WT         13      // west T
#define NT         14      // north T
#define CC         15      // central cross

#define HD         16      // horizontal door
#define VD         17      // vertical door
#define US         18      // up stairs
#define DS         19      // down stairs

#define F1         20      // floor 1
#define F2         21      // floor 2
#define F3         22      // floor 3
#define F4         23      // floor 4
#define F5         24      // floor 5
#define F6         25      // floor 6
#define F7         26      // floor 7
#define F8         27      // floor 8
#define F9         28      // floor 9
#define Fa         29      // floor 10
#define Fb         30      // floor 11
#define Fc         31      // floor 12
#define Fd         32      // floor 13
#define Fe         33      // floor 14
#define Ff         34      // floor 15

#define SIMPLE_FLOOR   0   // color used by SimplyfyMaze for floors
#define SIMPLE_WALL   15   // color used by SimplifyMaze for walls
#define SIMPLE_DOOR    8   // color used by SimplifyMaze for doors
#define SIMPLE_US     13   // color used by SimplifyMaze for up stairs
#define SIMPLE_DS      5   // color used by SimplifyMaze for down stairs

// location types, as used by PlaceObject
#define LOC_ANY     0      // place object in any unoccupied space
#define LOC_OPEN    1      // place anywhere except next to a wall
#define LOC_WALL    2      // place against a wall
#define LOC_CORNER  3      // place against 2 walls in a corner

/********************/
/* global variables */
/********************/
char maze[NUMLEVELS][MAZEY][MAZEX]; // made array 3 dimensional to store all
                                    //   levels so that stair positions could
                                    //   be calculated intelligently

struct CANDIDATES
   {
   int x;
   int y;
   char here;
   } candidates[MAXCANDI]; // for storing possible locations for doors

int dx[4] = { 0,-1, 0, 1}; // x offsets for a direction
int dy[4] = {-1, 0, 1, 0}; // y offsets for a direction
int candicount;            // current index into candidate array

/***********************/
/* function prototypes */
/***********************/
// maze making stuff
void MakeRooms(int level);
void ClarifyWalls(int level);
void MakeDoors(int level);
void TraceWallForDoor(int level, int sx, int sy);
int  SaveMaze(int level);
void SimplifyMaze(int level);
int  CalculateWallIndex(int level, int x, int y, int dir);
int  IsThisFloor(char object);
int  IsThisWall(char object);
void DisplayMaze(int level);

// utility stuff
void SetVideoMode(int mode);
void ClearScreen(char c);
void Rectangle(int level, int x1, int y1, int x2, int y2, char c);
void WriteMaze(int level, int x, int y, char c);
void PutPixel(int x, int y, char c);
char ReadMaze(int level, int x, int y);
void CheckOrder(int *a, int *b);

/***************************************************************************/
main(int argc, char *argv[])
   {
   int level;

   randomize();
   SetVideoMode(19);       // 320x200 256 color VGA graphics mode
   ClearScreen(1);         // easier on the eyes with a background color
   unlink(FILENAME);       // remove any old maze data file

   while(bioskey(1))       // chew up any keys waiting in buffer
      bioskey(0);

   // make a multi-level dungeon
   for (level=0; level<NUMLEVELS; level++)
      {
      MakeRooms(level);    // make some walled in rooms with rectangles
      ClarifyWalls(level); // calculate exact wall type for each wall
      MakeDoors(level);    // find walls and insert doors
      if (argc > 1)        // any command line arguments will cause us to
         SaveMaze(level);  //    save maze array to disk here
      SimplifyMaze(level); // convert maze to pretty colors
      DisplayMaze(level);  // draw it on VGA

#ifdef DEBUG
      if (bioskey(1))
         break;
#endif
      }

   bioskey(0);             // wait for a key
   SetVideoMode(3);        // return to 80x25 text mode
   return 0;
   }

/***************************************************************************/
/* MakeRooms creates a number of random outlined rectangles.  As   */
/* MAXFLOOR is increased it is more likely that the current room   */
/* color will not match any underlying room colors, thus there     */
/* will be more isolated "island" rooms of simpler shape.  As      */
/* MAXFLOOR is decreased the opposite is true, resulting in fewer  */
/* distinct rooms but each room being more complex in shape.       */
/* Note that while MAXFLOOR=1 will accomplish nothing, other       */
/* small values (including 2) can still generate interesting mazes */
/*******************************************************************/
void MakeRooms(int level)
   {
   // Room size distribution weights - prefer the smaller rooms
   // Note: the maximum weight value times the grid size should not exceed
   //       the dimension of the maze or it will frequently be clipped
   static int xweight[] = {1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,5,5,5,6,6,6,7,7,8,8,9,9,10,11,12,13,14,15,16,17,18,19,20};
   static int yweight[] = {1,1,1,1,2,2,2,2,3,3,3,3,4,4,4,5,5,5,6,6,6,7,7,8,8,9,9,10,11,12,13,14,15,16,17,18,19,20};
   int numxwt, numywt;
   int r, x1, y1, x2, y2, c;
   int numrect;            // how many rectangles to draw rooms with
   int maxfloor;           // how many different colors of floor (2-15)
                           //   (numrect and maxfloor used to be constants)

   // calculate how many weight values are available
   numxwt = (sizeof(xweight)/sizeof(int));
   numywt = (sizeof(yweight)/sizeof(int));

   // make each level based on different params (modified bell dist)
   // could have made these input parameters but totally random is OK
   maxfloor = random(4)+random(4)+random(4)+3;
   numrect  = random(4)+random(8)+random(16)+level+3;

   // first, create a "base" floor over the entire maze
   Rectangle(level, 0, 0, MAZEX-1, MAZEY-1, F1);

   for (r=numrect; r>0; r--)
      {
      x2 = x1 = random((int)((MAZEX-GRIDX)/GRIDX)) * GRIDX;
      y2 = y1 = random((int)((MAZEY-GRIDY)/GRIDY)) * GRIDY;

      // prevent width=0 or height=0
      while(x2==x1)
         x2 = x1 + xweight[random(numxwt)] * GRIDX;
      while(y2==y1)
         y2 = y1 + yweight[random(numywt)] * GRIDY;

      // prevent out-of-range coords, also prevent double walls along exterior
      if (x2 >= MAZEX-2)
         x2 = MAZEX-1;
      if (y2 >= MAZEY-2)
         y2 = MAZEY-1;

      c = random(maxfloor)+F1;
      Rectangle(level, x1, y1, x2, y2, c);

#ifdef DEBUG
         DisplayMaze();
         delay(DEBUG_DELAY);
#endif
      }
   }

/***************************************************************************/
/* ClarifyWalls identifies the exact wall type of each WALL object */
/* by examining its neighbors and updates maze.                    */
/* This routine always uses a direction of 0 (default) when        */
/* calculating the wall index (similar to original method)         */
/*******************************************************************/
void ClarifyWalls(int level)
   {
   register int x, y;

   for (y=0; y<MAZEY; y++)
      for (x=0; x<MAZEX; x++)
         if (IsThisWall( ReadMaze(level, x, y) ) )   // skip floor objects
            WriteMaze(level, x, y, CalculateWallIndex(level, x, y, 0) );

#ifdef DEBUG
   DisplayMaze();
   delay(DEBUG_DELAY);
#endif
   }

/***************************************************************************/
/* MakeDoors scans for vertical and horizontal walls and inserts         */
/* a door at a random position along the length or width of the wall.    */
/*************************************************************************/
void MakeDoors(int level)
   {
   register int x, y, here;

   for (y=1; y<MAZEY-1; y++)
      for (x=1; x<MAZEX-1; x++)
         {
         here = ReadMaze(level, x, y);
         if ((here == HO) OR (here == VE))  // found a horz or vert wall
            TraceWallForDoor(level, x, y);         //   try to make a door
         }
   }

/***************************************************************************/
/* TraceWallForDoor scans trying to identify a wall segment      */
/* if it finds one, it inserts a door randomly along its length. */
/* A wall segment is any length of wall, including corners that  */
/* is not interrupted by any more complex junctions, such as T's */
/* or 4-way intersections.  This is one step better than the     */
/* original method, but it can still generate too many doors.    */
/* Note "island" & "closet" rooms now only get a single door.    */
/*****************************************************************/
void TraceWallForDoor(int level, int sx, int sy)
   {
   int done;         // flag to break out of while loop
   int m;            // the current method of tracer movement
                     //   0 = trace left, 1 = trace right
   int d;            // the current direction of tracer movement
   int nx, ny;       // new (working) coordinates
   int index;        // the wall index for current position and direction
   char here;        // working storage for current maze object
   struct CANDIDATES *cptr; // pointer into candidate structure array for speed

   // reset candidate count
   cptr = &candidates[ candicount = 0 ];

   // save the starting position as a valid candidate for a door
   cptr->x = sx;
   cptr->y = sy;
   cptr->here = ReadMaze(level, sx, sy);
   candicount++;
   cptr++;

   for (m=0; m<=1; m++)   // first trace left, then trace right
      {
      nx = sx;            // start at initial coordinates
      ny = sy;
      here = ReadMaze(level, sx, sy);
      d = (here==HO) ? m*2+1 : m*2; // start LR or FB based on wall and method
      done = 0;
      while ((! done) AND (candicount<MAXCANDI)) // do until something breaks us free
         {
         // look at surrounding walls to determine new trace direction
         index = CalculateWallIndex(level, nx, ny, d);

         // figure out new direction and location
         switch(index)
            {
            case UR : // a left turn corner, turn left
                      d = (d+1)%4;
                      break;
            case UL : // a right turn corner, turn right
                      d = (d+3)%4;
                      break;
            case VE : // straight wall, no turn
                      break;
            default : // any other is an end of segment
                      done = 1;
                      continue;
            }

         // calculate new postion
         nx = nx + dx[d];
         ny = ny + dy[d];

         // back at starting location - island room
         if ((nx==sx) AND (ny==sy))
            {
            done = 1;
            continue;
            }

         // find out what's in the new location
         here = ReadMaze(level, nx, ny);

         if ((here==VD) OR (here==HD)) // already a door on this wall
            return;

         if ((here==VE) OR (here==HO)) // found a good section of wall
            {
            cptr->x = nx;
            cptr->y = ny;
            cptr->here = here;
            candicount++;
            cptr++;
            }

         }  // while
      }  // for

   d = random(candicount);    // pick a candidate at random for door
   cptr = &candidates[d];
   WriteMaze(level, cptr->x, cptr->y, (cptr->here==HO) ? HD : VD );

#ifdef DEBUG
   DisplayMaze();
   delay(DEBUG_DELAY);
#endif
   }

/***************************************************************************/
int SaveMaze(int level)
   {
   static char mazechars[] = " ͼĳ<>                    ";
   register int x, y;
   FILE *file;

   if ((file=fopen(FILENAME,"at")) == NULL)
      return 1;
   for (y=0; y<MAZEY; y++)
      {
      for (x=0; x<MAZEX; x++)
         fputc(mazechars[ReadMaze(level, x, y)], file);
      fputc(10, file);
      }
   fputc(10, file);
   fclose(file);
   return 0;
   }

/***************************************************************************/
/* CalculateWallIndex was broken out to a separate function so that */
/* it is more generally available, and easier to change if the      */
/* index calculation needs to change based on the numbering scheme  */
/* used for maze objects.  See above for explanation of index.      */
/********************************************************************/
int CalculateWallIndex(int level, int x, int y, int dir)
   {
   int index;
   int f, l, r, b;   // neighbor directions for front, left, right, behind

   // calculate neighbor directions
   f = dir;        // redundant, but easier to understand
   l = (dir+1)%4;
   b = (dir+2)%4;
   r = (dir+3)%4;

   // check if neighboring squares are numerically less than
   // floor type 1, indicating some type of wall, if a wall
   // then it affects the index value based on its location
   index  = ( IsThisWall(ReadMaze(level, x+dx[f], y+dy[f])) ) ? 1 : 0;
   index += ( IsThisWall(ReadMaze(level, x+dx[l], y+dy[l])) ) ? 2 : 0;
   index += ( IsThisWall(ReadMaze(level, x+dx[b], y+dy[b])) ) ? 4 : 0;
   index += ( IsThisWall(ReadMaze(level, x+dx[r], y+dy[r])) ) ? 8 : 0;

   return index;
   }

/***************************************************************************/
/* IsThisFloor is a classification function also designed to ease     */
/* the dependance on a particular numbering scheme for maze objects.  */
/**********************************************************************/
int IsThisFloor(char object)
   {
   if ((object >= F1) AND (object <= Ff))
      return 1;
   return 0;
   }

/***************************************************************************/
/* IsThisWall is a classification function also designed to ease      */
/* the dependance on a particular numbering scheme for maze objects.  */
/**********************************************************************/
int IsThisWall(char object)
   {
   if ((object >= NE) AND (object <= CC))
      return 1;
   // for this purpose doors are also considered walls (solid objects)
   if ((object == VD) OR (object == HD))
      return 1;
   return 0;
   }

/***************************************************************************/
/* SimplifyMaze un-translates all of the various floor, wall and door   */
/* objects into 3 simple floor, wall and door objects.  Makes it look   */
/* nicer with this simple VGA display, but it may be better to leave it */
/* complex for actual use in a game. (also translates stairs)           */
/************************************************************************/
void SimplifyMaze(int level)
   {
   register int x, y;

   for (y=0; y<MAZEY; y++)
      for (x=0; x<MAZEX; x++)
         switch(ReadMaze(level, x, y))
            {
            case FL :
            case F1 :
            case F2 :
            case F3 :
            case F4 :
            case F5 :
            case F6 :
            case F7 :
            case F8 :
            case F9 :
            case Fa :
            case Fb :
            case Fc :
            case Fd :
            case Fe :
            case Ff : WriteMaze(level, x, y, SIMPLE_FLOOR); break;
            case HD :
            case VD : WriteMaze(level, x, y, SIMPLE_DOOR); break;
            case DS : WriteMaze(level, x, y, SIMPLE_DS); break;
            case US : WriteMaze(level, x, y, SIMPLE_US); break;
            default : WriteMaze(level, x, y, SIMPLE_WALL); break;
            }
   }

/***************************************************************************/
/* DisplayMaze makes a demo display of the maze.                 */
/* This is the only routine that is  dependant on the specific   */
/* values of the maze parameters in  order to work properly.     */
/* Specifically, MAZEX/Y and NUMLEVELS must be such that         */
/* physical screen limits are not exceeded.                      */
/* Also the only routine that bypasses ReadMaze and accesses the */
/* maze array directly (for OOPers trying to encapsulate)        */
/*****************************************************************/
void DisplayMaze(int level)
   {
   register int x, y;
   int screenx, screeny;  // for calculating screen position of each level
   char *mptr = &maze[level][0][0];

   screenx = (level%4)*(MAZEX+6) + 24;
   screeny = (level/4)*(MAZEY+6) + 4;
   for (y=0; y<MAZEY; y++)
      {
      for (x=0; x<MAZEX; x++)
         PutPixel(screenx+x, screeny, *mptr++);
      screeny++;
      }
   }

/***************************************************************************/
/* BIOS call to set video mode */
/*******************************/
void SetVideoMode(int mode)
   {
   asm   xor   ah, ah
   asm   mov   al, byte ptr mode
   asm   int   10h
   }

/***************************************************************************/
/* ClearScreen destroys AX,CX,ES */
/*********************************/
void ClearScreen(char color)
   {
   asm   push  di
   asm   mov   ax, 0xa000
   asm   mov   es, ax
   asm   xor   di, di
   asm   mov   al, byte ptr color
   asm   mov   ah, al
   asm   mov   cx, 0x7d00
   asm   cld
   asm   rep   stosw               // 286
   asm   pop   di
   }

/***************************************************************************/
void Rectangle(int level, int x1, int y1, int x2, int y2, char c)
   {
   int x, y;

   /********************************************/
   /* verify that x2>=x1 and y2>=y1            */
   /* not really needed as the code sits, but  */
   /* best to be safe in case code is modified */
   /********************************************/
   CheckOrder(&x1, &x2);
   CheckOrder(&y1, &y2);

   /*******************************************************/
   /* draw interior of rectangle with desired floor color */
   /*******************************************************/
   for (y=y1+1; y<y2; y++)
      for (x=x1+1; x<x2; x++)
         WriteMaze(level, x, y, c);

   /*********************************************************/
   /* now draw smart border - only draw border if existing  */
   /* color is not the same as requested floor color - this */
   /* allows the rectangles to overlap and create more      */
   /* interesting shapes than simple rectangles             */
   /*********************************************************/
   for (x=x1; x<=x2; x++)
      {
      if (ReadMaze(level, x, y1) != c)
         WriteMaze(level, x, y1, CC);
      if (ReadMaze(level, x, y2) != c)
         WriteMaze(level, x, y2, CC);
      }

   for (y=y1; y<=y2; y++)
      {
      if (ReadMaze(level, x1, y) != c)
         WriteMaze(level, x1, y, CC);
      if (ReadMaze(level, x2, y) != c)
         WriteMaze(level, x2, y, CC);
      }
   }

/***************************************************************************/
void WriteMaze(int level, int x, int y, char c)
   {
   if ((x>=0) AND (x<MAZEX) AND (y>=0) AND (y<MAZEY))
      maze[level][y][x] = c;
   // could return an error if needed
   }

/***************************************************************************/
/* PutPixel destroys AX,BX,ES */
/******************************/
void PutPixel(int x, int y, char c)
   {
   asm   mov   ax, 0xa000
   asm   mov   es, ax
   asm   mov   bx, word ptr y
   asm   shl   bx, 1
   asm   mov   bx, word ptr ytable[bx]
   asm   add   bx, word ptr x
   asm   mov   al, byte ptr c
   asm   mov   byte ptr es:[bx], al
   }

/***************************************************************************/
char ReadMaze(int level, int x, int y)
   {
   if ((x>=0) AND (x<MAZEX) AND (y>=0) AND (y<MAZEY))
      return maze[level][y][x];
   else
      return F1;   // return floor if out of range
                   // so that ClarifyWalls works properly
   }

/***************************************************************************/
void CheckOrder(int *a, int *b)
   {
   int temp;

   if (*a > *b)
      {
      temp = *a;
      *a = *b;
      *b = temp;
      }
   }

/***************************************************************************/
/* end of roommaz2.c */

