/*** FILE "mapgif.h" *********************************************** 130793 ***/
#ifndef _mapgif_h_
#define _mapgif_h_

#include "mapin.h"
#include "fnd/sequence.h"
#include <fstream.h>

/*** Class hieracy ************************************************************/
/*

MapGIFBase -*-> MapGIF
            |
            *-> MapPCGIF
            |
            *-> MapSCGIF

*/
/*** Copyright notice *********************************************************/
/*
       The Graphics Interchange Format(c) is the Copyright property of
       CompuServe Incorporated. GIF(sm) is a Service Mark property of
       CompuServe Incorporated.
*/
/*** MapGIFBase ***************************************************************/

class MapGIFBase: public MapInBase
{
public:
  MapGIFBase(const char* const filename);  //Opens image-file, initializes
                                           //'Comments'.

protected:
  void GetImage(n1 infoOnly,
                pix x1, pix y1, pix x2, pix y2, MapInBase::Gravity g,
                spix CustomX, spix CustomY);
  //This procedure is the one, that actually decodes the image. The reason this
  //isn't done directly by the constructor is, that the constructor isn't
  //allowed to call pure virtual procedures.
  //All arguments come directly from the main constructor, where 'infoOnly'
  //indicates, if only the size of the image, -but not the image itself-, should
  //be read.
/* Note: "The effect of calling a pure virtual function directly or indirectly
 *        for the object being constructed from a constructor, except using
 *        explicit qualification, is undefined" - The C++ Programming Language
 *       (Bjarne Stroustrup), page 584.
 *       HOW can the dudes, which have thought this out, believe, I won't say
 *       that's some nice CRAP, when they don't state a REASON, too?
 *       Since it puts some though restrictions on the way, one's allowed to
 *       code, I'll suggest, they TRY to figure out some kind of ARGUMENTATION,
 *       that can be put in there, too! Maybe, then, it wouldn't make me so damn
 *       pissed!
 */

public:
  pix SizeX() const;
  pix SizeY() const;

  double GetAspectRatio() const;  //If multiple images exist, the ratio returned
                                  //will be the one contained within the last
                                  //image.

  n16 Colors() const;  //Returns the number of colors in the image.

  const char* GetComments() const;

  const char* Version() const;  //Returns the version number, which can be
                                //"87a" or "89a". The string ends with a '\0'.

private:
  void ReadColorTable(n16 NumColors);
  //Reads a color-table with NumColors in it.

  virtual void ColorTableInit(n16 Colors)=0;
  //Tells a subclass, that a new color-table is about to be read.

  virtual void NextColorTableEntry(n16 red, n16 green, n16 blue)=0;
  //Tells a subclass how the next color looks like.

  virtual void Pixel(n8 color, pix x, pix y)=0;
  //Tells a subclass to set the pixel (x,y) using the color 'color'.

  ifstream ImageFile;  //Input file containing the image.

protected:
  n8 Get();   //Reads the next byte from the file containing the image.
  n8 Peak();  //Returns the next byte from the file containing the image, but
              //does NOT remove it from the stream.

private:
  void Skip(n16 size);   //Scans across 'size' bytes.
  void SkipSubBlocks();  //Scans across a sequence of sub-blocks, returns after
                         //having eaten a Block Terminator.

/*** BEWARE - BEWARE - BEWARE: ***/
/* Note: The following, -up to and including _CustomY-, should have been LOCAL
 *       to GetImage, but that ain't possible; since functions can't be nested,
 *       data have to be replicated! See the comments below.
 */
  void GIF_Data_Stream();          //This block of operations represent actions
  void Logical_Screen();           //to be performed, when the associated left-
  void Data();                     //hand-side of the GIF-grammer is met.
  void Graphic_Block();
  void Graphic_Rendering_Block();
  void Table_Based_Image();
  void Special_Purpose_Block();

/* Note: Terminology: window=rectangle, where (a part of) the image is supposed
 *                           to appear. No part of the image will appear outside
 *                           the window.
 *                    active area=rectangle containing the image. The part, that
 *                                overlaps with the window, is the part, that
 *                                will appear in the map.
 */

  spix x0_activeArea;  //The minimum x-...
  spix y0_activeArea;  //...and the minimum y-coordinate of the active area
                       //relative to the used map. They may be negative.

  n1 _infoOnly;             //Indicates, if only the size of the image is
                            //wanted.
  pix _x1, _y1, _x2, _y2;   //The window, where the image will appear.
  MapInBase::Gravity _g;    //The gravity to use on the active area inside the
                            //window.
  spix _CustomX, _CustomY;  //If the gravity specified is of the "custom" type,
                            //these coordinates are used to specify the position
                            //of x0_activaArea and y0_activeArea relative to
                            //_x1 and _y1.

  pix sizeX;  //Dimensions of the active area containing the image.
  pix sizeY;

  double aspectRatio;       //The pixel aspect ratio factor.
  n16 colors;               //Number of colors.
  Sequence<char> Comments;  //All comments found in the image file.
  char version[4];          //Version number of GIF image (including '\0').

/*** BEWARE - BEWARE - BEWARE: ***/
/* Note: All the following variables and functions should have been LOCAL to
 *       the function _Table_Based_Image_, in a new scope beginning just after
 *       the comment "* Image Data: *". This was not possible, since C++ doesn't
 *       allow nested functions! I considered declaring the variables local, and
 *       do such a sick thing as use macros (define's) instead of real
 *       functions, but then I couldn't use recursive functions, like the OutPut
 *       function (I tried to simulate the recursion, but it just made the code
 *       run slower, and added too much complexity, so I trashed the idea).
 *       So, the variables exist after an image has been read, which is not
 *       good, but there aren't really any alternative, that doesn't cripple the
 *       code in some way; I'll just live with the mess, and hope, the ANSI C++
 *       committee becomes less smallminded, someday.
 *
 *       The fuckheads, which continue to state, there ain't any need for nested
 *       functions, can go to hell!
 */

  n32 bitsLeft;    //Bits left from the previous byte processed; it's the
                   //'numBitsLeft' least significant bits, that are the valid
                   //ones.
  n8 numBitsLeft;  //Number of bits left from the previous byte processed.

  /* The string-table: */
  struct
  { n8 firstByte;    //The first byte (actually the first byte of the string
                     //pointed to by the 'firstBytes' code).
    n16 firstBytes;  //The code for the sequence of bytes, EXCEPT the last.
                     //'firstBytes' is meaningless for entries less than CC,
                     //since these sequences only consists of a single byte!
    n8 lastByte;     //The last byte.
  } Table[4096];
  /* Note: Each possible code corresponds to a single entry in the string-table.
   *       Each entry descripes a sequence a bytes. For speed, the first byte of
   *       each sequence is immediately available. The hole sequence, -except
   *       the very last byte-, can always be descriped by another entry in the
   *       table; 'firstBytes' refer to this entry.
   */

  n8 rootCodeSize;  //Number of bits used by the root codes, that is, the
                    //number of bits used to descripe all possible, different
                    //bytes in the output. This is one less than the initial
                    //value for 'codeSize'.

  n8 codeSize;   //Number of bits used by the current codes.
  n16 codeMask;  //Used to mask off codes. If f.ex. codeSize is 7, then
                 //codeMask will be the binary mask 1111111.

  n16 CC;   //The clear code.
  n16 EOI;  //The end-of-line code.

  n1 EOImissing;  //Indicates, if the EOI code has not yet been seen. If it has been
                  //seen, all eventually following codes should be ignored.

  n16 nextFree;  //The next, not yet used code.
  n1 firstCode;  //Indicates, if the first code has been seen; the first code
                 //is treated differently than the other codes.
  n16 oldCode;   //The previous code, used when processing codes different from
                 //the first code.

  n1 interlaced;     //Indicates, if the image is interlaced.
  n8 interlacePass;  //The pass-number for an interlaced image.

  spix minX;  //Limits for the current image (a GIF stream may contain multiple
  spix minY;  //images) relative to the used map. The limits may be negative,
  spix maxX;  //since they are within the active area.
  spix maxY;

  spix x;  //Current pixel position relative to the map, inside the limits of
  spix y;  //the current image. May be negative.

  void Init();              //Initializes the string table.
  void NextByte(n8 byte);   //Processes the next byte read.
  void NextCode(n16 code);  //Processes the next code.
  void Output(n16 code);    //Outputs the sequence of bytes associated with the
                            //code 'code'.
};

/*** MapGIF *******************************************************************/

class MapGIF: public MapGIFBase
{
public:
  MapGIF(const char* const filename);
  //Reads the size, number of colors, version number and the aspect ratio of the
  //image in the file 'filename', but not the image itself (actually, the aspect
  //ratio returned is the one associated with the last image, if multiple
  //images exist).

  MapGIF(const char* const filename, MapBase& M,
         MapInBase::Gravity g=MapInBase::Center,
         spix CustomX=0, spix CustomY=0);
  //Reads the image in the file 'filename' into the map M, converting colors to
  //Black and White (assumes Black=black and White=white).
  //'g', 'CustomX' and 'CustomY' are used to specify the kind of gravity,
  //-relative to the map-, that should be used.

  MapGIF(const char* const filename, MapBase& M,
         pix x1, pix y1, pix x2, pix y2,
         MapInBase::Gravity g=MapInBase::Center,
         spix CustomX=0, spix CustomY=0);
  //Reads the image in the file 'filename' into the map M, converting colors to
  //Black and White (assumes Black=black and White=white).
  //The arguments 'x1', 'y1', 'x2' and 'y2' specifies a window, in which the
  //image will appear.
  //'g', 'CustomX' and 'CustomY' are used to specify the kind of gravity,
  //-relative to the window in the map-, that should be used.

/* Note: Even though multiple image within a single GIF file can be read, it
 *       can't switch back and forth between color tables; it's always the
 *       last read color table, that will be used!
 */

private:
  MapBase* R;  //Map to store image in (R for receiver).
               //Only used while reading an image.

  n8 index;           //Index into BitColors, used while reading color-tables.
  n1 BitColors[256];  //Each of the 256 possible colors is mapped to a bit-
                      //color - either Black or White.

  void ColorTableInit(n16 Colors);
  void NextColorTableEntry(n16 red, n16 green, n16 blue);
  void Pixel(n8 color, pix x, pix y);
};

/*** MapPCGIF *****************************************************************/

class MapPCGIF: public MapGIFBase
{
public:
  MapPCGIF(const char* const filename);
  //Reads the size, number of colors, version number and the aspect ratio of the
  //image in the file 'filename', but not the image itself (actually, the aspect
  //ratio returned is the one associated with the last image, if multiple
  //images exist).

  MapPCGIF(const char* const filename, MapPCBase& M,
           MapInBase::Gravity g=MapInBase::Center,
           spix CustomX=0, spix CustomY=0,
           n1 OverrideBW=1);
  //Reads the image in the file 'filename' into the map M, allocating colors
  //from M. The reading will fail, if not enough colors can be allocated.
  //'g', 'CustomX' and 'CustomY' are used to specify the kind of gravity,
  //-relative to the map-, that should be used.
  //OverrideBW indicates, if Black and White may be used, that is, assigned a
  //new color-triplet.

  MapPCGIF(const char* const filename, MapPCBase& M,
           pix x1, pix y1, pix x2, pix y2,
           MapInBase::Gravity g=MapInBase::Center,
           spix CustomX=0, spix CustomY=0,
           n1 OverrideBW=1);
  //Reads the image in the file 'filename' into the map M, allocating colors
  //from M. The reading will fail, if not enough colors can be allocated.
  //The arguments 'x1', 'y1', 'x2' and 'y2' specifies a window, in which the
  //image will appear.
  //'g', 'CustomX' and 'CustomY' are used to specify the kind of gravity,
  //-relative to the window in the map-, that should be used.
  //OverrideBW indicates, if Black and White may be used, that is, assigned a
  //new color-triplet.

/* Note: Even though multiple image within a single GIF file can be read, it
 *       can't switch back and forth between color tables; it's always the
 *       last read color table, that will be used!
 */

  virtual ~MapPCGIF();  //Calls DeallocColors.

  void ForgetColors();   //Forgets all allocated colors.
  void DeallocColors();  //Deallocates all allocated colors.

private:
  MapPCBase* R;  //Map to store image in (R for receiver).
                 //Not only used while reading an image, but active until the
                 //allocated colors have been deallocated.

  n1 overrideBW;     //Indicates, if Black and White may be assigned another
                     //triplet, than the one they already have been assigned.
  n1 BnotOverrided;  //Indicates, if Black hasn't been overrided.
  n1 WnotOverrided;  //Indicates, if White hasn't been overrided.
  n1 Bfree;          //Indicates, if Black is free for overriding.
  n1 Wfree;          //Indicates, if White is free for overriding.
  /* Note: For further clarification on this override stuff, see the actual
   *       implementation!
   */

  n1 deallocateColors;            //Indicates, if the allocated palette entries
                                  //should be deallocated, when this class is
                                  //destructed.
  Sequence<n16> AllocatedColors;  //All allocated palette entries.
  n8 index;                       //Index into Colors, used while reading
                                  //color-tables.
  n16 Entries[256];               //Each of the 256 possible colors is mapped
                                  //to an palette entry.
  
  void ColorTableInit(n16 Colors);
  void NextColorTableEntry(n16 red, n16 green, n16 blue);
  void Pixel(n8 color, pix x, pix y);
};

/*** MapSCGIF *****************************************************************/

class MapSCGIF: public MapGIFBase
{
public:
  MapSCGIF(const char* const filename);
  //Reads the size, number of colors, version number and the aspect ratio of the
  //image in the file 'filename', but not the image itself (actually, the aspect
  //ratio returned is the one associated with the last image, if multiple
  //images exist).

  MapSCGIF(const char* const filename, MapSCBase& M,
           MapInBase::Gravity g=MapInBase::Center,
           spix CustomX=0, spix CustomY=0);
  //Reads the image in the file 'filename' into the map M.
  //'g', 'CustomX' and 'CustomY' are used to specify the kind of gravity,
  //-relative to the map-, that should be used.

  MapSCGIF(const char* const filename, MapSCBase& M,
           pix x1, pix y1, pix x2, pix y2,
           MapInBase::Gravity g=MapInBase::Center,
           spix CustomX=0, spix CustomY=0);
  //Reads the image in the file 'filename' into the map M.
  //The arguments 'x1', 'y1', 'x2' and 'y2' specifies a window, in which the
  //image will appear.
  //'g', 'CustomX' and 'CustomY' are used to specify the kind of gravity,
  //-relative to the window in the map-, that should be used.

/* Note: Even though multiple image within a single GIF file can be read, it
 *       can't switch back and forth between color tables; it's always the
 *       last read color table, that will be used!
 */

private:
  MapSCBase* R;  //Map to store image in (R for receiver).
                 //Only used while reading an image.

  n8 index;        //Index into Red/Green/Blue, used while reading color-tables.
  n16 Red[256];    //Each of the 256 possible colors is mapped to a RGB-triplet.
  n16 Green[256];
  n16 Blue[256];

  void ColorTableInit(n16 Colors);
  void NextColorTableEntry(n16 red, n16 green, n16 blue);
  void Pixel(n8 color, pix x, pix y);
};

#endif
/******** "mapgif.h" **********************************************************/

