/* SharedQueue.c -- implementation of shared queues


Class SharedQueue provides a queue that can be used for communication
between two separate processes.  Semaphores are used to block a process
if next() is called and there are no objects in the queue, or if
nextPut() is called and there is no space in the queue.


*/             
#include "nihclstd.h"
#pragma hdrstop

#pragma -vi
#include "SharedQu.h"
#include "nihclIO.h"

#include <assert.h>

#define	THIS	SharedQueue
#define	BASE	Object
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES ArrayOb::desc(),Semaphore::desc(),Semaphore::desc()
#define VIRTUAL_BASE_CLASSES Object::desc()

DEFINE_CLASS(SharedQueue,2,"",NULL);

SharedQueue::SharedQueue(unsigned queueSize)
:
	queue(queueSize),
	valueAvailable(0),
	spaceAvailable(queueSize)
{
	readPosition = writePosition = 0;
}

SharedQueue::SharedQueue(const SharedQueue& q)
:
	queue(q.queue),
	valueAvailable(q.valueAvailable),
	spaceAvailable(q.spaceAvailable)
{
	readPosition = q.readPosition;
	writePosition = q.writePosition;
}

Object* SharedQueue::next()
{
	valueAvailable.wait();
	Object* ob = queue[readPosition];
	queue[readPosition++] = nil;
	if (readPosition >= queue.capacity()) readPosition = 0;
	spaceAvailable.signal();
	return ob;
}

Object* SharedQueue::nextPut(Object& ob)
{
	spaceAvailable.wait();
	queue[writePosition++] = &ob;
	if (writePosition >= queue.capacity()) writePosition = 0;
	valueAvailable.signal();
	return &ob;
}

unsigned SharedQueue::capacity() const	{ return queue.capacity(); }

bool SharedQueue::isEmpty() const	{ return valueAvailable.value() <= 0; }

unsigned SharedQueue::size() const	{ return MAX(valueAvailable.value(),0); }

Object* SharedQueue::copy() const	{ return deepCopy(); }

void SharedQueue::deepenShallowCopy()
{
	queue.deepenShallowCopy();
	valueAvailable.deepenShallowCopy();
	spaceAvailable.deepenShallowCopy();
}

void SharedQueue::dumpOn(ostream& strm) const
{
	strm << className() << "[\n";
	strm << "valueAvailable ";  valueAvailable.dumpOn(strm);
	strm << "spaceAvailable ";  spaceAvailable.dumpOn(strm);
	strm << "queue:\n";
	int i = readPosition;
	for (int n = valueAvailable.value(); n>0; n--) {
		queue[i++]->dumpOn(strm);
		if (i == queue.capacity()) i = 0;
	}
	strm << "]\n";
}

unsigned long SharedQueue::hash() const	{ return (unsigned long)this; }

bool SharedQueue::isEqual(const Object& ob) const   { return isSame(ob); }

void SharedQueue::printOn(ostream& strm) const
{
	int i = readPosition;
	for (int n = valueAvailable.value(); n>0; n--) {
		if (i != readPosition) strm << '\n';
		queue[i++]->printOn(strm);
		if (i == queue.capacity()) i = 0;
	}
}

static unsigned sharedqueue_capacity;

SharedQueue::SharedQueue(OIOin& strm)
	: BASE(strm),
	queue((strm >> sharedqueue_capacity, sharedqueue_capacity)),
	valueAvailable(strm),
	spaceAvailable(strm)
{
	readPosition = 0;
	writePosition = valueAvailable.value();
	for (int i =0; i<writePosition; i++) queue[i] = Object::readFrom(strm);
}

void SharedQueue::storer(OIOout& strm) const
{
	BASE::storer(strm);
	strm << queue.capacity();
	valueAvailable.storeMemberOn(strm);
	spaceAvailable.storeMemberOn(strm);
	int i = readPosition;
	while (i != writePosition) {
		queue[i++]->storeOn(strm);
		if (i == queue.capacity()) i = 0;
	}
}

SharedQueue::SharedQueue(OIOifd& fd)
	: BASE(fd),
	queue((fd >> sharedqueue_capacity, sharedqueue_capacity)),
	valueAvailable(fd),
	spaceAvailable(fd)
{
	readPosition = 0;
	writePosition = valueAvailable.value();
	for (int i=0; i < writePosition; i++)
	  queue[i] = Object::readFrom(fd);
}

void SharedQueue::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	fd << queue.capacity();
	valueAvailable.storeMemberOn(fd);
	spaceAvailable.storeMemberOn(fd);
	int i = readPosition;
	while (i != writePosition) {
	  queue[i++]->storeOn(fd);
	  if (i == queue.capacity()) i = 0;
	  };
}

int SharedQueue::compare(const Object&) const
{
	shouldNotImplement("compare");
	return 0;
}
