/* MS-DOS version of NIH class library - Michael F. Murphy 4/93 */
/* Unix source: Set.c,v 3.18 92/07/26 14:33:38  */

/*
Function:
	
A Set is an unordered collection of objects in which no object is
duplicated.  Duplicate objects are defined by the function isEqual.
Sets are implemented using a hash table.  The capacity() function
returns the 1/2 the capacity of the hash table and the size() function
returns the number of objects currently in the Set.  For efficiency, the
capacity is always a power of two and is doubled whenever the table
becomes half full.
*/                                                 

#include "nihclstd.h"
#pragma hdrstop
 
#include "Set.h"
#include "OrderedC.h"
#include "nihclIO.h"

#define	THIS	Set
#define	BASE	Collection
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES ArrayOb::desc()
#define VIRTUAL_BASE_CLASSES

DEFINE_CLASS(Set,1,"$"__FILE__" "__DATE__" "__TIME__"$",NULL)

extern const int NIHCL_ALLOCSIZE,NIHCL_REMOVEERR;


#if defined(MIBADDEFAULTFCNS)
Set::Set(const Set& copyfrom)
 : count(copyfrom.count), nbits(copyfrom.nbits), mask(copyfrom.mask),
	contents(copyfrom.contents)
{
}

Set& Set::operator=(const Set& rhs)
{
	if(this == &rhs)
		return *this;

	BASE::operator=(rhs);

	count = rhs.count;
	nbits = rhs.nbits;
	mask = rhs.mask;
	contents = rhs.contents;

	return *this;
}
#endif



unsigned Set::setCapacity(unsigned int size)
/*
	Establish the Set capacity.  Round size up to the next highest
	power of two, if necessary.
*/
{

	if (size==0) setError(NIHCL_ALLOCSIZE,DEFAULT,this,className());
	if (size==1) size = 2; // to avoid problem with reSize threshhold
	count = 0;
	nbits = 0;
	for (unsigned s=size; s!=0; s>>=1) nbits++;
	if ((size&(size-1)) != 0) nbits++;	// round up if not power of 2
	size = 1<<nbits;
	mask = size-1;
	return size;		// return hash table size
}

//Set::Set(unsigned size) : contents(setCapacity(size)) {}
Set::Set(unsigned size)
{
	 int temp = setCapacity(size);
	 contents.reSize(temp);
}

int Set::h(unsigned long K) const
/*
multiplicative hash function

Enter:
	K = key to be hashed
	
Returns:
	hash table index
	
Knuth Vol. 3, Section 6.4, pp. 508-512
*/
{
	const unsigned long Aw = 2654435769UL;
//	const unsigned long Aw = 40503;		use for 16 bit machines?
	// MFM cast to quiet compiler
	return (int)(((Aw*K)>>((8*sizeof(unsigned))-nbits)) & mask);
}

int Set::findIndexOf(const Object& ob) const
/*
Search this set for the specified object

Enter:
	ob = pointer to object to search for

Returns:
	index of object if found or of nil slot if not found
	
Algorithm L, Knuth Vol. 3, p. 519
*/
{
	int i;
	for (i = h(ob.hash()); contents[i]!=nil; i = (i-1)&mask) {
		if (contents[i]->isEqual(ob)) return i;
	}
	return i;
}

unsigned long Set::hashOf(const Object& ob) const
/*
Apply hash function to an object in this Set.  Classes IdentDict and
IdentSet override this to return object address instead of hash().
*/
{
	return ob.hash();
}

void Set::reSize(unsigned newSize)
/*
	Change the capacity of this Set to newSize.
*/
{
	if (newSize <= count) return;
	if (count > 0) {
		OrderedCltn oldcontents(count);
		unsigned n = contents.size();
		for (unsigned int i=0; i<n; i++) {         	// unsigned MFM
			Object* ob = contents[i];
			if (ob != nil) {
				oldcontents.add(*ob);
				contents.elem(i) = nil;
			}
		}
		contents.reSize(setCapacity(newSize));
//		addAll(oldcontents);
// Use non-virtual addToSet to avoid side effects in derived classes
                DO(oldcontents,Object,o) this->addToSet(*o); OD              
	}
	else contents.reSize(setCapacity(newSize));
}

/* virtual */
Object* Set::add(Object& ob)
{
    return addToSet(ob);
}

/* non-virtual */
Object* Set::addToSet(Object& ob)
/*
	Add an object to this Set, making the Set larger if it
	becomes half full.
*/
{
	int i = findIndexOf(ob);
	if (contents[i]==nil) {		// add new object to set 
		contents[i] = &ob;
		if (++count > capacity())
			reSize(capacity()*EXPANSION_FACTOR);
		return &ob;		// successful add
	}
	else {
		return contents[i];	// object already in set
	}
}

#pragma warning(disable: 4127) // conditional expression is constant
Object* Set::remove(const Object& ob)
/*
remove object from set

Enter:
	ob = reference to object to be removed

Returns:
	pointer to removed object

Algorithm R, Knuth Vol. 3 p. 527
*/
{
	int i = findIndexOf(ob);
	Object* rob = contents[i];
	if (rob==nil) {
				setError(NIHCL_REMOVEERR,DEFAULT,this,className(),ob.className(),&ob);
				/* not reached */
				return NULL;
	} else {
		int j,r;
		while (YES) {
			contents[j=i] = nil;
			do {
				i = (i-1)&mask;
				if (contents[i]==nil) {
					count--;
					return rob;
				}
				r = h(hashOf(*contents[i]));
			} while ((i<=r&&r<j) || (r<j&&j<i) || (j<i&&i<=r));
			contents[j] = contents[i];
		}
	}
}
#pragma warning(default: 4127)

void Set::removeAll()
{
	if (count != 0) {
		contents.removeAll();
		count = 0;
	}
}

bool Set::operator==(const Set& s) const
/*
	Return YES if the specified Set equals this Set.
*/
{
	if (count!=s.count) return NO;
	unsigned n = contents.capacity();
	for (unsigned int i=0; i<n; i++) { 					// unsigned MFM
		if (contents[i]!=nil && !s.includes(*contents[i])) return NO;
	}
	return YES;
}

Set Set::operator-(const Set& s) const
/*
	Returns a Set of all of the objects that are contained in this
	Set but not in the specified Set.
*/
{
	Set diff = *this;
	unsigned n = contents.capacity();
	for (unsigned int i=0; i<n; i++) {              // unsigned MFM
		if (contents[i]!=nil && s.includes(*contents[i])) diff.remove(*contents[i]);
	}
	return diff;
}

Set Set::operator&(const Set& s) const
/*
	Returns a Set of all objects that are in both this Set and
	the specified Set.
*/
{
	Set intersection = *this;
	unsigned n = contents.capacity();
	for (unsigned int i=0; i<n; i++) {       		// unsigned MFM
		if (contents[i]!=nil && !s.includes(*contents[i])) intersection.remove(*contents[i]);
	}
	return intersection;
}

Set Set::operator|(const Set& s) const
/*
	Returns a Set of all objects that are in either this Set
	or the specified Set.
*/
{
	Set u = *this;
	u.addAll(s);
	return u;
}

Object*& Set::at(int i)				{ return contents[i]; }

const Object *const& Set::at(int i) const
{  
#ifdef REFTOTEMP 
	return (const Object *const&)contents[i];
#else 
   return contents[i];
#endif	
}

bool Set::isEqual(const Object& p) const
/*
	Returns YES if this Set equals the specified object.
*/
{
	return p.isSpecies(classDesc) && *this==castdown(p);
}

const Class* Set::species() const { return &classDesc; }

void Set::deepenShallowCopy()
{
	BASE::deepenShallowCopy();
	contents.deepenShallowCopy();
}

Object* Set::doNext(Iterator& pos) const
{
	Object* ob;
	unsigned n = contents.capacity();
	while (pos.index < n) {
		if ((ob = (Object*)contents[pos.index++]) != nil) {
			return ob;
		}
	}
	return NULL;
}

unsigned Set::capacity() const  { return contents.capacity()>>1; }

unsigned long Set::hash() const
{
	unsigned long h = 0L;
	DO(*this,Object,o) h ^= o->hash(); OD
	return h;
}

unsigned Set::occurrencesOf(const Object& ob) const
/*
	Return the number of occurences of thw specified object
	in this Set (either 0 or 1).
*/
{
	if (contents[findIndexOf(ob)]!=nil) return 1;
	else return 0;
}

Object* Set::findObjectWithKey(const Object& ob) const
{
	return (Object*)contents[findIndexOf(ob)];
}

unsigned Set::size() const		{ return count; }

Set Collection::asSet() const
/*
	Convert this Collection to a Set.
*/
{  unsigned t_size = MAX(size(),DEFAULT_CAPACITY);
	Set cltn(t_size);
	addContentsTo(cltn);
	return cltn;
}

static unsigned set_capacity;

Set::Set(OIOin& strm)
:
#ifdef MI
	Object(strm),
#endif
	BASE(strm),
	contents((strm >> set_capacity, setCapacity(set_capacity)))
{
	_reader(strm);
}

Set::Set(OIOifd& fd)
:
#ifdef MI
	Object(fd),
#endif
	BASE(fd),
	contents((fd >> set_capacity, setCapacity(set_capacity) ))
{
	_reader(fd);
}

void Set::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	_storer(fd);
}

void Set::storer(OIOout& strm) const
{
	BASE::storer(strm);
	_storer(strm);
}

int Set::compare(const Object&) const
{
	shouldNotImplement("compare");
	return 0;
}
