/* MS-DOS version of NIH class library - Michael F. Murphy 4/93 */
/* Unix source: String.c,v 3.13 92/12/19 15:53:36 */

/*
Function:
	
Class String implements character string objects.  Operations provided
include & (concatenation) and () (substring extraction).  Type
conversions between String and char* are provided, permitting the two
to be used interchangeably in many contexts.  Note also that
SubStrings are not derived classes from Object.
*/
/*
 * change SubString::replace for MSDOS mem alloc
 * assert for malloc, realloc
 * pragmas
 */
#include "nihclstd.h"
#pragma hdrstop   

#include <ctype.h>
#ifdef __MS_DOS__
#include <stddef.h>     //ptrdiff_t
#endif

#include "NIHStrin.h"
#include "Range.h"
#include "nihclIO.h"

#define DEBUG
#include <assert.h>


#define	THIS	String
#define	BASE	Object

#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES
#define VIRTUAL_BASE_CLASSES Object::desc()

DEFINE_CLASS(String,1,"$"__FILE__" "__DATE__" "__TIME__"$",NULL)

extern const int NIHCL_BADCLASS, NIHCL_BADRANGE, NIHCL_INDEXRANGE, NIHCL_SUBSTRERR;

/* System-independent versions of toupper and tolower */
inline char to_upper(unsigned char c)	{ return (islower(c) ? (char)(c-'a'+'A') : (char)c); }
inline char to_lower(unsigned char c)	{ return (isupper(c) ? (char)(c-'A'+'a') : (char)c); }
// (char) MFM

//==== SubString functions:

/*
The following compare functions were implemented because strncmp is
not adequate for comparing character strings of unequal length.  For
example, strncmp("abc","abcd",3) will return 0.
*/

int SubString::compare(const char* cs) const
/*
Return integer greater than, equal to, or less than 0, according as
this SubString is lexicographically greater than, equal to, or less
than cs.
*/
{
	size_t cl = strlen(cs);                       // size_t MFM
	int result = strncmp(sp,cs,sl);
	if (result != 0 || sl == cl) return result;
	return (sl>cl ? 1 : -1);
}

int SubString::compare(const String& s) const
/*
Return integer greater than, equal to, or less than 0, according as
this SubString is lexicographically greater than, equal to, or less
than s.
*/
{
	int result = strncmp(sp,s.p,sl);
	if (result != 0 || sl == s.len) return result;
	return (sl>s.len ? 1 : -1);
}

int SubString::compare(const SubString& ss) const
/*
Return integer greater than, equal to, or less than 0, according as
this SubString is lexicographically greater than, equal to, or less
than SubString ss.
*/
{
	int result = strncmp(sp,ss.sp,MIN(sl,ss.sl));
	if (result != 0 || sl == ss.sl) return result;
	return (sl>ss.sl ? 1 : -1);
}

void SubString::dumpOn(ostream& strm) const
// Dump this SubString on output stream strm.
{
	strm << String(*this);
	strm << '[' << st->p << '(' << position() << ',' << sl << ")]";
}

void SubString::printOn(ostream& strm) const
// Print this SubString on output stream strm.
{
	strm << String(*this);
}

void SubString::operator=(const String& s)
{
	if (sl == s.length()) strncpy(sp,s.p,sl);
	else replace(s.p,s.len);
}

void SubString::operator=(const SubString& ss)
{
	if (sl == ss.sl) strncpy(sp,ss.sp,sl);
	else replace(ss.sp,ss.sl);
}

void SubString::operator=(const char* cs)
{
	size_t cslen = strlen(cs);                      // size_t MFM
	if (sl == cslen) strncpy(sp,cs,sl);
	else replace(cs,cslen);
}

#pragma warn -sig
#pragma warn -pia
void SubString::replace(const char* src, unsigned srclen)
// Replace this SubString with the argument string
// Terminology:
//	head: portion of destination string before this SubString
//	tail: portion of destination string after this SubString
{
#ifdef DEBUG_STRING
cerr << "replacing " << *this << " by ";
cerr.write(src, srclen);
cerr << " ...\n";
#endif
	bool overlap = NO;	// src overlaps destination String
	int tailDelta = 0;	// amount to adjust for tail movement
	char* srcbuf = 0;	// buffer to hold src if it overlaps SubString
// src overlap destination String?
	if (src >= st->p && src <= st->p+st->len) {
		overlap = YES;	// src overlaps destination String
// src overlap only tail of destination String?
		if (src >= sp+sl) tailDelta = srclen-sl;
		else {		
// src overlap this SubString?
			if (src+srclen > sp) {	// move src to buffer
				srcbuf = new char[srclen];
				strncpy(srcbuf,src,srclen);
				src = srcbuf;
				overlap = NO;	// no overlap now
			}
		}
	}
#ifdef DEBUG_STRING
cerr << "overlap=" << overlap << "  tailDelta=" << tailDelta << "  srcbuf=" << (int)srcbuf << '\n';
#endif
	if (srclen+st->len >= sl+st->alloc) {	// need to make String bigger

#ifdef __MS_DOS__
		ptrdiff_t ss_offset = sp - st->p ;
		ptrdiff_t src_offset= (char*)src - st->p;
#else
		char* p = st->p;
#endif
		st->alloc = st->len+srclen-sl+DEFAULT_STRING_EXTRA+1;
		st->p = (char*)realloc(st->p, st->alloc);
		assert(st->p);
#ifdef __MS_DOS__
		sp = st->p + (int)ss_offset;
#else
		sp += st->p-p;
#endif
		if (overlap)
#ifdef __MS_DOS__
			src =  st->p + (int)src_offset;
#else
			src += st->p-p;
#endif
#ifdef DEBUG_STRING
cerr << "realloc(" << st->alloc << ")  " << *this << '\n';
#endif
	}
	if (sl > srclen) {	// shift tail down
		const char* p = sp+sl;
		char* q = sp+srclen;
		while ((*q++ = *p++) != 0);      	// MFM quiet compiler
	}
	else {			// shift tail up
		const char* p = st->p+st->len;
		char* q = (char*)p+srclen-sl;
		unsigned n = (unsigned)(p-(sp+sl)+1);  // MFM quiet compiler
		while (n--) *q-- = *p--;
	}
	src += tailDelta;
	st->len += srclen-sl;
#ifdef DEBUG_STRING
cerr << "target " << *this << " source ";
cerr.write(src, srclen);
cerr << endl;
#endif
	strncpy(sp,src,srclen);		// insert src into destination
	if (srcbuf) free(srcbuf);
#ifdef DEBUG_STRING
cerr << "... result: " << *this << '\n';
#endif
}
#pragma warn .sig
#pragma warn .pia

String SubString::operator&(const SubString& ss) const
{
	String t(sl + ss.sl);
	strncpy (t.p, sp, sl);
	strncpy (&(t.p[sl]), ss.sp, ss.sl);
	t.len = sl + ss.sl;
	t.p[t.len] = '\0';
	return t;
}

String SubString::operator&(const String& s) const
{
	String t(sl + s.alloc -1);
	strncpy(t.p, sp, sl);
	strcpy(&(t.p[sl]), s.p);
	t.len = sl + s.len;
	return t;
}

String SubString::operator&(const char* cs) const
{
	size_t cslen = strlen(cs);                        // size_t MFM
	String t(sl + cslen);
	strncpy(t.p,sp,sl);
	strcpy(&(t.p[sl]),cs);
	t.len = sl + cslen;
	return t;
}

String operator&(const char* cs, const SubString& ss)
{
	unsigned cslen = strlen(cs);
	String t(cslen + ss.sl);
	strcpy(t.p,cs);
	strncpy(&(t.p[cslen]),ss.sp,ss.sl);
	t.len = cslen + ss.sl;
	t.p[t.len] = '\0';
	return t;
}

void SubString::checkSubStr() const
// check for legal SubString
{
	unsigned pos = position();
	unsigned len = st->len;
	if (pos+sl <= len) return;
	if (sl == 0 && pos == len) return;
	setError(NIHCL_SUBSTRERR,DEFAULT,st,st->className(),pos,sl);
}

//==== String Private functions:

void String::init(unsigned l, unsigned extra)
{
	len = l;
	alloc = len + extra + 1;
	p = (char*)malloc(alloc);
   assert(p);
	p[len] = '\0';
}

void String::indexRangeErr() const
{
	setError(NIHCL_INDEXRANGE,DEFAULT,this,className());
}

//==== String Constructors:

String::String(char c, unsigned l, unsigned extra)
{
	init(l,extra);
	unsigned i=len;
	while (i > 0) p[--i] = c;
}

String::String(const char* cs)
{ 
	len = strlen(cs);
	alloc = len + DEFAULT_STRING_EXTRA + 1;
	p = (char*)malloc(alloc);
   assert(p);
  	strcpy(p,cs); 
}

String::String(const char* cs, unsigned extra)
{ 
	len = strlen(cs);
	alloc = len + extra + 1;
	p = (char*)malloc(alloc);
   assert(p);
  	strcpy(p,cs); 
}

String::String(const String& s)
#if defined(__BCPLUSPLUS__) && defined(MI)
	: BASE(s)
#endif
{
	len = s.len;
	alloc = len + DEFAULT_STRING_EXTRA + 1;
	p = (char*)malloc(alloc);
   assert(p);
	strcpy (p,s.p);
}
 
String::String(const String& s, unsigned extra)
{
	len = s.len;
	alloc = len + extra + 1;
	p = (char*)malloc(alloc);
   assert(p);
	strcpy(p,s.p);
}

String::String(const SubString& ss)
{
	len = ss.sl;
	alloc = len + DEFAULT_STRING_EXTRA + 1;
	p = (char*)malloc(alloc);
   assert(p);
	strncpy(p,ss.sp,ss.sl);
	p[len] = '\0';
}

String::String(const SubString& ss, unsigned extra)
{
	len = ss.sl;
	alloc = len + extra + 1;
	p = (char*)malloc(alloc);
   assert(p);
	strncpy(p,ss.sp,ss.sl);
	p[len] = '\0';
}

String::~String()	{ free(p); }

//==== Operators:

SubString String::operator()(const Range& r)
{
	if (r.valid()) return SubString(*this,r.firstIndex(),r.length());
	else  setError(NIHCL_BADRANGE,DEFAULT,this,className(),"operator()",r.firstIndex(),r.length());
	/* not reached */
	return SubString(*this,r.firstIndex(),r.length());
}

const SubString String::operator()(const Range& r) const
{
	if (r.valid()) return SubString(*this,r.firstIndex(),r.length());
	else  setError(NIHCL_BADRANGE,DEFAULT,this,className(),"operator()",r.firstIndex(),r.length());
	/* not reached */
	return SubString(*this,r.firstIndex(),r.length());
}

void String::operator=(const String& s)
{
#if defined(__BCPLUSPLUS__) && defined(MI)
	if(isSame(s))
		return;
#endif
	if (p == s.p)
		return;
#if defined(__BCPLUSPLUS__) && defined(MI)
	BASE::operator=(s);
#endif
	len = s.len;
	if (len >= alloc) {
		free(p);
		p = (char*)malloc(alloc = s.alloc);
      assert(p);
	}
	strcpy(p,s.p);
}

void String::operator=(const SubString& ss)
{
	len = ss.sl;
	if (this == ss.st) {		// s = s(pos,len)
		if (ss.sp == p) {	// s = s(0,len)
			p[len] = '\0';
			return;
		}
		const char* src = ss.sp;
		char* dst = p;
		unsigned n = len;
		while (n--) *dst++ = *src++;
		*dst = '\0';
		return;
	}
	else if (len >= alloc) {
		alloc = ss.sl + DEFAULT_STRING_EXTRA + 1;
		free(p);
		p = (char*)malloc(alloc);
      assert(p);
	}
	strncpy(p, ss.sp, ss.sl);
	p[len] = '\0';
}

void String::operator=(const char* cs)
{
	len = strlen(cs);
	if (len >= alloc) {
		alloc = len + DEFAULT_STRING_EXTRA + 1;
		free(p);
		p = (char*)malloc(alloc);
      assert(p);
	}	
	strcpy(p,cs);
}

String String::operator&(const String& s) const
{
	String t(len+s.len);
	strcpy(t.p,p);
	strcpy(&(t.p[len]), s.p);
	t.len = len+s.len;
	return t;
}

String String::operator&(const SubString& ss) const
{
	String t(len+ss.sl);
	strcpy(t.p,p);
	strncpy(&(t.p[len]), ss.sp, ss.sl);
	t.len = len+ss.sl;
	t.p[t.len] = '\0';
	return t;
}

String String::operator&(const char* cs) const
{
	unsigned cslen = strlen(cs);
	String t(len+cslen);
	strcpy (t.p,p);
	strcpy (&(t.p[len]), cs);
	t.len = len+cslen;
	return t;
}

String operator&(const char* cs, const String& s)
{
	int cslen=strlen(cs);
	String t(cslen + s.len);
	strcpy(t.p,cs);
	strcpy(&(t.p[cslen]),s.p);
	t.len = cslen + s.len;
	return t;
}

String& String::operator&=(const String& s)
// Concatenate a String with another
{
	if (alloc <= len + s.len) {
		alloc += s.len + DEFAULT_STRING_EXTRA;
		p = (char*)realloc(p, alloc);
      assert(p);
	}
	strcpy(&p[len],s.p);
	len += s.len;
	return *this;
}	

String& String::operator&=(const SubString& ss)
{
	if (alloc <= len + ss.sl) {
		alloc += ss.sl + DEFAULT_STRING_EXTRA;
		if (this == ss.st) {	// s &= s(pos,len)
			char* t = p;
			p = (char*)malloc(alloc);
         assert(p);
			strcpy(p,t);
			strncpy(&(p[len]),ss.sp,ss.sl);
			free(t);
			len += ss.sl;
			p[len] = '\0';
			return *this;
		}
		else {
			p = (char*)realloc(p,alloc);
			assert(p);
      }
	}		
	strncpy(&(p[len]),ss.sp,ss.sl);
	len += ss.sl;
	p[len] = '\0';
	return *this;
}

String& String::operator&=(const char* cs)
{
	int cslen = strlen(cs);
	if (alloc <= len + cslen) {
		alloc += cslen + DEFAULT_STRING_EXTRA;
		p = (char*)realloc(p,alloc);
      assert(p);
	}
	strcpy(&(p[len]),cs);
	len += cslen;
	return *this;
}

void String::toAscii()
{
	unsigned i = len;
	char* q = p;
	while (i--) { *q = (char)toascii(*q); q++; }             // (char) MFM
}

void String::toLower()
{
	unsigned i = len;
	char* q = p;
	while (i--) { *q = to_lower(*q); q++; }
}

void String::toUpper()
{
	unsigned i = len;
	char* q = p;
	while (i--) { *q = to_upper(*q); q++; }
}

int String::compare(const Object& ob) const
{
	assertArgClass(ob,classDesc,"compare");
	return strcmp(p,castdown(ob).p);
}

void String::deepenShallowCopy() {}

static union char_mask {
	unsigned in[sizeof(int)];
	char ch[sizeof(int)*sizeof(int)];
	char_mask();
} mask;

char_mask::char_mask()
{
	for (unsigned i=0; i<sizeof(int); i++) {
		for (unsigned j=0; j<sizeof(int); j++) 
			ch[sizeof(int)*i+j] = (char)(j<i ? 0xff : 0);
	}
}

bool String::operator==(const String& s) const
{
	if (len != s.len) return NO;
	unsigned i = div_sizeof_int(len);
	const unsigned* q = (unsigned*)p;
	const unsigned* r = (unsigned*)s.p;
	while (i--) if (*q++ != *r++) return NO;
	if ((i = mod_sizeof_int(len)) != 0)
		if ((*q & mask.in[i]) != (*r & mask.in[i])) return NO;
	return YES;
}

unsigned long String::hash() const
{
	unsigned h = len;
	unsigned i = div_sizeof_int(len);
	unsigned* q = (unsigned*)p;
	while (i--) h ^= *q++;
	if ((i = mod_sizeof_int(len)) != 0)
		h ^= *q & mask.in[i];
	return h;
}

bool String::isEqual(const Object& a) const
{
	return a.isSpecies(classDesc) && *this==castdown(a);
}

void String::printOn(ostream& strm) const	{ strm << p; }

void String::scanFrom(istream& strm)
//	Read next line of input from strm into this String.
{
	ostream* os = strm.tie((ostream*)0);
	if (os != 0) {
		os->flush();
		strm.tie(os);
	}
	char c;
	strm.get(c);
	if (c != '\n') strm.putback(c);
	char temp[513];
	strm.get(temp,513);
	*this = String(temp);
}

unsigned String::reSize(unsigned new_capacity)
{
	if (new_capacity < len) new_capacity = len;
	if (alloc != new_capacity+1) {
		p = (char*)realloc(p,alloc = new_capacity+1);
      assert(p);
	}
	return alloc - 1;
}

unsigned String::size() const		{ return len; }

unsigned String::capacity() const	{ return alloc - 1; }

const Class* String::species() const
{
	return String::desc();
}

String::String(OIOifd& fd)
	: BASE(fd)
{
	fd >> len >> alloc;
	p = (char*)malloc(alloc);
   assert(p);
	fd.get(p,len+1);
}

void String::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	fd << len << alloc;
	fd.put(p,len+1);	// store terminating null character
}

String::String(OIOin& strm)
	: BASE(strm)
{
	strm >> len >> alloc;
	p = (char*)malloc(alloc);
   assert(p);
	strm.getCString(p,len+1);
}

void String::storer(OIOout& strm) const
{
	BASE::storer(strm);
	strm << len << alloc;
	strm.putCString(p);
}


