/******************************************************************************
* Module    :   UUdecode --- Test for and decode a uuencoded text.
*
* Author    :   John W. M. Stevens
******************************************************************************/

#include    "compiler.h"

#include    "unpost.h"
#include    "regexp.h"
#include    "parse.h"
#include    "uudec.h"
#include    "decode.h"
#include    "modflnm.h"
#include    "utils.h"

/*-----------------------------------------------------------------------------
| Routine   :   OpenBinFile() --- Find the begin line, and open the output
|               binary file.
|
| Inputs    :   InFlPtr - Pointer to input source file.
|               InBfr   - Buffer to read lines into when searching for
|                         begin line.
|
| Returns   :   Pointer to the binary file to write to.
-----------------------------------------------------------------------------*/

static
FILE        *OpenBinFile(FILE   *InFlPtr,
                         char   *FlName,
                         char   *InBfr)
{
    auto        FILE            *OutFlPtr;
    auto        char            **RetStrs;

    extern      FILE            *ErrFile;

    /*  Get begin line. */
    if (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
    {
        /*  Is this the begin line? */
        if ( MatchBegin(InBfr, &RetStrs) )
        {
            /*  Check to make sure that file does not already exist.    */
            if ( FileExists( FlName ) )
            {
                fprintf(ErrFile,
                        "\t%s %d : Error - file '%s' already exists.\n",
                        __FILE__,
                        __LINE__,
                        FlName);
                return( NULL );
            }

            /*  Open the file.  */
            if ((OutFlPtr = fopen(FlName, BIN_WRITE)) == NULL)
            {
                fprintf(ErrFile,
                        "\t%s %d : Error - %s\n\t'%s'\n",
                        __FILE__,
                        __LINE__,
                        strerror( errno ),
                        FlName);
                return( NULL );
            }

            /*  Return the file pointer.    */
            return( OutFlPtr );
        }
        else
        {
            /*  Missing begin line. */
            fprintf(ErrFile,
                    "\t%s %d : Error - missing begin line.\n",
                    __FILE__,
                    __LINE__);
            fprintf(ErrFile,
                    "\t'%s'\n",
                    InBfr);
        }
    }
    else
    {
        /*  Missing begin line. */
        fprintf(ErrFile,
                "\t%s %d : Error - missing begin line.\n",
                __FILE__,
                __LINE__);
    }

    /*  Return NULL file pointer.   */
    return( NULL );
}

/*-----------------------------------------------------------------------------
| Routine   :   WriteDesc() --- Write descriptions to a separate file.
|
| Inputs    :   InFlPtr     - Pointer to source file.
|               DescFlName  - Name of description file.
|               FlDesc      - Pointer to file descriptor structure.
|               InBfr       - Pointer to buffer to read lines into.
-----------------------------------------------------------------------------*/

static
int     WriteDesc(FILE      *InFlPtr,
                  char      *DescFlName,
                  FL_LIST   *FlDesc,
                  char      *InBfr)
{
    auto        CHK_UU_ENC      UULnType;
    auto        char            **RetStrs;
    auto        int             EncLen;
    auto        IDENT           *Hdr;
    auto        IDENT           *Body;
    auto        FILE            *DescFile;

    extern      FILE            *ErrFile;

	/*  Check to make sure that file does not already exist.    */
	if ( FileExists( DescFlName ) )
	{
		fprintf(ErrFile,
				"\t%s %d : Error - file '%s' already exists.\n",
				__FILE__,
				__LINE__,
				DescFlName);
		return( ERROR );
	}

    /*  Open the description file.  */
    if ((DescFile = fopen(DescFlName, TXT_WRITE)) == NULL)
    {
        fprintf(ErrFile,
                "\t%s %d : Error - Could not open description file ",
                __FILE__,
                __LINE__);
        fprintf(ErrFile,
                "'%s' for writing.\n",
                DescFlName);
        return( ERROR );
    }

    /*  Check to see which one of the segments exist.   */
    if ( FlDesc->Segs[0].Exists )
    {
        /*  Position file pointer to start of segment.  */
        if (fseek(InFlPtr, FlDesc->Segs[0].SegOfs, SEEK_SET) != 0)
        {
            fprintf(ErrFile,
                    "\t%s %d : Error - %s\n",
                    __FILE__,
                    __LINE__,
                    strerror( errno ));
            fclose( DescFile );
            return( ERROR );
        }

        /*  Dump lines until the start of the next segment is seen. */
        if (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
        {
            /*  Print the line. */
            fprintf(DescFile, "%s\n", InBfr);

            /*  Print description.  */
            while (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
            {
                /*  Is this a SEGMENT begin line?    */
                if ( MatchSegment(InBfr, &Hdr, &Body) )
                    break;

                /*  Print the line. */
                fprintf(DescFile, "%s\n", InBfr);
            }
        }
    }

    /*  Position file pointer to start of segment.  */
    if (fseek(InFlPtr, FlDesc->Segs[1].SegOfs, SEEK_SET) != 0)
    {
        fprintf(ErrFile,
                "\t%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                strerror( errno ));
        fclose( DescFile );
        return( ERROR );
    }

    /*  Dump lines until first UU encoded line is seen. */
    while (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
    {
        /*  Print the line. */
        fprintf(DescFile, "%s\n", InBfr);

        /*  Check to see if this line is a UU encoded line. */
        UULnType = ChkUULine(InBfr, &RetStrs, &EncLen);
        if (UULnType == UU_BEGIN ||
            UULnType == UU_END   ||
            UULnType == IS_UU_LINE)
            break;
    }

    /*  Return no error.    */
    fprintf(DescFile, "\n");
    fclose( DescFile );
    return( OK );
}

/*-----------------------------------------------------------------------------
| Routine   :   DecSeg() --- Decode a single segment.
|
| Inputs    :   InFlPtr     - Pointer to source file.
|               OutFlPtr    - Pointer to output binary file.
| Outputs   :   UULnType    - Returns UU line type.
|
| Returns   :   EOF     - For end of file.
|               ERROR   - For a binary file write error.
-----------------------------------------------------------------------------*/

int     DecSeg(FILE         *InFlPtr,
               FILE         *OutFlPtr,
               CHK_UU_ENC   *UULnType)
{
    auto        int     OutLen;

    /*  Externals used by this function.    */
    extern      char    InBfr[];
    extern      BYTE    OutBfr[];
    extern      FILE    *ErrFile;

    /*  Decode lines until end of segment.  */
    for ( ; ; )
    {
        /*  Get a line from the file.   */
        if (ReadLine(InFlPtr, InBfr, BFR_SIZE) == EOF)
            return( EOF );

        /*  Check the line to make sure that it is a UUENCODE
        *   line, and if it is, decode it.
        */
        *UULnType = DecUULine(InBfr, &OutLen, OutBfr);
        if (*UULnType == NOT_UU_LINE ||
            *UULnType == UU_BEGIN    ||
            *UULnType == UU_END)
            break;
        else if (*UULnType == UU_SPACE)
            continue;

        /*  Are there any bytes to write?   */
        if ( OutLen )
        {
            /*  Write the buffer to the output file.    */
            if (fwrite(OutBfr, 1, OutLen, OutFlPtr) != OutLen)
            {
                fprintf(ErrFile,
                        "\t%s %d : Error - Bad write to binary file.\n",
                        __FILE__,
                        __LINE__);
                return( ERROR );
            }
        }
    }

    /*  No errors, all is cool. */
    return( OK );
}

/*-----------------------------------------------------------------------------
| Routine   :   DeCode() --- Decode the file.
|
| Inputs    :   InFlPtr - Pointer to source file.
|               FlDesc  - Pointer to file descriptor.
-----------------------------------------------------------------------------*/

int         DeCode(FILE     *InFlPtr,
                   FL_LIST  *FlDesc)
{
    register    int         i;
    auto        FILE        *OutFlPtr;
    auto        int         ret;
    auto        CHK_UU_ENC  UULnType;
    auto        char        OutFlNm[FL_NM_SZ];

    /*  Externals used by this function.    */
    extern      int     DumpDesc;
    extern      char    InBfr[];
    extern      FILE    *ErrFile;

    /*  Check to make sure that all segments are present before we
    *   decode.
    */
    for (i = 1; i <= FlDesc->NoSegs; i++)
    {
        /*  Check for missing segments. */
        if (FlDesc->Segs[i].SegNo == 0)
        {
            fprintf(ErrFile,
                    "\t%s %d : Error - missing segment #%d.\n",
                    __FILE__,
                    __LINE__,
                    i);
            fprintf(ErrFile,
                    "\tBinary ID: '%s'\n",
                    FlDesc->IDString);
            return( ERROR );
        }
    }

    /*  Check for file name.  If none, report error.    */
    if (! FlDesc->FlName)
    {
        fprintf(ErrFile,
                "\t%s %d : Error - missing file name.\n",
                __FILE__,
                __LINE__);
        fprintf(ErrFile,
                "\tBinary ID: '%s'\n",
                FlDesc->IDString);
        return( ERROR );
    }

    /*  If we want descriptions, dump the first part of segment one.    */
    if ( DumpDesc )
    {
        /*  Dump description for zero segment, if one exists, and
        *   dump the first part of segment one (up to and including
        *   the begin line.
        */
        ModExten(FlDesc->FlName, ".dsc", OutFlNm);
        WriteDesc(InFlPtr, OutFlNm, FlDesc, InBfr);
    }

    /*  Now that we have scanned all lines, decode file.    */
    for (i = 1; i <= FlDesc->NoSegs; i++)
    {
        /*  Position file pointer to first UUencoded
        *   line of segment.
        */
        if (fseek(InFlPtr, FlDesc->Segs[i].UUOfs, SEEK_SET) != 0)
        {
            fprintf(ErrFile,
                    "\t%s %d : Error - %s\n",
                    __FILE__,
                    __LINE__,
                    strerror( errno ));
            return( ERROR );
        }

        /*  Open the binary file.   */
        if (i == 1)
        {
            /*  Open the binary file.   */
            if ((OutFlPtr = OpenBinFile(InFlPtr,
                                        FlDesc->FlName,
                                        InBfr)) == NULL)
                return( ERROR );
        }

        /*  Decode lines until end of segment.  */
        if ((ret = DecSeg(InFlPtr, OutFlPtr, &UULnType)) == EOF ||
            ret == ERROR)
        {
            /*  Close output file, return error.    */
            fclose( OutFlPtr );
            return( ret );
        }

        /*  Check for various errors.   */
        if (UULnType == NOT_UU_LINE)
        {
            /*  Is there supposed to be a end line in this segment? */
            if (i == FlDesc->NoSegs)
            {
                fprintf(ErrFile,
                        "\t%s %d : Error - Missing UU end line.\n",
                        __FILE__,
                        __LINE__);
                fprintf(ErrFile,
                        "\t'%s'\n",
                        InBfr);
                fclose( OutFlPtr );
                return( ERROR );
            }
        }
        else if (UULnType == UU_END)
        {
            /*  Is this supposed to be the end? */
            if (i < FlDesc->NoSegs)
            {
                fprintf(ErrFile,
                        "\t%s %d : Warning - Early uuencode end line.\n",
                        __FILE__,
                        __LINE__);
            }
            break;
        }
        else if (UULnType == UU_BEGIN)
        {
            /*  What the HELL is this doing here?!  */
            fprintf(ErrFile,
                    "\t%s %d : Error - Unexpected UU begin line.\n",
                    __FILE__,
                    __LINE__);
            return( ERROR );
        }
    }

    /*  Close the output file.  */
    fclose( OutFlPtr );
    return( OK );
}

/*-----------------------------------------------------------------------------
| Routine   :   MultiWriteDesc() --- Write descriptions to a separate file.
|
| Inputs    :   DescFlName  - Name of description file.
|               FlDesc      - Pointer to file descriptor structure.
|               InBfr       - Pointer to buffer to read lines into.
-----------------------------------------------------------------------------*/

static
int		MultiWriteDesc(char		*DescFlName,
					   FL_LIST	*FlDesc,
					   char		*InBfr)
{
	auto		FILE			*InFlPtr;
    auto        CHK_UU_ENC      UULnType;
    auto        char            **RetStrs;
    auto        int             EncLen;
    auto        IDENT           *Hdr;
    auto        IDENT           *Body;
    auto        FILE            *DescFile;

    extern      FILE            *ErrFile;

	/*  Check to make sure that file does not already exist.    */
	if ( FileExists( DescFlName ) )
	{
		fprintf(ErrFile,
				"\t%s %d : Error - file '%s' already exists.\n",
				__FILE__,
				__LINE__,
				DescFlName);
		return( ERROR );
	}

    /*  Open the description file.  */
    if ((DescFile = fopen(DescFlName, TXT_WRITE)) == NULL)
    {
        fprintf(ErrFile,
                "\t%s %d : Error - Could not open description file ",
                __FILE__,
                __LINE__);
        fprintf(ErrFile,
                "'%s' for writing.\n",
                DescFlName);
        return( ERROR );
    }

    /*  Check to see which one of the segments exist.   */
    if ( FlDesc->Segs[0].Exists )
    {
		/*	Open the first file.	*/
		if ((InFlPtr = fopen(FlDesc->Segs[0].SrcFlNm, BIN_READ)) == NULL)
		{
			fprintf(ErrFile,
					"%s %d : Error - could not open file '%s' for reading.\n",
					__FILE__,
					__LINE__,
					FlDesc->Segs[0].SrcFlNm);
			fclose( DescFile );
			return( ERROR );
		}

        /*  Position file pointer to start of segment.  */
        if (fseek(InFlPtr, FlDesc->Segs[0].SegOfs, SEEK_SET) != 0)
        {
            fprintf(ErrFile,
                    "\t%s %d : Error - %s\n",
                    __FILE__,
                    __LINE__,
                    strerror( errno ));
            fclose( DescFile );
            return( ERROR );
        }

        /*  Dump lines until the start of the next segment is seen. */
        if (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
        {
            /*  Print the line. */
            fprintf(DescFile, "%s\n", InBfr);

            /*  Print description.  */
            while (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
            {
                /*  Is this a SEGMENT begin line?    */
                if ( MatchSegment(InBfr, &Hdr, &Body) )
                    break;

                /*  Print the line. */
                fprintf(DescFile, "%s\n", InBfr);
            }
        }
		fclose( InFlPtr );
    }

	/*	Open the first file.	*/
	if ((InFlPtr = fopen(FlDesc->Segs[1].SrcFlNm, BIN_READ)) == NULL)
	{
		fprintf(ErrFile,
				"%s %d : Error - could not open file '%s' for reading.\n",
				__FILE__,
				__LINE__,
				FlDesc->Segs[0].SrcFlNm);
		fclose( DescFile );
		return( ERROR );
	}

    /*  Position file pointer to start of segment.  */
    if (fseek(InFlPtr, FlDesc->Segs[1].SegOfs, SEEK_SET) != 0)
    {
        fprintf(ErrFile,
                "\t%s %d : Error - %s\n",
                __FILE__,
                __LINE__,
                strerror( errno ));
        fclose( InFlPtr );
        fclose( DescFile );
        return( ERROR );
    }

    /*  Dump lines until first UU encoded line is seen. */
    while (ReadLine(InFlPtr, InBfr, BFR_SIZE) != EOF)
    {
        /*  Print the line. */
        fprintf(DescFile, "%s\n", InBfr);

        /*  Check to see if this line is a UU encoded line. */
        UULnType = ChkUULine(InBfr, &RetStrs, &EncLen);
        if (UULnType == UU_BEGIN ||
            UULnType == UU_END   ||
            UULnType == IS_UU_LINE)
            break;
    }
    fprintf(DescFile, "\n");

    /*  Return no error.    */
	fclose( InFlPtr );
    fclose( DescFile );
    return( OK );
}

/*-----------------------------------------------------------------------------
| Routine   :   MultiDeCode() --- Decode a uuencoded posting that is
|				scattered across multiple source files.
|
| Inputs    :   InFlPtr - Pointer to source file.
|               FlDesc  - Pointer to file descriptor.
-----------------------------------------------------------------------------*/

int	MultiDeCode(FL_LIST	*FlDesc)
{
    register    int         i;
    auto        FILE        *InFlPtr;
    auto        FILE        *OutFlPtr;
    auto        int         ret;
    auto        CHK_UU_ENC  UULnType;
    auto        char        OutFlNm[FL_NM_SZ];

    /*  Externals used by this function.    */
    extern      int     DumpDesc;
    extern      char    InBfr[];
    extern      FILE    *ErrFile;

    /*  Check to make sure that all segments are present before we
    *   decode.
    */
    for (i = 1; i <= FlDesc->NoSegs; i++)
    {
        /*  Check for missing segments. */
        if (FlDesc->Segs[i].SegNo == 0)
        {
            fprintf(ErrFile,
                    "\t%s %d : Error - missing segment #%d.\n",
                    __FILE__,
                    __LINE__,
                    i);
            fprintf(ErrFile,
                    "\tBinary ID: '%s'\n",
                    FlDesc->IDString);
            return( ERROR );
        }
    }

    /*  Check for file name.  If none, report error.    */
    if (! FlDesc->FlName)
    {
        fprintf(ErrFile,
                "\t%s %d : Error - missing file name.\n",
                __FILE__,
                __LINE__);
        fprintf(ErrFile,
                "\tBinary ID: '%s'\n",
                FlDesc->IDString);
        return( ERROR );
    }

	/*	Modify the file name for writing descriptions.	*/
	if ( DumpDesc )
	{
		ModExten(FlDesc->FlName, ".dsc", OutFlNm);
		MultiWriteDesc(OutFlNm, FlDesc, InBfr);
	}

    /*  Now that we have scanned all lines, decode file.    */
    for (i = 1; i <= FlDesc->NoSegs; i++)
    {
		/*	Open the first file.	*/
		if ((InFlPtr = fopen(FlDesc->Segs[i].SrcFlNm, BIN_READ)) == NULL)
		{
			fprintf(ErrFile,
					"%s %d : Error - could not open file '%s' for reading.\n",
					__FILE__,
					__LINE__,
					FlDesc->Segs[i].SrcFlNm);
			return( ERROR );
		}

        /*  Position file pointer to first UUencoded
        *   line of segment.
        */
        if (fseek(InFlPtr, FlDesc->Segs[i].UUOfs, SEEK_SET) != 0)
        {
            fprintf(ErrFile,
                    "\t%s %d : Error - %s\n",
                    __FILE__,
                    __LINE__,
                    strerror( errno ));
			fclose( InFlPtr );
            return( ERROR );
        }

        /*  Open the binary file.   */
        if (i == 1)
        {
            /*  Open the binary file.   */
            if ((OutFlPtr = OpenBinFile(InFlPtr,
                                        FlDesc->FlName,
                                        InBfr)) == NULL)
			{
				fclose( InFlPtr );
                return( ERROR );
			}
        }

        /*  Decode lines until end of segment.  */
        if ((ret = DecSeg(InFlPtr, OutFlPtr, &UULnType)) == EOF ||
            ret == ERROR)
        {
            /*  Close output file, return error.    */
			fclose( InFlPtr );
            fclose( OutFlPtr );
            return( ret );
        }

        /*  Check for various errors.   */
        if (UULnType == NOT_UU_LINE)
        {
            /*  Is there supposed to be a end line in this segment? */
            if (i == FlDesc->NoSegs)
            {
                fprintf(ErrFile,
                        "\t%s %d : Error - Missing UU end line.\n",
                        __FILE__,
                        __LINE__);
                fprintf(ErrFile,
                        "\t'%s'\n",
                        InBfr);
                fclose( InFlPtr );
                fclose( OutFlPtr );
                return( ERROR );
            }
        }
        else if (UULnType == UU_END)
        {
            /*  Is this supposed to be the end? */
            if (i < FlDesc->NoSegs)
            {
                fprintf(ErrFile,
                        "\t%s %d : Warning - Early uuencode end line.\n",
                        __FILE__,
                        __LINE__);
            }
            break;
        }
        else if (UULnType == UU_BEGIN)
        {
            /*  What the HELL is this doing here?!  */
            fprintf(ErrFile,
                    "\t%s %d : Error - Unexpected UU begin line.\n",
                    __FILE__,
                    __LINE__);
			fclose( InFlPtr );
			fclose( OutFlPtr );
            return( ERROR );
        }

		/*	Close the input file pointer.	*/
		fclose( InFlPtr );
    }

    /*  Close the output file.  */
    fclose( OutFlPtr );
    return( OK );
}
