/****************************************************************************
*
*                         The Universal VESA VBE
*
*                   Copyright (C) 1993 Kendall Bennett.
*                           All rights reserved.
*
* Filename:     $RCSfile: ports.c $
* Version:      $Revision: 1.1 $
*
* Language:     ANSI C
* Environment:  IBM PC (MS DOS)
*
* Description:  Module for performing simplified I/O port manipulations.
*               These routines are not particularly fast, but the serve
*               the purpose that we require since speed is not essential.
*
*               MUST be compiled in the large memory model.
*
* $Id: ports.c 1.1 1993/09/19 01:26:00 kjb release $
*
* Revision History:
* -----------------
*
* $Log: ports.c $
* Revision 1.1  1993/09/19  01:26:00  kjb
* Initial revision
*
****************************************************************************/

#include <dos.h>
#include "univbe.h"

/*----------------------------- Implementation ----------------------------*/

int rdinx(int port,int index)
/****************************************************************************
*
* Function:     rdinx
* Parameters:   port    - I/O port to read value from
*               index   - Port index to read
* Returns:      Byte read from 'port' register 'index'.
*
****************************************************************************/
{
    if (port == 0x3C0)  inp(0x3DA);     /* Reset attribute reg flip-flop*/
    outp(port,index);
    return inp(port+1);
}

void wrinx(int port,int index,int value)
/****************************************************************************
*
* Function:     wrinx
* Parameters:   port    - I/O port to write to
*               index   - Port index to write
*               value   - Byte to write to port
*
* Description:  Writes a byte value to the 'port' register 'index'.
*
****************************************************************************/
{
    outp(port,index);
    outp(port+1,value);
}

void modinx(int port,int index,int mask,int value)
/****************************************************************************
*
* Function:     modinx
* Parameters:   port    - I/O port to modify
*               index   - Port index register to modify
*               mask    - Mask of bits to modify (1 for valid bit)
*               value   - New value to store in the modified bits
*
* Description:  Read the current value of the specified port, and modifies
*               the specified bits and store the result back in the
*               port.
*
****************************************************************************/
{
    wrinx(port,index,(rdinx(port,index) & ~mask) | (value & mask));
}

bool tstreg(int port,int mask)
/****************************************************************************
*
* Function:     tstreg
* Parameters:   port    - I/O port to test
*               mask    - Mask of bits to test
* Returns:      True if the I/O port bits could be modified.
*
* Description:  Checks to see if the bits specified in the mask can be
*               modified correctly for the specified I/O port. Returns
*               true if successful.
*
****************************************************************************/
{
    int old,zeros,ones;

    old = inp(port);                /* Save old port value              */
    outp(port,old & ~mask);         /* Clear all bits specified in mask */
    zeros = inp(port) & mask;       /* Read back value now set          */
    outp(port,old | mask);          /* Set all bits specified in mask   */
    ones = inp(port) & mask;        /* Read back value now set          */
    outp(port,old);                 /* Restore original port value      */
    return (zeros == 0 && ones == mask);
}

bool tstinx(int port,int index,int mask)
/****************************************************************************
*
* Function:     tstinx
* Parameters:   port    - I/O port to test
*               index   - Register index to test
*               mask    - Mask of bits to test for
* Returns:      True if the 'port' register 'index' is alive.
*
* Description:  Checks to see if the bits specified in the mask can be
*               modified correctly for the specified port register. Returns
*               true if successful.
*
****************************************************************************/
{
    int old,zeros,ones;

    old = rdinx(port,index);        /* Save old port value              */
    wrinx(port,index,old & ~mask);  /* Clear all bits specified in mask */
    zeros = rdinx(port,index) & mask;   /* Read back value now set      */
    wrinx(port,index,old | mask);   /* Set all bits specified in mask   */
    ones = rdinx(port,index) & mask;    /* Read back value now set      */
    wrinx(port,index,old);          /* Restore original port value      */
    return (zeros == 0 && ones == mask);
}
