/*===========================================================================
 |                  P r o g r a m   K B D T R F _ I
 |--------------------------------------------------------------------------
 |                    Mark Meier, CIS: 76330,3402
 |                           Begun: 04/94
 |                        Last Change: 04/94
 *=========================================================================*/
/*===========================================================================
 |     I n c l u d e   F i l e s
 *=========================================================================*/
#include <stdio.h>
#include <math.h>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include "pxp.h"
#include "trig.h"

void mm_atof(char *, double *, int *, char *);
void mm_process_chunk(char *s, double *, int *, char *errmsg);
double mm_convert_str(char *s);
int mm_instr(char *s, char ch);
int mm_valid_str(char *s, char *valid);
int mm_modify_vlist(float master_scale, int master_type, char *errmsg);

/*===========================================================================
 |     D e c l a r a t i o n s
 *=========================================================================*/
#define MM_CHECK_VERSION    0x0200
#define MM_MASTER_SCALE     0x0201
#define MM_CREATE_COPY      0X0202
#define MM_GET_VLIST        0x0204
#define MM_PUT_VLIST        0x0206
#define MM_GET_FLIST        0x0208
#define MM_PUT_FLIST        0x0210
#define MM_GET_EXTENTS      0x0212
#define MM_CREATE_OBJ       0x0214
#define MM_OK_EXIT          0x0216
#define MM_ERROR_EXIT       0x0218

#define MM_STATUS_OKAY    1
#define MM_STATUS_ERROR   0

#define MM_RELATIVE       0
#define MM_ABSOLUTE       1

#define MM_TRANS          0
#define MM_SCALE          1
#define MM_ROTATE         2

#define MM_STRING_SIZE   16
#define MM_FLAT_TOL  0.000001

#define MM_INCH_SYM     ((char) 34)
#define MM_FEET_SYM     ((char) 39)
#define MM_FRAC_DELIM   ((char) 45)
#define MM_FRAC_SYM     ((char) 47)
#define MM_STR_TERM     ((char) 0)

/*===========================================================================
 |     L i n e   I D   D e c l a r a t i o n s
 *========================================================================*/
#define MM_VERSION    0x71FB
#define MM_TRF             1

#define MM_XRAD1           2
#define MM_XSTR1           3
#define MM_YRAD1           4
#define MM_YSTR1           5
#define MM_ZRAD1           6
#define MM_ZSTR1           7

#define MM_ROTRAD          8
#define MM_ROT             9

#define MM_XSC            10
#define MM_YSC            11
#define MM_ZSC            12
 
#define MM_CRDRAD         13

#define MM_XRAD2          14
#define MM_XSTR2          15
#define MM_YRAD2          16
#define MM_YSTR2          17
#define MM_ZRAD2          18
#define MM_ZSTR2          19

#define MM_OBJECT_NAME    20

static VData *vdata = NULL;
static FData *fdata = NULL;
static ExtInfo extents;
static int vlen, flen, flags, tverts;
static char obj_name[MM_STRING_SIZE], errmsg[128];
static float master_scale;
static int master_type;

typedef struct {
   ulong version;
   int trf;

   int xrad1;
   char xstr1[MM_STRING_SIZE];
   int yrad1;
   char ystr1[MM_STRING_SIZE];
   int zrad1;
   char zstr1[MM_STRING_SIZE];

   int rotrad;
   float rot;

   char xsc[MM_STRING_SIZE];
   char ysc[MM_STRING_SIZE];
   char zsc[MM_STRING_SIZE];

   int crdrad;

   int xrad2;
   char xstr2[MM_STRING_SIZE];
   int yrad2;
   char ystr2[MM_STRING_SIZE];
   int zrad2;
   char zstr2[MM_STRING_SIZE];

   char name[MM_STRING_SIZE];
} State;

static State init_state = {
    MM_VERSION, 
    0, 
    3, "0", 3, "0", 3, "0", 
    0, 0.0,
    "1.0", "1.0", "1.0",
    0,
    1, "0", 1, "0", 1, "0", 
    "Object01"};

static State state = { 
    MM_VERSION, 
    0, 
    3, "0", 3, "0", 3, "0", 
    0, 0.0,
    "1.0", "1.0", "1.0",
    0,
    1, "0", 1, "0", 1, "0", 
    "Object01"};

DlgEntry cdialog[] = {
   0, "TITLE=\"Object Transformations via the Keyboard\"",
   0, "TITLE=\"by Mark Meier, 76330,3402, Version 1.0a\"",
   0, "TITLE=\"\"",
   MM_TRF, "RADIO=\"Transformation:\",\"Translate\",\"Scale\",\"Rotate\"",
   0, "TITLE=\"Translate / Scale Mode:\"",
   MM_CRDRAD, "-RADIO=\"\",\"Relative\",\"Absolute\"",

   0, "TITLE=\"\"",
   0, "TITLE=\"Translation Displacement (if Mode = Relative) -or-\"",
   0, "TITLE=\"Translation Source Point (if Mode = Absolute) -or-\"",
   0, "TITLE=\"Center of Rotation / Scale\"",
   MM_XRAD1, "RADIO=\"\",\"Width Min \",\"Width Cen \",\"Width Max \", \"Coord->\"",
   MM_XSTR1, "-STRING=\"\", 14, 10",
   MM_YRAD1, "RADIO=\"\",\"Depth Min \",\"Depth Cen \",\"Depth Max \", \"Coord->\"",
   MM_YSTR1, "-STRING=\"\", 14, 10",
   MM_ZRAD1, "RADIO=\"\",\"Height Min\",\"Height Cen\",\"Height Max\", \"Coord->\"",
   MM_ZSTR1, "-STRING=\"\", 14, 10",

   0, "TITLE=\"\"",
   0, "TITLE=\"Scale:\"",
   MM_XSC, "-STRING=\"Width->\", 14, 9",
   MM_YSC, "-STRING=\"Depth->\", 14, 9",
   MM_ZSC, "-STRING=\"Height->\", 14, 9",

   0, "TITLE=\"\"",
   0, "TITLE=\"Rotation Axis:\"",
   MM_ROTRAD, "-RADIO=\"\",\"Width\",\"Depth\",\"Height\"",
   MM_ROT, "-FLOAT=\"Angle->\", 8",

   0, "TITLE=\"\"",

   0, "TITLE=\"Translation Destination Point (if Mode = Absolute)\"",
   MM_XRAD2, "RADIO=\"\",\"Width Min \",\"Width Cen \",\"Width Max \", \"Coord->\"",
   MM_XSTR2, "-STRING=\"\", 14, 10",
   MM_YRAD2, "RADIO=\"\",\"Depth Min \",\"Depth Cen \",\"Depth Max \", \"Coord->\"",
   MM_YSTR2, "-STRING=\"\", 14, 10",
   MM_ZRAD2, "RADIO=\"\",\"Height Min\",\"Height Cen\",\"Height Max\", \"Coord->\"",
   MM_ZSTR2, "-STRING=\"\", 14, 10",

   0, "TITLE=\"\"",
   MM_OBJECT_NAME, "STRING=\"Transformed Object Name:\", 10, 10",
   0, "TITLE=\"\"",
   0, NULL
};

/*===========================================================================
 |     C l i e n t   F u n c t i o n s
 *=========================================================================*/
void ClientSetStateVar(int id, void *ptr)
{
   OVL o;
   ulong *ul;
   char *s;

   ul = (ulong *)ptr;
   s = (char *)ptr;
   o.ul = *ul;

   switch (id) {
      case MM_TRF: state.trf = o.i; break;
      case MM_XRAD1: state.xrad1 = o.i; break;
      case MM_XSTR1: strcpy(state.xstr1, s); break;
      case MM_YRAD1: state.yrad1 = o.i; break;
      case MM_YSTR1: strcpy(state.ystr1, s); break;
      case MM_ZRAD1: state.zrad1 = o.i; break;
      case MM_ZSTR1: strcpy(state.zstr1, s); break;

      case MM_CRDRAD: state.crdrad = o.i; break;

      case MM_XRAD2: state.xrad2 = o.i; break;
      case MM_XSTR2: strcpy(state.xstr2, s); break;
      case MM_YRAD2: state.yrad2 = o.i; break;
      case MM_YSTR2: strcpy(state.ystr2, s); break;
      case MM_ZRAD2: state.zrad2 = o.i; break;
      case MM_ZSTR2: strcpy(state.zstr2, s); break;

      case MM_ROTRAD: state.rotrad = o.i; break;
      case MM_ROT: state.rot = o.f; break;

      case MM_XSC: strcpy(state.xsc, s); break;
      case MM_YSC: strcpy(state.ysc, s); break;
      case MM_ZSC: strcpy(state.zsc, s); break;

      case MM_OBJECT_NAME: strcpy(state.name, s); break;
   };
}

ulong ClientGetStateVar (int id)
{
   OVL o;

   switch (id) {
      case MM_TRF: o.i = state.trf; break;
      case MM_XRAD1: o.i = state.xrad1; break;
      case MM_XSTR1: o.s = state.xstr1; break;
      case MM_YRAD1: o.i = state.yrad1; break;
      case MM_YSTR1: o.s = state.ystr1; break;
      case MM_ZRAD1: o.i = state.zrad1; break;
      case MM_ZSTR1: o.s = state.zstr1; break;

      case MM_CRDRAD: o.i = state.crdrad; break;

      case MM_XRAD2: o.i = state.xrad2; break;
      case MM_XSTR2: o.s = state.xstr2; break;
      case MM_YRAD2: o.i = state.yrad2; break;
      case MM_YSTR2: o.s = state.ystr2; break;
      case MM_ZRAD2: o.i = state.zrad2; break;
      case MM_ZSTR2: o.s = state.zstr2; break;

      case MM_ROTRAD: o.i = state.rotrad; break;
      case MM_ROT: o.f = state.rot; break;

      case MM_XSC: o.s = state.xsc; break;
      case MM_YSC: o.s = state.ysc; break;
      case MM_ZSC: o.s = state.zsc; break;

      case MM_OBJECT_NAME: o.s = state.name; break;
   };
   return (o.ul);   
}

int ClientVarSize(int id)
{
   int size;

   switch (id) {
      case MM_TRF: size = 1; break;
      case MM_XRAD1: size = 1; break;
      case MM_XSTR1: size = 4; break;
      case MM_YRAD1: size = 1; break;
      case MM_YSTR1: size = 4; break;
      case MM_ZRAD1: size = 1; break;
      case MM_ZSTR1: size = 4; break;

      case MM_CRDRAD: size = 1; break;

      case MM_XRAD2: size = 1; break;
      case MM_XSTR2: size = 4; break;
      case MM_YRAD2: size = 1; break;
      case MM_YSTR2: size = 4; break;
      case MM_ZRAD2: size = 1; break;
      case MM_ZSTR2: size = 4; break;

      case MM_ROTRAD: size = 1; break;
      case MM_ROT: size = 1; break;

      case MM_XSC: size = 4; break;
      case MM_YSC: size = 4; break;
      case MM_ZSC: size = 4; break;

      case MM_OBJECT_NAME: size = 3; break;

      default: size = 1; break;
   };
   return (size);
}

char *ClientGetState(int *size)
{
   *size = sizeof(State);
   return((char *)&state);
}

void ClientResetState(void)
{
   state = init_state;
}

DlgEntry *ClientDialog(int n)
{
   return (&cdialog[n]);
}

char *ClientName(void)
{
   return("KBDTRF_I");
}

int ClientUsesInitDialog(void)
{
   return 1;
}

void ClientTerminate(void)
{
   if(vdata)
      free(vdata);
   vdata=NULL;
   if(fdata)
      free(fdata);
   fdata=NULL;
}

int ClientIsUniversal(void){
   return(0);
}

void ClientStartup(EXPbuf *b)
{
   b->status = 1;
   b->opcode = EXP_NOP;
   b->usercode = MM_CHECK_VERSION;
}

/*===========================================================================
 | Function: ClientUserCode
 |  Purpose: To process the packet commands.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void ClientUserCode(EXPbuf *b)
{
  /*
   * Declarations */
   int mm_status;
  /*
   * Process the usercodes passed back via EXPbuf... */
   switch (b->usercode) {
      case MM_CHECK_VERSION: 
        /*
         * Check the version */
         if (studio_version() < 300) {
            strcpy(b->data.string, "Sorry: 3D Studio r3.0 Required. :(");
            b->opcode = EXP_CONTINUE;
            b->usercode = MM_ERROR_EXIT;
         }
         else {
            b->opcode = EXP_MASTER_SCALE;
            b->usercode = MM_MASTER_SCALE;
         }
         break;
      case MM_MASTER_SCALE:
        /*
         * Store the master scale information */
         master_scale = b->data.msc.master_scale;
         master_type = b->data.msc.type;
         b->opcode = EXP_PICK_OBJECT;
         b->usercode = MM_CREATE_COPY;
         strcpy(b->data.string, "Pick the Source Object to Transform:");
         break;
      case MM_CREATE_COPY:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
            strcpy(obj_name, b->data.object.name);
            vlen = b->data.object.verts;
            flen = b->data.object.faces;
            tverts = b->data.object.tverts;

            b->opcode = EXP_READY_OBJ;
            b->usercode = MM_GET_EXTENTS;
            b->data.object.flags = 0;
         }
         break;
      case MM_GET_EXTENTS:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
            b->opcode = EXP_OBJ_EXTENTS;
            b->usercode = MM_GET_VLIST;
            strcpy(b->data.exts.name, obj_name);
         }
         break;
      case MM_GET_VLIST:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
            extents = b->data.exts;
            mm_status = mm_alloc_mem(b, vlen, flen);
            if (mm_status == MM_STATUS_OKAY) {
               b->opcode = EXP_GET_VLIST;
               b->usercode = MM_PUT_VLIST;
               b->data.vlist.start = 0;
               b->data.vlist.count = vlen;
               b->data.vlist.data = (VData _far *)vdata;
            }
            else {
               b->opcode=EXP_CONTINUE;
               b->usercode=EXP_TERMINATE;
               strcpy(b->data.string, "No RAM for object");
            }
         }
         break;
      case MM_PUT_VLIST:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
            mm_status = mm_modify_vlist(master_scale, master_type, errmsg);
            if (mm_status == MM_STATUS_OKAY) {
               b->opcode = EXP_PUT_VLIST;
               b->usercode = MM_GET_FLIST;
               b->data.vlist.start = 0;
               b->data.vlist.count = vlen;
               b->data.vlist.data = (VData _far *)vdata;
            }
            else {
               b->opcode=EXP_CONTINUE;
               b->usercode=EXP_TERMINATE;
               strcpy(b->data.string, errmsg);
            }
         }
         break;
      case MM_GET_FLIST:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
            b->opcode = EXP_GET_FLIST;
            b->usercode = MM_PUT_FLIST;
            b->data.flist.start = 0;
            b->data.flist.count = flen;
            b->data.flist.data = (FData _far *)fdata;
         }
         break;
      case MM_PUT_FLIST:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
            b->opcode = EXP_PUT_FLIST;
            b->usercode = MM_CREATE_OBJ;
            b->data.flist.start = 0;
            b->data.flist.count = flen;
            b->data.flist.data = (FData _far *)fdata;
         }
         break;
      case MM_CREATE_OBJ:
         if (b->status == 0) {
            ipas_terminate(b);
         }
         else {
           /*
            * Create the object */
            strcpy(b->data.object.name, state.name);
            b->data.object.flags = 0;
            b->data.object.verts = vlen;
            b->data.object.faces = flen;
            b->data.object.tverts = tverts;
            b->opcode = EXP_CREATE_OBJ;
            b->usercode = MM_OK_EXIT;
         }
         break;
      case MM_OK_EXIT:
         b->status = 1;
         b->opcode = b->usercode = EXP_TERMINATE;
         break;
      case MM_ERROR_EXIT:
         b->status = 0;
         b->opcode = b->usercode = EXP_TERMINATE;
         break;
      default:
         b->status = 0;
         b->opcode = b->usercode = EXP_TERMINATE;
         break;
   };
}

/*===========================================================================
 | Function: mm_modify_vlist
 |  Purpose: To transform the coords of the object based on the radio 
 |           button selected.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
int mm_modify_vlist(float master_scale, int master_type, char *errmsg)
{
   VData *vptr;
   float x1, y1, z1, x2, y2, z2, rot, xs, ys, zs;
   float xtr, ytr, ztr;
   int status;

   vptr = &vdata[0];

  /*
   * Convert the strings to floats */
   get_values(&x1, &y1, &z1, &x2, &y2, &z2, &rot, &xs, &ys, &zs,
      master_scale, master_type, &status, errmsg);
   if (status == MM_STATUS_ERROR) {
      return status;
   }

   switch(state.trf) {
      case MM_TRANS:
         if (state.crdrad == MM_ABSOLUTE) {
            xtr = x2-x1; ytr = y2-y1; ztr = z2-z1;
            mm_translate_vlist(vptr, vlen, xtr, ytr, ztr);
         }
         else {
            mm_translate_vlist(vptr, vlen, x1, y1, z1);
         }
         break;
      case MM_ROTATE:
         xtr = 0.0-x1; ytr = 0.0-y1; ztr = 0.0-z1;
         mm_translate_vlist(vptr, vlen, xtr, ytr, ztr);
         mm_rotate_vlist(vptr, vlen, state.rotrad, rot);
         xtr = -xtr; ytr = -ytr; ztr = -ztr;
         mm_translate_vlist(vptr, vlen, xtr, ytr, ztr);
         break;
      case MM_SCALE:
         xtr = -x1; ytr = -y1; ztr = -z1;
         mm_translate_vlist(vptr, vlen, xtr, ytr, ztr);
         mm_scale_vlist(vptr, vlen, xs, ys, zs);
         mm_translate_vlist(vptr, vlen, x1, y1, z1);
         break;
   };
   return(MM_STATUS_OKAY);
}

/*===========================================================================
 | Function: get_values
 |  Purpose: To compute the coords the user has entered based on the state
 |           of the radio buttons.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void get_values(x1, y1, z1, x2, y2, z2, rot, xs, ys, zs, 
   master_scale, master_type, status, errmsg)
float *x1, *y1, *z1, *x2, *y2, *z2, *rot, *xs, *ys, *zs, master_scale;
int *status;
char *errmsg;
{
   char errmsg1[128];
   double dx1, dy1, dz1, dx2, dy2, dz2, axs, ays, azs,
      x_factor, y_factor, z_factor;
  /*
   * Store the rotation value */
   *rot = state.rot;
  /* 
   * Handle the scale factors */
   if (state.trf == MM_SCALE) {
      if (state.crdrad == MM_RELATIVE) {
         x_factor = 1.0;
         y_factor = 1.0;
         z_factor = 1.0;
      }
      else { 
         x_factor = (extents.maxx-extents.minx);
         if (fabs(x_factor) < MM_FLAT_TOL) 
            x_factor = 1.0;
         y_factor = (extents.maxy-extents.miny);
         if (fabs(y_factor) < MM_FLAT_TOL) 
            y_factor = 1.0;
         z_factor = (extents.maxz-extents.minz);
         if (fabs(z_factor) < MM_FLAT_TOL) 
            z_factor = 1.0;
      }
      mm_atof(state.xsc, &axs, status, errmsg1);
      if (*status == MM_STATUS_ERROR) {
         sprintf(errmsg, "Error in Width Scale: %s", errmsg1);
         return;
      }
      else {
         *xs = (float) axs/x_factor;
      }
      mm_atof(state.ysc, &ays, status, errmsg1);
      if (*status == MM_STATUS_ERROR) {
          sprintf(errmsg, "Error in Depth Scale: %s", errmsg1);
         return;
      }
      else {
         *ys = (float) ays/y_factor;
      }
      mm_atof(state.zsc, &azs, status, errmsg1);
      if (*status == MM_STATUS_ERROR) {
         sprintf(errmsg, "Error in Height Scale: %s", errmsg1);
         return;
      }
      else {
         *zs = (float) azs/z_factor;
      }
   }
  /*
   * Coords conversions */
   switch(state.xrad1) {
      case 0: *x1 = extents.minx; break;
      case 1: *x1 = extents.centerx; break;
      case 2: *x1 = extents.maxx; break;
      case 3: mm_atof(state.xstr1, &dx1, status, errmsg1);
         *x1 = (float) dx1;
         if (*status == MM_STATUS_ERROR) {
            sprintf(errmsg, "Error in Width Source Pt: %s", errmsg1);
            return;
         }
         break;
   };
   switch(state.yrad1) {
      case 0: *y1 = extents.miny; break;
      case 1: *y1 = extents.centery; break;
      case 2: *y1 = extents.maxy; break;
      case 3: mm_atof(state.ystr1, &dy1, status, errmsg1);
         *y1 = (float) dy1;
         if (*status == MM_STATUS_ERROR) {
            sprintf(errmsg, "Error in Depth Source Pt: %s", errmsg1);
            return;
         }
         break;
   };
   switch(state.zrad1) {
      case 0: *z1 = extents.minz; break;
      case 1: *z1 = extents.centerz; break;
      case 2: *z1 = extents.maxz; break;
      case 3: mm_atof(state.zstr1, &dz1, status, errmsg1);
         *z1 = (float) dz1;
         if (*status == MM_STATUS_ERROR) {
            sprintf(errmsg, "Error in Height Source Pt: %s", errmsg1);
            return;
         }
         break;
   };
   switch(state.xrad2) {
      case 0: *x2 = extents.minx; break;
      case 1: *x2 = extents.centerx; break;
      case 2: *x2 = extents.maxx; break;
      case 3: mm_atof(state.xstr2, &dx2, status, errmsg1); 
         *x2 = (float) dx2;
         if (*status == MM_STATUS_ERROR) {
            sprintf(errmsg, "Error in Width Dest Pt: %s", errmsg1);
            return;
         }
         break;
   };
   switch(state.yrad2) {
      case 0: *y2 = extents.miny; break;
      case 1: *y2 = extents.centery; break;
      case 2: *y2 = extents.maxy; break;
      case 3: mm_atof(state.ystr2, &dy2, status, errmsg1); 
         *y2 = (float) dy2;
         if (*status == MM_STATUS_ERROR) {
            sprintf(errmsg, "Error in Depth Dest Pt: %s", errmsg1);
            return;
         }
         break;
   };
   switch(state.zrad2) {
      case 0: *z2 = extents.minz; break;
      case 1: *z2 = extents.centerz; break;
      case 2: *z2 = extents.maxz; break;
      case 3: mm_atof(state.zstr2, &dz2, status, errmsg1);
         *z2 = (float) dz2;
         if (*status == MM_STATUS_ERROR) {
            sprintf(errmsg, "Error in Height Dest Pt: %s", errmsg1);
            return;
         }
         break;
   };

   *status = MM_STATUS_OKAY;
}

/*===========================================================================
 | Function: translate_vlist
 |  Purpose: To translate a set of verticies passed and return with the
 |           list modified accordingly
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void mm_translate_vlist(v, vlen, xt, yt, zt)
VData *v;
int vlen;
float xt, yt, zt;
{
   int i;
   float x, y, z;

   for (i = 0; i < vlen; i++) {
      x = v->x+xt; v->x = x;
      y = v->y+yt; v->y = y;
      z = v->z+zt; v->z = z;
      v++;
   }
}

/*===========================================================================
 | Function: rotate_vlist
 |  Purpose: To rotate the verticies passed about the axis passed and 
 |           store the modified values back in the original list.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void mm_rotate_vlist(v, vlen, axis, ang)
VData *v;
int vlen, axis;
float ang;
{
   int i;
   float a, x, y, z;

   a = DegToRad(ang);

   for (i = 0; i < vlen; i++) {
      switch(axis) {
         case 0: /* X Axis */
            x = v->x;
            y = v->y*cos(a)-v->z*sin(a);
            z = v->y*sin(a)+v->z*cos(a);
            break;
         case 1: /* Y Axis */
            x = v->x*cos(a)+v->z*sin(a);
            y = v->y;
            z = -1*v->x*sin(a)+v->z*cos(a);
            break;
         case 2: /* Z Axis */
            x = v->x*cos(a)-v->y*sin(a);
            y = v->x*sin(a)+v->y*cos(a);
            z = v->z;
            break;
         default:
            break;
      };
      v->x = x; v->y = y; v->z = z;
      v++;
   }
}

/*===========================================================================
 | Function: scale_vlist
 |  Purpose: To scale a set of verticies by the scale factors passed.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void mm_scale_vlist(v, vlen, xs, ys, zs)
VData *v;
int vlen;
float xs, ys, zs;
{
   int i;
   float x, y, z;

   for (i = 0; i < vlen; i++) {
      x = v->x*xs; v->x = x;
      y = v->y*ys; v->y = y;
      z = v->z*zs; v->z = z;
      v++;
   }
}

/*===========================================================================
 | Function: mm_atof
 |  Purpose: To convert a series of character representing a distance 
 |           expressed in feet and inches and return a result as a decimal
 |           number of inches.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void mm_atof(char *s, double *result, int *status, char *errmsg)
{
   int l, have_feet, have_inch;
   char output[128], *o;
   double value, sign, feet, inch, conv_factor;
  /*
   * Initialize */
   feet = inch = 0.0;
   *status = MM_STATUS_OKAY; /* Innocent until proven guilty */
  /*
   * Perform a valid character check */
   if (mm_valid_str(s, ".0123456789+-/'\"") == 0) {
      *result = 0.0;
      *status = MM_STATUS_ERROR;
      strcpy(errmsg, "Invalid character in string");
      return;
   }
  /*
   * Handle the sign if present */
   if (s[0] == MM_FRAC_DELIM) {
      sign = -1.0;
      s++;
   }
   else if (s[0] == '+') {
      sign = 1.0;
      s++;
   }
   else {
      sign = 1.0;
   }
  /*
   * Scan up to the feet sign */
   o = output;
   if (mm_instr(s, MM_FEET_SYM)) {
      while (*s != MM_FEET_SYM)
	 *o++ = *s++;
      *o = '\0';
      s++; // Point past the ' sign
      have_feet = 1;
     /*
      * Convert and store this portion */
      mm_process_chunk(output, &feet, status, errmsg);
      if (*status == MM_STATUS_ERROR) {
	 return;
      }
   }
  /*
   * Check if there is any more of the string to process */
   if (strlen(s)) {
     /*
      * Scan up to the inch sign or end */
      o = output;
      if (mm_instr(s, MM_INCH_SYM)) have_inch = 1;
      while ((*s != MM_INCH_SYM) && (*s != MM_STR_TERM))
	 *o++ = *s++;
      *o = MM_STR_TERM;
     /*
      * Convert and store this portion */
      mm_process_chunk(output, &inch, status, errmsg);
      if (*status == MM_STATUS_ERROR) {
	 return;
      }
   }
  /*
   * If the number was entered as feet and inches, convert it if we
   * are in metric */
   if (have_feet || have_inch) {
      switch(master_type) {
         case 1: // '
            conv_factor = (1.0/12.0)*master_scale;
            break;
         case 2: // CM
            conv_factor = 2.54*master_scale;
            break;
         case 3: // M
            conv_factor = 0.0254*master_scale;
            break;
         default: // "
            conv_factor = 1.0*master_scale;
            break;
      }
   }
   else {
      conv_factor = 1.0;
   }
  /*
   * Compute and return the result */
   *status = MM_STATUS_OKAY;
   *result = (sign*(feet*12.0+inch))*conv_factor;
}

/*===========================================================================
 | Function: mm_process_chunk
 | Purpose:  To process the portion of the string passed (which was
 |           up to the feet or inches sign).  This function will break the
 |           string down further into numerator and denominator.
 |  History: Mark Meier, 03/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
void mm_process_chunk(char *s, double *result, int *status, char *errmsg)
{
   double value, whole, frac, num, den;
   char output[128], *o;
  /*
   * Initialize */
   whole = num = frac = 0.0; den = 1.0;
  /*
   * If this is a zero length string, return a value of 0.0 */
   if (strlen(s) == 0) {
      *result = 0.0;
      *status = MM_STATUS_OKAY;
      return;
   }
  /*
   * Determine if there is a whole number */
   if ((mm_instr(s, MM_FRAC_DELIM)  && mm_instr(s, MM_FRAC_SYM)) ||
       (mm_instr(s, MM_FRAC_SYM) == 0)) {
     /*
      * Process the whole number */
      o = output;
      while ((*s != MM_FRAC_DELIM) && (*s != MM_STR_TERM))
	 *o++ = *s++;
      *o = MM_STR_TERM;
      if (strlen(s))
	 s++; // Skip the delimeter
      whole = mm_convert_str(output); // Convert whole number portion
   }
  /*
   * Process any fraction */
   if (strlen(s)) {
      if ((mm_instr(s, MM_FRAC_SYM))) {
	 o = output;
	 while (*s != MM_FRAC_SYM)
	    *o++ = *s++;
	 *o = MM_STR_TERM;
	 s++; // Skip / sign
	 num = mm_convert_str(output); // Convert numerator
	 if (num == 0.0) {
	    strcpy(errmsg, "Fraction specified w/o a numerator");
	    *status = MM_STATUS_ERROR;
	    return;
	 }
	 o = output;
	 while (*s) // Put the remainder of the string in the denominator
	    *o++ = *s++;
	 *o = MM_STR_TERM;
	 if (strlen(output)) {
	    den = mm_convert_str(output); // Convert denominator
	    if (den != 0.0) {
	       frac = num/den;
	       *result = whole+frac;
	       *status = MM_STATUS_OKAY;
	       return; // Done, and all is well :)
	    }
	    else {
	       strcpy(errmsg, "Zero denominator in fraction");
	       *status = MM_STATUS_ERROR;
	       return;
	    }
	 }
	 else {
	    strcpy(errmsg, "Fraction specified w/o a denominator");
	    *status = MM_STATUS_ERROR;
	    return;
	 }
      }
      else {
	 strcpy(errmsg, "Fractional seperator specified w/o '/' delimiter");
	 *status = MM_STATUS_ERROR;
	 return;
      }
   }
   *result = whole;
   *status = MM_STATUS_OKAY;
}

/*===========================================================================
 | Function: mm_convert_str
 |  Purpose: The final step, convert a simple ASCII string to a double.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
double mm_convert_str(char *s)
{
   double value;
   sscanf(s, "%lf", &value);
   return (value);
}

/*===========================================================================
 | Function: mm_instr
 |  Purpose: To check if the character ch is contained in the string 's'.
 |           If so, 1 is returned, 0 otherwise.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
int mm_instr(char *s, char ch)
{
   int i;
   while (*s)
      if (*s++ == ch) return(1);
   return(0);
}

/*===========================================================================
 | Function: mm_valid_str
 |  Purpose: To verify if every character in 's' is in the list 'valid'.
 |           If so, 1 is returned, 0 otherwise.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: None.
 *=========================================================================*/
int mm_valid_str(char *s, char *valid)
{
   while (*s) {
      if (mm_instr(valid, s[0]) == 0) return(0);
      s++;
   }
   return(1);
}

/*===========================================================================
 | Function: mm_alloc_mem
 |  Purpose: To allocate the memory for the new object to be created.
 |  History: Mark Meier, 04/94, Wrote and documented routine.
 |  Remarks: This routine tries twice to malloc because of a bug in 
 |           Watcom 9.5.
 *=========================================================================*/
int mm_alloc_mem(EXPbuf *b, int vlen, int flen)
{
   if((vdata=(VData *)malloc(vlen*sizeof(VData)))==NULL)
   {
      if((vdata=(VData *)malloc(vlen*sizeof(VData)))==NULL)
      {
         return(MM_STATUS_ERROR);
      }
   }
   if((fdata=(FData *)malloc(flen*sizeof(FData)))==NULL)
   {
      if((fdata=(FData *)malloc(flen*sizeof(FData)))==NULL)
      {
         return(MM_STATUS_ERROR);
      }
   }
   return(MM_STATUS_OKAY);
}

void ipas_terminate(EXPbuf *b)
{
   b->opcode = b->usercode = EXP_TERMINATE;
   b->status = 0;
}
