/****************************************************************************
*
*                         The Universal VESA VBE
*
*                   Copyright (C) 1993 Kendall Bennett.
*                           All rights reserved.
*
* Filename:     $RCSfile: _paradis.c $
* Version:      $Revision: 1.2 $
*
* Language:     ANSI C
* Environment:  IBM PC (MS DOS)
*
* Description:  C language support routines for Paradise SuperVGA's. Contains
*               detection code, mode translation tables and chipset
*               names.
*
* $Id: _paradis.c 1.2 1993/09/24 05:22:30 kjb release $
*
* Revision History:
* -----------------
*
* $Log: _paradis.c $
* Revision 1.2  1993/09/24  05:22:30  kjb
* Fixed a number of bugs.
*
* Revision 1.1  1993/09/19  01:26:26  kjb
* Initial revision
*
****************************************************************************/

/* Mode translation table to determine the values to place into
 * AX and BX in order to initialise a particular video mode
 * via int 10h.
 */

short PARADModes[] = {  0x58, 0,        /* 800x600 16 color     */
                        0x5D, 0,        /* 1024x768 16 color    */
                        0x64, 0,        /* 1280x1024 16 color   */
                        0, 0,           /* 640x350 256 color    */
                        0x5E, 0,        /* 640x400 256 color    */
                        0x5F, 0,        /* 640x480 256 color    */
                        0x5C, 0,        /* 800x600 256 color    */
                        0x60, 0,        /* 1024x768 256 color   */
                        0, 0,           /* 1280x1024 256 color  */
                        0, 0,           /* 320x200 32k color    */
                        0, 0,           /* 640x350 32k color    */
						0x61, 0,        /* 640x400 32k color    */
                        0x62, 0,        /* 640x480 32k color    */
                        0x63, 0,        /* 800x600 32k color    */
                        0, 0,           /* 1024x768 32k color   */
                        0, 0,           /* 1280x1024 32k color  */
                        0, 0,           /* 320x200 64k color    */
                        0, 0,           /* 640x350 64k color    */
                        0, 0,           /* 640x400 64k color    */
                        0, 0,           /* 640x480 64k color    */
                        0, 0,           /* 800x600 64k color    */
                        0, 0,           /* 1024x768 64k color   */
                        0, 0,           /* 1280x1024 64k color  */
                        0, 0,           /* 320x200 16m color    */
                        0, 0,           /* 640x350 16m color    */
                        0, 0,           /* 640x400 16m color    */
                        0x72, 0,        /* 640x480 16m color    */
                        0, 0,           /* 800x600 16m color    */
                        0, 0,           /* 1024x768 16m color   */
                        0, 0,           /* 1280x1024 16m color  */
                        };

typedef enum {
    grPARA_PVGA1A,          /* Paradise PVGA1A chipset                  */
    grPARA_90C00,           /* Paradise WD90C00 chipset                 */
    grPARA_90C10,           /* Paradise WD90C10 chipset                 */
    grPARA_90C11,           /* Paradise WD90C11 chipset                 */
    grPARA_90C2x,           /* Paradise WD90C2x chipset                 */
    grPARA_90C30,           /* Paradise WD90C30 chipset                 */
    grPARA_90C31,           /* Paradise WD90C31 chipset                 */
    } PARA_chipsets;

/* Names of chip revision id's. */

char *ParadiseNames[] = {
    "PVGA1A",
    "WD90C00",
    "WD90C10",
    "WD90C11",
    "WD90C2x",
    "WD90C30",
    "WD90C31",
    };

bool findParadise(int *superVGA,int *chipID,int *memory,int *dac,int *pageFlip)
/****************************************************************************
*
* Function:     findParadise
* Parameters:   superVGA    - ID of the SuperVGA card
*               chipID      - Internal chip ID number
*               memory      - Amount of memory of card
*               dac         - Type of DAC installed
*               pageFlip    - True if page flipping supported
* Returns:      True if card was detected
*
* Description:  Detects the presence of Paradise/Western Digital based
*               SuperVGA's.
*
****************************************************************************/
{
    int         old;
    union REGS  regs;

    old = rdinx(GRC,0xF);
    modinx(GRC,0xF,0x17,0);         /* Lock paradise registers          */
    if (tstinx(GRC,9,0x7F))
        goto NoParadise;
    wrinx(GRC,0xF,5);               /* Unlock paradise registers        */
    if (!tstinx(GRC,9,0x7F))
        goto NoParadise;

    /* We have a Paradise SuperVGA, do determine the chipset type */

    *superVGA = grSVGA_PARADISE;
    *pageFlip = true;

    old = rdinx(CRTC,0x29);         /* Read value of WD90CXX lock       */
    modinx(CRTC,0x29,0x8F,0x85);    /* Unlock WD90Cxx registers         */
    if (!tstinx(CRTC,0x2B,0xFF))
        goto HavePVGA;
    old = rdinx(SEQ,6);             /* Read value of WD90C1x lock       */
    wrinx(SEQ,6,0x48);
    if (!tstinx(SEQ,7,0xF0))
        goto Have90C00;
    if (!tstinx(SEQ,0x10,0xFF))
        goto Have90C2x;
    if (!tstinx(SEQ,0x14,0xF))
        goto Have90C1x;
    if (rdinx(CRTC,0x37) == 0x31)
        *chipID = grPARA_90C31;
    else {
        *chipID = grPARA_90C30;
        unsupportedMode(vbe_1280x1024x16,PARADModes);
        }
    wrinx(SEQ,6,old);               /* Restore register value           */
    goto MemCheck;

Have90C00:
    wrinx(SEQ,6,old);               /* Restore register value           */
    *chipID = grPARA_90C00;
    unsupportedMode(vbe_1280x1024x16,PARADModes);
    unsupportedMode(vbe_1024x768x256,PARADModes);
    goto MemCheck;

Have90C1x:
    *chipID = grPARA_90C10;
    unsupportedMode(vbe_1280x1024x16,PARADModes);
    unsupportedMode(vbe_1024x768x256,PARADModes);
    if (tstinx(SEQ,0x10,4))
        *chipID = grPARA_90C11;
    wrinx(SEQ,6,old);               /* Restore register value           */
    goto MemCheck;

Have90C2x:
    wrinx(CRTC,0x34,0xA6);
    if (rdinx(CRTC,0x32) != 0)
        wrinx(CRTC,0x34,0);
    *chipID = grPARA_90C2x;
    unsupportedMode(vbe_1024x768x16,PARADModes);
    unsupportedMode(vbe_1280x1024x16,PARADModes);
    unsupportedMode(vbe_800x600x256,PARADModes);
    unsupportedMode(vbe_1024x768x256,PARADModes);
    goto MemCheck;

HavePVGA:
    wrinx(CRTC,0x29,old);           /* Restore WD90Cxx register lock    */
    *chipID = grPARA_PVGA1A;
    unsupportedMode(vbe_1024x768x16,PARADModes);
    unsupportedMode(vbe_1280x1024x16,PARADModes);
    unsupportedMode(vbe_800x600x256,PARADModes);
    unsupportedMode(vbe_1024x768x256,PARADModes);

MemCheck:
    old = rdinx(GRC,0xB) >> 6;
    switch (old) {
        case 1:     *memory = 256;  break;
        case 2:     *memory = 512;  break;
        default:    *memory = 1024;
        }

/* Ok, now check to see if the card is capable of 800x600x16 color mode. If
 * it isn't, then the card is probably running on an LCD display, so we
 * disable all modes above 640x480.
 */

    saveMode();
    regs.x.ax = 0x58 | 0x80;
    int86(0x10,&regs,&regs);
    if ((peekb(0x40,0x49) & 0x7F) != 0x58) {
        unsupportedMode(vbe_800x600x16,PARADModes);
        unsupportedMode(vbe_1024x768x16,PARADModes);
        unsupportedMode(vbe_1280x1024x16,PARADModes);
        unsupportedMode(vbe_800x600x256,PARADModes);
        unsupportedMode(vbe_1024x768x256,PARADModes);
        unsupportedMode(vbe_800x600x32k,PARADModes);
        }
    restoreMode();
    return true;

NoParadise:
    wrinx(GRC,0xF,old);             /* Restore old register             */
    return false;
}
