/****************************************************************************
*
*                         The Universal VESA VBE
*
*                   Copyright (C) 1993 Kendall Bennett.
*                           All rights reserved.
*
* Filename:     $RCSfile: _tseng.c $
* Version:      $Revision: 1.1 $
*
* Language:     ANSI C
* Environment:  IBM PC (MS DOS)
*
* Description:  C language support routines for Tseng Labs SuperVGA's.
*               Contains detection code, mode translation tables and chipset
*               names.
*
* $Id: _tseng.c 1.1 1993/09/19 01:26:40 kjb release $
*
* Revision History:
* -----------------
*
* $Log: _tseng.c $
* Revision 1.1  1993/09/19  01:26:40  kjb
* Initial revision
*
****************************************************************************/

/* Mode translation table to determine the values to place into
 * AX and BX in order to initialise a particular video mode
 * via int 10h.
 */

short TSENGModes[] = {  0x29, 0,        /* 800x600 16 color     */
                        0x37, 0,        /* 1024x768 16 color    */
                        0x3D, 0,        /* 1280x1024 16 color   */
                        0x2D, 0,        /* 640x350 256 color    */
                        0x2F, 0,        /* 640x400 256 color    */
                        0x2E, 0,        /* 640x480 256 color    */
                        0x30, 0,        /* 800x600 256 color    */
                        0x38, 0,        /* 1024x768 256 color   */
                        0, 0,           /* 1280x1024 256 color  */
                        0x10F0, 0x13,   /* 320x200 32k color    */
                        0x10F0, 0x2D,   /* 640x350 32k color    */
                        0x10F0, 0x2F,   /* 640x400 32k color    */
                        0x10F0, 0x2E,   /* 640x480 32k color    */
                        0x10F0, 0x30,   /* 800x600 32k color    */
                        0, 0,           /* 1024x768 32k color   */
                        0, 0,           /* 1280x1024 32k color  */
                        0x10F0, 0x13,   /* 320x200 64k color    */
                        0x10F0, 0x2D,   /* 640x350 64k color    */
                        0x10F0, 0x2F,   /* 640x400 64k color    */
                        0x10F0, 0x2E,   /* 640x480 64k color    */
                        0x10F0, 0x30,   /* 800x600 64k color    */
                        0, 0,           /* 1024x768 64k color   */
                        0, 0,           /* 1280x1024 64k color  */
                        0, 0,           /* 320x200 16m color    */
                        0x10F0, 0x2DFF, /* 640x350 16m color    */
                        0x10F0, 0x2FFF, /* 640x400 16m color    */
                        0x10F0, 0x2EFF, /* 640x480 16m color    */
                        0, 0,           /* 800x600 16m color    */
                        0, 0,           /* 1024x768 16m color   */
                        0, 0,           /* 1280x1024 16m color  */
                        };

typedef enum {
    grET3000,               /* ET3000 chipset                           */
    grET4000,               /* Standard Tseng MegaEva/2 ET4000 board    */
    grET4000_SPEEDSTAR,     /* Diamond Speedstar 24 board (24 bit)      */
    grET4000_GENOA,         /* Genoa 7900 ET4000 board (24 bit)         */
    grET4000_W32,           /* ET4000 W32 accelerator chipset           */
    } ET4000_chipsets;

/* Names of chip revision id's. */

static char *TsengNames[] = {
    "ET3000",
    "ET4000",
    "ET4000 Speedstar 24",
    "ET4000 Genoa 7900",
    "ET4000/W32",
    };

bool findTseng(int *superVGA,int *chipID,int *memory,int *dac,int *pageFlip)
/****************************************************************************
*
* Function:     findTseng
* Parameters:   superVGA    - ID of the SuperVGA card
*               chipID      - Internal chip ID number
*               memory      - Amount of memory of card
*               dac         - Type of DAC installed
*               pageFlip    - True if page flipping supported
* Returns:      True if card was detected
*
* Description:  Detects the presence of Tseng Labs based SuperVGA's.
*
****************************************************************************/
{
    union REGS  regs;
    int         membits,old,seg;

    outp(0x3BF,0x03);
    outp(0x3D8,0xA0);               /* Enable ET4000 extensions         */
    if (!tstreg(0x3CD,0x3F))        /* Test bank switch register        */
        return false;               /* No bank register - not Tseng     */

    /* We have a Tseng Labs SuperVGA, so determine if it is an ET3000
     * or an ET4000 based SuperVGA.
     */

    *superVGA = grSVGA_TSENG;
    *pageFlip = true;
    if (tstinx(CRTC,0x33,0x0F)) {   /* Test for ET4000 extended start   */
        /* We have an ET4000 based SuperVGA */

        if (tstinx(0x3CB,0x33,0xFF)) {
            /* We have an ET4000/W32 accelerator chipset */

            *chipID = grET4000_W32;
            membits = rdinx(CRTC,0x37) & 0xB;
            switch (membits) {
                case 0x9:   *memory = 256;  break;
                case 0xA:   *memory = 512;  break;
                case 0xB:   *memory = 1024; break;
                case 0x0:
                case 0x2:   *memory = 2048; break;
                default:    *memory = 4096; break;
                }
            }
        else {
            /* Normal ET4000AX chipset  */

            *chipID = grET4000;
            membits = rdinx(CRTC,0x37) & 0xB;
            switch (membits) {
                case 0xA:   *memory = 512;  break;
                case 0xB:   *memory = 1024; break;
                default:    *memory = 256;  break;
                }
            }

        if (*dac == grTCDAC) {
            /* We have a TrueColor DAC on board, so we need to determine
             * what type of board it is, a MegaEva board, Speedstar 24
             * or Genoa 7900 which all use different video mode numbers
             * to set the 24 bit modes.
             */

            saveMode();
            regs.x.ax = 0x10E0;
            regs.x.bx = 0x2E | 0x80;
            int86(0x10,&regs,&regs);
            if (regs.x.ax == 0x0010) {
                /* Board is a Diamond Speedstar 24 */

                unsupportedMode(vbe_640x350x16m,TSENGModes);
                unsupportedMode(vbe_640x400x16m,TSENGModes);
                TSENGModes[52] = 0x10E0;    /* Correct mode numbers     */
                TSENGModes[53] = 0x2E;

                setBytesPerLine(vbe_640x480x16m,2048);
                *chipID = grET4000_SPEEDSTAR;
                }
            else {
                regs.x.ax = 0x10F0;
                regs.x.bx = 0x3E | 0x80;
                int86(0x10,&regs,&regs);
                if (regs.x.ax == 0x0010) {
                    /* Board is a Genoa 7900 */

                    unsupportedMode(vbe_640x350x16m,TSENGModes);
                    unsupportedMode(vbe_640x400x16m,TSENGModes);
                    TSENGModes[52] = 0x10F0;    /* Correct mode numbers */
                    TSENGModes[53] = 0x3E;

                    setBytesPerLine(vbe_640x480x16m,2048);
                    *chipID = grET4000_GENOA;
                    }
                }
            restoreMode();
            }
        }
    else {
        /* We have an ET3000 based SuperVGA, so find the amount of
         * memory installed on the card, and remove all invalid modes
         * from mode table.
         */

        unsupportedMode(vbe_1280x1024x16,TSENGModes);
        unsupportedMode(vbe_640x400x256,TSENGModes);
        unsupportedMode(vbe_1024x768x256,TSENGModes);
        *chipID = grET3000;

        /* Detect the memory size */

        saveMode();
        regs.x.ax = 0x13;
        int86(0x10,&regs,&regs);    /* Set 320x200x256                  */
        inp(0x3DA);
        outp(0x3C0,(old = rdinx(0x3C0,0x36)) | 16);
        switch ((rdinx(GRC,0x6) >> 2) & 3) {
            case 0:
            case 1: seg = 0xA000;   break;
            case 2: seg = 0xB000;   break;
            case 3: seg = 0xB800;   break;
            }

        poke(seg,1,0x5678);
        poke(seg,3,0x1234);
        if (peek(seg,2) == 0x3456)
            *memory = 512;
        else *memory = 256;

        wrinx(0x3C0,0x36,old);      /* Restore value and renable DAC    */
        restoreMode();
        }

    return true;
}
