/*****************************************************************************
 SCROLL.C

 This is a simple demo of how to use VGL to create scrolling tile effects.
 It loads a 320x200 "map" and allows you to pan around it using the arrow
 keys.  The map is stored as a GIF file.  The map is lame.  I tried, but I
 just didn't have the time nor talent to create an "interesting" and detailed
 map.  Sorry...

 Once you run this program you can use the arrow keys to scroll around.  It
 makes use of the VGLKEY.C module, so you can try holding down combinations
 of keys to see it in action.  The '+' and '-' keys will increase (double)
 and decrease (half) the scrolling "speed".  The speed is actually how many
 pixels we scroll the tiles per frame.  Pressing 'T' will toggle the display
 of some text that shows the current X, Y, and Speed.  Pressing 'C' will
 toggle the cycling of part of the palette (the colors that the water tile
 happens to use).  Press ESCAPE to exit.

 NOTE: Because displaying text is quite time-consuming, the FPS rate will
       likely slow down quite a bit when the text flag is set.  Play with
       it a bit to see how it effects the FPS on your machine.
       For the record, I get 70+ FPS on my 486DX/33 when no text is displayed.
       I get around 45 FPS when the text is turned on.  On my 486DX/50 I
       get 70+ FPS either way.

 Finally: this is a demo.  It is not intended to demonstrate the best method
 of doing a tile based game.  It is simply meant to show *one* possible
 method.  I'm sure you'll think of others.  Hopefully this'll get your
 imagination going.

 Mark
 morley@camosun.bc.ca
*****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include "keys.h"
#include "vgl.h"

#define MAPW            320             /* Width of the map in tiles        */
#define MAPH            200             /* Height of the map in tiles       */
#define MAXTILES        32              /* Max number of tiles allowed      */

char far  VS[64000];                    /* Out virtual screen               */
char far  Tile[MAXTILES][256];          /* Array of tile bitmaps            */
char far  Map[MAPH][MAPW];              /* The map data                     */
char far  Pal[768];                     /* Our palette                      */

Intro()
{
  /* Black the palette */
   vglBlack();

   /* Load in a font */
   vglLoadFont( "fonts\\tuxedo.fon" );

   /* Set text color to bright red */
   vglTextColor( 12 );

   /* Position the "cursor" */
   vglGotoXY( 14, 110 );

   /* Display a string */
   vglPuts( "VGL Scrolling Demo" );

   /* Fade in */
   vglFadeIn( Pal );

   /* Delay a bit */
   sleep( 1 );

   /* Fade out */
   vglFadeOut( Pal );

   /* Clear the screen */
   vglClear( 0 );

   /* Set text color to bright purple */
   vglTextColor( 13 );

   /* Position the "cursor" */
   vglGotoXY( 38, 110 );

   /* Display some text */
   vglPuts( "By Mark Morley" );

   /* Fade in */
   vglFadeIn( Pal );

   /* Delay a bit */
   sleep( 1 );

   /* Fade out */
   vglFadeOut( Pal );
}

main()
{
   int  cx, cy;                         /* Current X and Y position (tile)  */
   int  ox, oy;                         /* Offset within the current tile   */
   int  x, y;                           /* Used in for loops                */
   int  key;                            /* To get keypresses                */
   int  ok = 1;                         /* Loop until ok==0                 */
   int  s = 2;                          /* Scroll speed (2 pixels per frame)*/
   long t1, t2, n;                      /* For calculating FPS              */
   int  text = 0;                       /* Text display flag                */
   int  c = 0;                          /* Counter for cycling              */
   int  cycle = 0;                      /* Cycle flag                       */
   int  i;                              /* Miscellaneous variable           */
   char buf[20];                        /* For formatting text              */

   /* Enter mode 13h */
   vglInit();

   /* Load in the map data */
   vglGif( "map.gif", (char far*) Map, 0, 0, 0 );

   /* Load a couple tiles.  You can load additonal tiles here.  Color 0 in
      the map represents tile number 0.  Color 1 in the map represents tile
      number 1, and so on.  If you load, say, a brick wall in tile number 2,
      then anywhere color 2 appears in the map you'll see a brick wall. */
   vglGif( "water.gif", Tile[0], Pal, 0, 0 );
   vglGif( "grass.gif", Tile[1], Pal, 0, 0 );

   /* Display a little intro screen */
   Intro();

   /* Load and set up the font characteristics */
   vglLoadFont( "fonts\\comix.fon" );
   vglShadowColor( 1 );
   vglShadowOn( 1 );
   vglTextColor( 15 );

   /* Display the "game" screen directly to video memory */
   vglGif( "scroll.gif", VIDMEM, 0, 0, 0 );

   /* We'll start at the top left corner of the map */
   cx = 0;
   cy = 0;
   ox = 0;
   oy = 0;

   /* Frame count starts at 0 */
   n = 0;

   /* We'll draw into our virtual screen */
   vglBuffer( VS );

   /* Display the initial tiles (only so there's something to fade in to) */
   for( x = 0; x < 11; x++ )
      for( y = 0; y < 11; y++ )
         vglPut( (x << 4) + 15 - ox, (y << 4) + 15 - oy, 16, 16, Tile[Map[cy + y][cx + x]] );
   vglCopyW( 16, 16, 160, 160 );

   /* Fade in from black */
   vglFadeIn( Pal );

   /* Enable the trapping of keys */
   vglTrapKeys();

   t1 = time( 0 );

   /* Loop until we're done */
   while( ok )
   {
      /* Increment the frame count */
      n++;

      /* Update the real screen */
      vglCopyW( 16, 16, 160, 160 );

      /* Draw the tiles */
      for( x = 0; x < 11; x++ )
         for( y = 0; y < 11; y++ )
            vglPut( (x << 4) + 15 - ox, (y << 4) + 15 - oy, 16, 16, Tile[Map[cy + y][cx + x]] );

      /* If the text flag is set, display some coordinate info */
      if( text )
      {
         sprintf( buf, "X=%d Y=%d S=%d", (cx << 4) + ox, (cy << 4) + oy, s );
         vglGotoXY( 18, 25 );
         vglPuts( buf );
      }

      /* If a key is hit, check it out */
      if( kbhit() )
      {
         if( (key = getch()) == 0 )
            key = 256 * getch();
         switch( key )
         {
            case Escape            : ok = 0;
                                     break;
            case 't'               :
            case 'T'               : text = 1 - text;
                                     break;
            case 'c'               :
            case 'C'               : cycle = 1 - cycle;
                                     break;
            case '+'               : if( s < 16 )
                                        s <<= 1;
                                     break;
            case '-'               : if( s > 1 )
                                        s >>= 1;
                                     break;
         }
      }

      /* Check the arrow key status bytes and change the cx,cy and ox,oy
         variables as appropriate */
      if( vglKeyStatus[0] )
      {
         for( i = 0; i < s; i++ )
         {
            if( oy )
               oy--;
            else if( cy  )
            {
               oy = 15;
               cy--;
            }
         }
      }
      if( vglKeyStatus[1] )
      {
         for( i = 0; i < s; i++ )
         {
            if( oy < 15 )
               oy++;
            else if( cy < MAPH - 11 )
            {
               oy = 0;
               cy++;
            }
         }
      }
      if( vglKeyStatus[2] )
      {
         for( i = 0; i < s; i++ )
         {
            if( ox )
               ox--;
            else if( cx )
            {
               ox = 15;
               cx--;
            }
         }
      }
      if( vglKeyStatus[3] )
      {
         for( i = 0; i < s; i++ )
         {
            if( ox < 15 )
               ox++;
            else if( cx < MAPW - 11 )
            {
               ox = 0;
               cx++;
            }
         }
      }

      /* If the cycle flag is set, cycle the palette */
      if( cycle && ++c == 8 )
      {
         c = 0;
         vglPartCycleR( 232, 8, &Pal[232 * 3] );
      }
   }
   t2 = time( 0 );

   /* Stop trapping the arrow keys */
   vglReleaseKeys();

   /* Fade out to black */
   vglFadeOut( Pal );

   /* Return to text mode */
   vglTerm();

   /* Display some stats */
   printf( "%ld seconds, %ld frames (%ld FPS)\n", t2 - t1, n, n / (t2 - t1 ) );

   return;
}
