/*******************************************************************************
 VGLPAL.C

 These routines are used to manipulate a palette.  Routines to fade in and out
 and cycle palette entries are included.

 Mark Morley
 morley@camosun.bc.ca
*******************************************************************************/

#include "vgl.h"

static char far Pal[768];

/*******************************************************************************
 Sets the active palette to all black.  Useful for drawing a screen and then
 fading in later.
*******************************************************************************/
vglBlack()
{
   memset( Pal, 0, 768 );
   vglSetPal( Pal );
   return;
}

/*******************************************************************************
 Fades in from black to the specified palette.
*******************************************************************************/
vglFadeIn( char far* palette )
{
   int i, j, v;

   for( i = 0; i < 32; i++ )
   {
      for( j = 0; j < 768; j++ )
         Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      vglSetPal( Pal );
   }
   return;
}

/*******************************************************************************
 Fades to black from a specified palette.
*******************************************************************************/
vglFadeOut( char far* palette )
{
   int i, j, v;

   for( i = 31; i >= 0; i-- )
   {
      for( j = 0; j < 768; j++ )
         Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      vglSetPal( Pal );
   }
   return;
}

/*******************************************************************************
 Fades in only a part of the palette.
*******************************************************************************/
vglPartFadeIn( int first, int num, char far* palette )
{
   int i, j, v;

   for( i = 0; i < 32; i++ )
   {
      for( j = 0; j < num * 3; j++ )
         Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      vglSetPartPal( first, num, Pal );
   }
   return;
}

/*******************************************************************************
 Fades out only a part of the palette.
*******************************************************************************/
vglPartFadeOut( int first, int num, char far* palette )
{
   int i, j, v;

   for( i = 31; i >= 0; i-- )
   {
      for( j = 0; j < num * 3; j++ )
         Pal[j] =  (i * (unsigned)palette[j]) >> 5;
      vglSetPartPal( first, num, Pal );
   }
   return;
}

/*******************************************************************************
 Cycles 'num' palette entries, starting at the 'first' entry.  Entries are
 cycled to the "left".
*******************************************************************************/
vglPartCycleL( int first, int num, char far* palette )
{
   int i, a, b, c;

   num--;
   a = palette[0];
   b = palette[1];
   c = palette[2];
   for( i = 0; i < num * 3; i++ )
      palette[i] = palette[i + 3];
   palette[i] = a;
   palette[i + 1] = b;
   palette[i + 2] = c;
   vglSetPartPal( first, num + 1, palette );
   return;
}

/*******************************************************************************
 Cycles 'num' palette entries, starting at the 'first' entry.  Entries are
 cycled to the "right".
*******************************************************************************/
vglPartCycleR( int first, int num, char far* palette )
{
   int i, a, b, c;

   i = (num * 3) - 1;
   a = palette[i - 2];
   b = palette[i - 1];
   c = palette[i];
   for( ; i >= 3; i-- )
      palette[i] = palette[i - 3];
   palette[0] = a;
   palette[1] = b;
   palette[2] = c;
   vglSetPartPal( first, num, palette );
   return;
}
