/*																									*/
/*	RDSGEN.C - Random-Dot Stereogram Generator										*/
/*																									*/
/*	Written by: Fred Feucht																	*/
/*																									*/
/*																									*/
/*	This program will take a type 2 TARGA or GIF file as input and 			*/
/*	generate a type 3 TARGA or GIF output file as an RDS.							*/
/*																									*/
/*	The original code comes from RDS.C by Alexander R. Enzmann, which I 		*/
/* found	to be too cumbersome.  The RDS algorithm is a modification of		*/
/*	the routine documented by Paul McMahon in "The Emporors New Clothes".	*/
/* With the additions of GIF encoding and decoding routines from TGA2GIF	*/
/* and POVRay respectively (authors are mentioned below).						*/
/*																									*/
/*	My thanks to all the previous authors whose code fragments appear 		*/
/*	herein.  I have given credit wherever possible to original authors.		*/
/* To those not specifically mentioned, I am grateful as well.					*/
/*																									*/
/* Permission is given by the author to freely redistribute and include		*/
/* this code in any program as long as this credit is given where due.		*/
/*																									*/
/* GIF and 'Graphics Interchange Format' are trademarks (tm) of				*/
/* Compuserve, Incorporated, an H&R Block Company.									*/
/*																									*/

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <math.h>
#include <ctype.h>
#include <graphics.h>

#include "rdsgen.h"

/*							*/
/* Global variables	*/
/*							*/

static int					stripc, density, direction, outfile, display, seed, adapt, index_points;
static int					stripw, intype, bad_code_count, code_size, point_max;
static int					fc, oc;
static long					limit, zmax, zmin, newval, oldval;
static unsigned int		xres, yres;

static unsigned int *	depths;
static unsigned char *	odata;
static unsigned char *	emit_index;

static int 					Width, Height;
static int 					curx, cury;
static long 				CountDown;
static int 					Pass = 0;
static int 					Interlace;

static FILE *				ofile;
static char *				oname;

static FILE *				ifile;
static char *				iname;

static int				 	pal[256];
static unsigned char		idbuf[256];
static unsigned char * 	cmap;
static unsigned int		ftype, idlen, cmlen, cmsiz, psize;

static int n_bits;                        /* number of bits/code */
static int maxbits = BITS;                /* user settable max # bits/code */
static code_int maxcode;                  /* maximum code, given n_bits */
static code_int maxmaxcode = (code_int)1 << BITS; /* should NEVER generate this code */
static count_int htab [HSIZE];
static unsigned short codetab [HSIZE];

static code_int free_ent = 0;                  /* first unused entry */
static int exit_stat = 0;

static int clear_flg = 0;

static int offset;
static long int in_count = 1;            	/* length of input */
static long int out_count = 0;           	/* # of codes output (for debugging) */

static int g_init_bits;
static FILE *g_outfile;

static int ClearCode;
static int EOFCode;

static code_int 			hsize = HSIZE; 	/* for dynamic table sizing */
static count_int 			fsize;

static unsigned long 	cur_accum = 0;
static int  				cur_bits = 0;

LOCAL WORD curr_size;                     /* The current code size */
LOCAL WORD clear;                         /* Value for a clear code */
LOCAL WORD ending;                        /* Value for a ending code */
LOCAL WORD newcodes;                      /* First available code */
LOCAL WORD top_slot;                      /* Highest code for current size */
LOCAL WORD slot;                          /* Last read code */

LOCAL WORD navail_bytes = 0;              /* # bytes left in block */
LOCAL WORD nbits_left = 0;                /* # bits left in current byte */
LOCAL UTINY b1;                           /* Current byte */
LOCAL UTINY byte_buff[257];               /* Current block */
LOCAL UTINY *pbytes;                      /* Pointer to next byte in block */

LOCAL UTINY *dstack;      						/* Stack for storing pixels */
LOCAL UTINY *suffix;      						/* Suffix table */
LOCAL UWORD *prefix;      						/* Prefix linked list */

LOCAL ULONG code_mask[17] = {
  0,
  0x0001, 0x0003,
  0x0007, 0x000F,
  0x001F, 0x003F,
  0x007F, 0x00FF,
  0x01FF, 0x03FF,
  0x07FF, 0x0FFF,
  0x1FFF, 0x3FFF,
  0x7FFF, 0xFFFF
  };

const char GIFstr[] = "GIF87a";


/*                       						*/
/* Main processing routine 					*/
/*                       						*/

void main(argc, argv)

int argc;
char **argv;

{
	int	i;

	int	driver, mode, errcode;



	stripc = 10;				/* strip length = 1/10 of screen width */

   density = 50;				/* density 50 out of 100 pixels black	*/

	direction = 0;				/* picture recedes from the viewer		*/ 

	outfile = 0;				/* output file is GIF format				*/

	display = 0;				/* do not display output while working */

	seed = 1;					/* random number seed						*/

	adapt = 0;					/* no background substitution				*/

	index_points = 0;			/* no indexing triangles in RDS output */
	
	point_max = 0;				/* size of index points						*/
	
	newval = 0xFFFF;			/* adaptive background replacement 		*/

	if (argc < 3)
	{
		showparms();

		exit(0);
	}

	iname = argv[1];
   oname = argv[2];

	for(i=3; i<argc; i++)
	{
		if(argv[i][0] == '-' || argv[i][0] == '/')
		{
			switch (argv[i][1])
			{
				case	'?':	showparms();
								break;
				case	'a':
				case	'A':	if(argv[i][2] == '\0') error("Missing adaptive background substitution value\n");
								adapt = atoi(argv[i]+2);
								if(adapt < 1 || adapt > 64) error("Adaptive background offset, %i, is out of range\n", adapt);
								break;
				case	'd':
				case	'D':	if(argv[i][2] == '\0') error("Missing density value\n");
								density = atoi(argv[i]+2);
								if(density < 0 || density > 100) error("Density value, %i, is out of range\n", density);
								break;
				case	'g':
				case	'G':	outfile = 0;
								break;
				case	'i':
				case	'I':	index_points = 1;
								break;
				case	'n':
				case	'N':	direction = 0;
								break;
				case	'p':
				case	'P':	direction = 1;
								break;
			   case	'r':
				case	'R':	if(argv[i][2] == '\0') error("Missing random seed value\n");
								seed = atoi(argv[i]+2);
								break;
			   case	's':
				case	'S':	if(argv[i][2] == '\0') error("Missing strip count value\n");
								stripc = atoi(argv[i]+2);
								break;
				case	't':
				case	'T':	outfile = 1;
								break;
				case	'v':
				case	'V':	display = 1;
								break;
				default:		error("Unknown option: %s\n", argv[i]);
			}
		}
		else error("Invalid switch\n");

	}

	oldval = (direction) ? 1 : 0xFFFF; 

	ifile = fopen(iname, "rb");

	if (ifile == (FILE *)NULL)	error("Cannot open input file \"%s\".", iname);

	read_header_data(&xres, &yres);

	printf("xres: %d, yres: %d\n", xres, yres);
   
	point_max = yres >> 6;
	
	if(adapt) 
	{
		newval = zmin - ((zmax - zmin) / adapt);

		zmin = newval;
	}
	
	printf("zmin: %u, zmax: %u\n", (unsigned) zmin, (unsigned) zmax);

	if((stripc < 1) || (stripc > xres)) error("Strip count value, %i, is out of range\n", stripc);

	stripw = xres / stripc;

	limit = (long) density * RAND_MAX / 100;

	if((seed < 1) || (seed > RAND_MAX)) error("Random number seed value, %i, is out of range\n", seed);

	srand(seed);
	
	if ((depths= (unsigned int *)malloc(xres * sizeof(unsigned int))) == NULL) error("Failed to allocate data row array\n");

   if ((odata = malloc(xres * sizeof(unsigned char))) == NULL) error("Failed to allocate the output image buffer");

	if(outfile == 1)
	{
		printf("Generating TGA output file\n");
	}
	else
	{
		printf("Generating GIF output file\n");

		for(i=0; i<256; i++) pal[i] = i;
	}

   if(display)
	{
		switch(xres)
		{
			case 320:	if(yres != 200) error("Unable to display %i by %i image\n", xres, yres); 
							driver = CGA;
							mode = 1;	
							break;
			case 640:	if(yres == 480)
							{
								driver = VGA;
								mode = 2;
								break;
							}
							driver = EGA;
							if(yres == 350)
							{
								mode = 1;
								break;
							}
							if(yres == 200)
							{
								mode = 0;
								break;
							}
			default:		error("Unable to display %i by %i image\n", xres, yres); 
		}

		initgraph(&driver, &mode, 0);

		errcode = graphresult();

		if(errcode != grOk)
		{
			printf("%s\n", grapherrormsg(errcode));
			exit(0);
		}
	}

	if(outfile == 1)
	{
		TGAEncode(oname,						/* file name			*/
					 xres,yres);				/* image size			*/
	}
	else
	{
		GIFEncode(oname,				   	 /* file name 			*/
					 xres, yres,        	    /* image size 		*/
					 0,                      /* no interlace 		*/
					 0,                      /* bkg color 			*/
					 8,                      /* bits per pixel 	*/
					 pal, pal, pal,			 /* palette arrays 	*/
					 get_pix);					 /* pixel function 	*/
	
	}

	if(display == 0)
	{
		printf("\n");
	}
	else
	{
		beep();
	
		while(!kbhit());

		closegraph();
	}

	fclose(ifile);
}

/* 				                    														*/
/*  Generate a line of RDS from a line of depth data								*/
/*       		  						  														*/	
/*	 For first strip, just generate random dots										*/
/*																									*/
/*	 For all remaining dots, dot color = dot color of (previous + scale)		*/
/*	 Where...		previous = current offset - strip width						*/
/*						scale = strip width / depth value								*/
/*																									*/
/*																									*/


void gen_rds_line(int width)
{
	int				d, i, o, w;

	long           l, z;

   for (i=0;i<width;i++) odata[i] = 255;		/* white out output line					*/

	w = stripw - (stripw >> 2);					/* limit useable strip to 3/4 length	*/

	z = zmax - zmin + 1;								/* calculate useable portion of strip	*/

	if(z < w) z = w;

	/* for first strip, just generate random dots	*/
	
	for(i=0; i<stripw; i++) if(rand() < limit) odata[i] = 0;

	for(; i<width; i++)
	{
		l = depths[i] & 0xFFFF;						/* extract depth value			      		 	 	 */
				
		o = (((l - zmin) * w) / z);				/* scale to strip width									 */

		d = (i - stripw + o);						/* calculate displacement to corresponding pixel */
		
		odata[i] = odata[d];					 	   /* copy depth adjusted pixel from previous strip */
	}
}

/*						                     	  	*/
/* Open a grey-scale Targa for output.	 		*/
/*                					       		*/

void open_targa_file(char * oname, unsigned width, unsigned height)
{
   unsigned char tgaheader[18];

	memset(tgaheader, 0, 18);
   
	if ((ofile = fopen(oname, "wb")) == NULL)	error("Failed to open output file %s\n", oname);

	tgaheader[2] = 3;
   tgaheader[12] = (unsigned char) (width & 0xFF);
   tgaheader[13] = (unsigned char) ((width >> 8) & 0xFF);
   tgaheader[14] = (unsigned char) (height & 0xFF);
   tgaheader[15] = (unsigned char) ((height >> 8) & 0xFF);
   tgaheader[16] = 8;
   tgaheader[17] = 0x20;
   
	fwrite(tgaheader, 18, 1, ofile);
}

/*                       									  */
/* Write a line of greyscale points to a Targa file  */
/*										                       */

void write_targa_line(unsigned width)
{
   int i;

	for (i=0;i<width;i++) fputc(odata[i], ofile);
}

/*							                       	*/
/* Read and evaluate one unit of input data	*/
/*                   							   */

unsigned int get_value()
{
	int 				i, j, k;

	register int	r, g, b;

	unsigned int 	val;

	unsigned char	bytes[4];


	if(intype)
	{
		/*												*/
		/* get input data from GIF decoder	*/
		/*												*/

		if((k = decode_gif()) < 0) error("Error reading input file %s\n", iname);
	
		if(k == ending) return(ending);

		k = k & 0xFF;

		j = cmsiz * k;

		val = cmap[j++] << 8 | cmap[j];
	}
	else
	{
		/*												*/
		/* get input data from TGA file		*/
		/*												*/

		if (ftype == 2)
		{
			/*										  	*/
			/* input data is 8/16/24 bit rgb	*/
			/*										  	*/

			switch(psize)
			{
				case 1:	if((r = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);
							val = r << 8;
							break;

				case 2:	if((g = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);
							if((r = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);
               		val = r << 8 | g;
							break;

				case 3:	if((g = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);
							if((g = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);
							if((r = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);
               		val = r << 8 | g;
							break;

				default:	error("Invalid color map byte count, %i\n", psize);
			}
		}
		else
		{
			/*										  	*/
			/* input data is color map index	*/
			/*										  	*/

			if ((k = fgetc(ifile)) == EOF) error("Premature EOF reading input file: %s\n", iname);

			j = cmsiz * k;

			if (psize == 2)
			{
				val = cmap[j++] | (cmap[j] << 8);
      	}
      	else
			{
				val = cmap[++j] | (cmap[++j] << 8);
      	}
   	}
	}

	if(direction != 0) val = -val;

	return(val);
}


/*						                       			*/
/* Add index points to to of image					*/
/*                						     			*/

void add_index(int line)
{
	int	i, c, d;

	c = point_max - line;

	d = ((xres - stripw) >> 1) - c;

	for(i=0; i<c<<1; i++) odata[d+i] = 0;

	d = ((xres + stripw) >> 1) - c;

	for(i=0; i<c<<1; i++) odata[d+i] = 0;
}

/*						                       			*/
/* Read a line's worth of the input file 			*/
/*                						     			*/

void	read_data_line(int width)
{
	int				i, j, k;

	unsigned int 	val;

	unsigned char	bytes[4];


	for (i=0;i<width;i++)
	{
		val = get_value();
	
		if(intype) if(val == ending) error("Premature end of object reading input GIF file %s\n", iname);

		if(adapt) if(val == oldval) val = newval;

		depths[i] = val;
	}
}

/*          				           			*/
/* Read next entry for GIF output routine	*/
/*				                       			*/

int get_pix(int x, int y)
{
	int			i, val;

	static int	line = -1;	

	if(y != line)
	{
		line = y;
		
		if(display == 0) printf("Line: %d  \r", y);

		if(line > yres) error("Request for invalid input location [%i][%i]\n", x, y);

		read_data_line(xres);

		gen_rds_line(xres);

		if(index_points) if(y<point_max) add_index(y);

	   test_exit;

		if(display)
		{
			for(i=0; i<xres; i++)
			{
				if(odata[i] == 255) putpixel(i, y, WHITE);
			}
		}
	}

	val = odata[x];

	return(val);

}


/*          				           			*/
/* Read header portion of input file		*/
/*				                       			*/


void read_header_data(unsigned *xsize, unsigned *ysize)
{
	int				i, j, k, l;

	unsigned char 	header[18];

	unsigned char	bytes[4];

	int 				width, height;

	long				count, val, position;


	zmax = 0;
	zmin = 65536;

   if(fread(&header[0], 3, 1, ifile) != 1) error("Couldn't read header from input file: %s\n", iname);

	intype = strncmp((const char *) header, GIFstr, 3);

	intype = (intype) ? 0 : 1;

	if(intype)
	{
		if(fread(&header[3], 10, 1, ifile) != 1) error("Couldn't read GIF header from input file %s\n", iname);

		if(!(header[10] & 0x80)) error("Input file %s, has no color map\n", iname);

		cmlen 	= 1 << ((header[10] & 0x0F) + 1);
		cmsiz		= 3; 

   	if (cmlen > 0)
		{
      	if ((cmap = malloc(sizeof(unsigned char)*cmsiz*cmlen)) == NULL) error("Failed to allocate memory for color map\n");

			for (i=0;i<cmlen * cmsiz;i++)
			{
         	if ((k = fgetc(ifile)) == EOF) error("Premature EOF in color map\n");
         
				cmap[i] = (unsigned char)k;
      	}
   	}

		k = 0;

		while(k != EOF)
		{
			switch(k = fgetc(ifile))
			{
			 	case ';':	k = EOF;
								break;

				case '!':	if((k = fgetc(ifile)) == EOF) error("Premature EOF in input file, %s\n", iname);
								if((k = fgetc(ifile)) == EOF) error("Premature EOF in input file, %s\n", iname);
								for(i=k; i>0; i--)  if((k = fgetc(ifile)) == EOF) error("Premature EOF in input file, %s\n", iname);
								break;

				case ',':	if (fread(header, 9, 1, ifile) != 1) error("Couldn't read object header from input file: %s\n", iname);

								width	 = header[4] | header[5] << 8;
								height =	header[6] | header[7] << 8;
								
								if ((header[8] & 0x80) == 0x80)
								{
									i = 1 << ((header[10] & 0x0F) + 1);
									for(; i<0; i++) if((k = fgetc(ifile)) == EOF) error("Premature EOF in input file, %s\n", iname);
								}
								
								position = ftell(ifile);
								
								if((k = allocate_gif_stacks()) != 0) error("Unable to allocate stack space for GIF decoder\n");
	
								printf("scanning GIF input for depth ranges\n");
								
								if((val = init_gif_decoder()) != 0) error("Error %d, initializing GIF decoder\n", val);
								
								while(val != ending)
								{
									val = get_value();

									if((!adapt) || (val != oldval))
									{
										if(zmax < val) zmax = val;
										if(zmin > val) zmin = val;
									}
								}

								if(bad_code_count > 0) error("%d out of range codes encountered in GIF input file %s\n", bad_code_count, iname);
								
								fseek(ifile, position, SEEK_SET);
								
								if((k = init_gif_decoder()) != 0) error("Error %d, reinitializing GIF decoder\n", k);
	
								k = EOF;
								
								break;

				default:		error("Unknown GIF format byte - %c\n", k);
			}
		}
	}
	else
	{
		if(fread(&header[3], 15, 1, ifile) != 1) error("Couldn't read TGA header from input file %s\n", iname);
	 
		idlen  = header[0];
   	ftype  = header[2];
   	cmlen  = header[5] | header[6] << 8;
   	cmsiz  = header[7];
   	width  = header[12] | header[13] << 8;
   	height = header[14] | header[15] << 8;
   	psize  = header[16];	 

   	if (!((ftype == 1 && psize <= 8) ||	(ftype == 2 && (psize == 16 || psize == 24)) ||	(ftype == 3 && psize == 8)))
		{
			error("Unsupported Targa type: %d\n", ftype);
		}

		if (cmlen > 256) error("Can't handle color maps of length %u\n", cmlen);

		if (cmlen > 0 && (cmsiz != 16 && cmsiz != 24)) error("Color maps entries must be 16 or 24 bits, not: %d\n", cmsiz);
   	else cmsiz /= 8;

   	if (psize != 8 && psize != 16 && psize != 24) error("%d bits/pixel not supported in height field, must be 8/16/24\n", psize);
   	else psize /= 8;

   	if (idlen > 0 && fread(idbuf, idlen, 1, ifile) != 1) error("Reading identification field of %s\n", iname);

   	if (cmlen > 0)
		{
      	if ((cmap = malloc(sizeof(unsigned char)*cmsiz*cmlen)) == NULL) error("Failed to allocate memory for color map\n");

			for (i=0;i<cmlen * cmsiz;i++)
			{
         	if ((k = fgetc(ifile)) == EOF) error("Premature EOF in color map\n");
         
				cmap[i] = (unsigned char)k;
      	}
   	}

		if (ftype == 1 && cmap == NULL) error("Targa color map must have entries\n");

		position = ftell(ifile);

		printf("scanning TGA input for depth ranges\n");

		for(count= (long) width*height; count>0; count--)
		{
			val = get_value() & 0xFFFF;

			if((!adapt) || (val != oldval))
			{
				if(zmax < val) zmax = val;
				if(zmin > val) zmin = val;
			}
		}
	
		fseek(ifile, position, SEEK_SET);
	}

  	*xsize = width;
  	*ysize = height;
}


/* 				                    						*/
/*  Generate RDS from Targa depth data 				*/
/*       		  						  						*/	

void TGAEncode(char * oname, int width, int height)
{
   int 		i, j;

	open_targa_file(oname, width, height);

	for (i=height;i>0;i--)
	{
		if(display == 0) printf("Line: %d  \r", i);

		read_data_line(width);

		gen_rds_line(width);

		if(index_points) if(i<point_max) add_index(i);

		if(display)
		{
			for(j=0; j<width; j++)
			{
				if(odata[j] == 255) putpixel(j, height-i-1, WHITE);
			}
		}

		test_exit;
		
		write_targa_line(width);
	}

	fclose(ofile);
}

/*                       						*/
/* Display command line parameters			*/
/*                       						*/

void	showparms()
{
	printf("\nUsage: rds input.ext output.ext options\n\n");
	printf("options: -d###   : Density value, range 0 to 100 (default 50)\n");
	printf("         -g      : Generate GIF output file (default)\n");
	printf("         -i      : Add indexing triangles to top of RDS\n");
	printf("         -n      : Produce a normal image (default)\n");
	printf("         -p      : Produce a precedence reversed image\n");
	printf("         -r##### : Seed number for random generator, range 1-32767 (default 1)\n");
	printf("         -s###   : Strip count, range 1-width (default 10)\n");
	printf("         -t      : Generate type 3 TGA output file\n");
	printf("         -v      : Display image on video while processing\n\n");
}


/*                       					*/
/* Beep, beep									*/
/*                       					*/

void beep()
{
	nosound();
	sound(1300);
	delay(80);
	sound(1600);
	delay(80);
	sound(1900);
	delay(80);
	nosound();
}

/*                       					*/
/* Display error message and exit		*/
/*                       					*/

void error(char *format, ...)
{
   va_list parms;
   
	/* fg_term(); */
   
	fprintf(stdout, "ERROR: ");

	va_start(parms, format);
   vfprintf(stdout, format, parms);
   va_end(parms);
   
	if(display) closegraph();

	exit(1);
}

/*														*/
/* Get next byte of input for GIF decoder	*/
/* 													*/

int get_byte()
{
	int	b;
	
	if((b = fgetc(ifile)) != EOF) return(b);

	printf("Premature EOF in GIF input file %s\n", iname);

	return(b);
}

/*																									*/
/*	GIF decoding routine																		*/
/*																									*/
/* This routine was borrowed from POVRAY which in turn borrowed the code 	*/
/* from FRACTINT.  It appears here in a somewhat modified format to 			*/
/*	comform to the requirements of my output driven logic.  Nevertheless,   */
/* I must give give appropriate credit to the orignal authors.					*/
/*																									*/
/* Based on :    	DECODER.C - An LZW decoder for GIF								*/
/*						Written by Steven A. Bennett										*/
/*						As inspired by Steve Wilhite										*/
/*						And modified by Bert Tyler and Timothy Wegner				*/
/* 																								*/

/*														*/
/* get the next code from the GIF file		*/
/*														*/

WORD get_next_code()
  {
  WORD i, x;
  ULONG ret;

	if (nbits_left == 0)
  	{
  		if (navail_bytes <= 0)
   	{
			/* Out of bytes in current block, so read next block */

	      pbytes = byte_buff;
      
			if ((navail_bytes = get_byte()) < 0) return(navail_bytes);
      	else if (navail_bytes)
	      {		
   	   	for (i = 0; i < navail_bytes; ++i)
      	   {
         		if ((x = get_byte()) < 0) return(x);
          	
					byte_buff[i] = (UTINY) x;
				}
   	   }
      }
    	b1 = *pbytes++;
    
		nbits_left = 8;
    	--navail_bytes;
	}

  	ret = b1 >> (8 - nbits_left);
  	
	while (curr_size > nbits_left)
   {
   	if (navail_bytes <= 0)
      {
			/* Out of bytes in current block, so read next block */
      	
			pbytes = byte_buff;
      	
			if ((navail_bytes = get_byte()) < 0) return(navail_bytes);
      	else if (navail_bytes)
        	{
        		for (i = 0; i < navail_bytes; ++i)
          	{
          		if ((x = get_byte()) < 0) return(x);
          
					byte_buff[i] = (UTINY) x;
          	}
        	}
		}
    	
		b1 = *pbytes++;
    	
		ret |= b1 << nbits_left;
    	
		nbits_left += 8;
    	--navail_bytes;
	}
  	
	nbits_left -= curr_size;
  	
	ret &= code_mask[curr_size];
  	
	return((WORD)(ret));
}


/*									*/
/* Allocate stacks			*/
/*									*/

int allocate_gif_stacks() 
{
	dstack = (UTINY *) malloc((MAX_CODES + 1)*sizeof(UTINY));
  	suffix = (UTINY *) malloc((MAX_CODES + 1)*sizeof(UTINY));
  	prefix = (UWORD *) malloc((MAX_CODES + 1)*sizeof(UWORD));

	return(0);
}


/*									*/
/* Release stacks				*/
/*									*/

void release_gif_stacks() 
{
	free(dstack);
  	free(suffix);
  	free(prefix);
}

/*														*/
/* Initialize for decoding a new image    */
/*														*/

int	init_gif_decoder()
{
  int		i, ret;


  ret = 0;
  
  if ((code_size = get_byte()) < 0) return(code_size);

  if (code_size < 2 || 9 < code_size) return(BAD_CODE_SIZE);

  curr_size = code_size + 1;
  
  top_slot = 1 << curr_size;
  
  clear = 1 << code_size;
  
  ending = clear + 1;
  
  slot = newcodes = ending + 1;
  
  navail_bytes = nbits_left = 0;
  
  /* Initialize in case they forgot to put in a clear code.				*/
  /* (This shouldn't happen, but we'll try and decode it anyway...)	*/

  oc = fc = 0;

  bad_code_count = 0;

  /* Set up the stack pointer and decode buffer pointer */

  emit_index = dstack;

  for(i=0; i<MAX_CODES+1; i++) prefix[i] = 0;
  for(i=0; i<MAX_CODES+1; i++) suffix[i] = 0;

  return(ret);
}


/*													*/
/* Get next byte from GIF decoder		*/
/*													*/


int decode_gif()

{
  	register int	c;
	register int	code;

  	/* This is the main loop.  For each code we get we pass through the
    * linked list of prefix codes, pushing the corresponding "character" for
    * each code onto the stack.  When the list reaches a single "character"
    * we push that on the stack too, and then start unstacking each
    * character for output in the correct order.  Special handling is
    * included for the clear code, and the whole thing ends when we get
    * an ending code.
    */

   if(emit_index > dstack) return(*(--emit_index));

	while ((c = get_next_code()) != ending)
   {
   	/* If we had a file error, return without completing the decode	*/

		if(c < 0) 
		return(0);

    	/* If the code is a clear code, reinitialize all necessary items	*/

		if(c == clear)
      {
      	curr_size = code_size + 1;
      	slot = newcodes;
      	top_slot = 1 << curr_size;

      	/* Continue reading codes until we get a non-clear code */
         /* (Another unlikely, but possible case...)				  */

			while ((c = get_next_code()) == clear);

	      /* If we get an ending code immediately after a clear code	*/
         /* (Yet another unlikely case), then break out of the loop.	*/

			if(c == ending) break;

			/* Finally, if the code is beyond the range of already set			*/
			/* codes, (This one had better NOT happen...  I have no idea 		*/
			/*	what will result from this, but I doubt it will look good... 	*/
			/* then set it	to color zero													*/

			if(c >= slot) c = 0;

      	oc = fc = c;

      	/* And let us not forget to put the character into the buffer		*/

			return(c);
		}
    	else
    	{
	 		/* In this case, it's not a clear code or an ending code, so			*/
      	/* it must be a code code...  So we can now decode the code into		*/
      	/* a stack of character codes. (Clear as mud, right?)						*/

			code = c;

      	/* Here we go again with one of those off chances...  If, on the		*/
	      /* off chance, the code we got is beyond the range of those already	*/
   	   /* set up (Another thing which had better NOT happen...) we trick		*/
      	/* the decoder into thinking it actually got the last code read.		*/
	      /* (Hmmn... I'm not sure why this works...  But it does...)				*/

			if(code >= slot)
			{
   	   	if(code > slot) ++bad_code_count;
      	  	code = oc;
        		*emit_index++ = (UTINY) fc;
	      }

   	   /* Here we scan back along the linked list of prefixes, pushing		*/
     		/* helpless characters (ie. suffixes) onto the stack as we do so.		*/
       
	      while (code >= newcodes)
	      {
   	   	*emit_index++ = suffix[code];
      	  	code = prefix[code];
	      }

	      /* Push the last character on the stack, and set up the new				*/
   	   /* prefix and suffix, and if the required slot number is greater		*/
      	/* than that allowed by the current bit size, increase the bit			*/
	      /* size.  (NOTE - If we are all full, we *don't* save the new			*/
   	   /* suffix and prefix...  I'm not certain if this is correct...			*/
      	/* it might be more proper to overwrite the last code...					*/

	      *emit_index++ = (UTINY) code;

			if(slot < top_slot)
      	{
	      	fc = code;
   	     	suffix[slot] = (UTINY) fc;
      	  	prefix[slot++] = oc;
	        	oc = c;
   	   }
	      
			if(slot >= top_slot)
			{
				if(curr_size < 12)
      	   {
         		top_slot <<= 1;
          		++curr_size;
	         }
			}

	      /* Now that we've pushed the decoded string (in reverse order)			*/
   	   /* onto the stack, lets pop it off and return it a byte at a 			*/
      	/* time to the caller																*/
       
   	   if(emit_index > dstack) return(*(--emit_index));
		}
	}
	return(c);
}


/******************************************************************************
 *
 * GIF output routines
 *
 ******************************************************************************/

/*	Number of characters so far in this 'packet' */

static int a_count;


/*	Set up the 'byte output' routine */

void char_init()
{
	a_count = 0;
}

/*	Define the storage for the packet accumulator */

static char accum[ 256 ];

/*
 * Add a character to the end of the current packet, and if it is 254
 * characters, flush the packet to disk.
 */

void char_out( c )
int c;
{
	accum[ a_count++ ] = c;
	if( a_count >= 254 )
		flush_char();
}

/*
 * Flush the packet to disk, and reset the accumulator
 */

void	flush_char()
{
	if( a_count > 0 ) {
		fputc( a_count, g_outfile );
		fwrite( accum, 1, a_count, g_outfile );
		a_count = 0;
	}
}


/*													*/
/* Write out a word to the GIF file		*/
/*													*/

void Putword(int w, FILE *fp )
{
	fputc( w & 0xff, fp );
	fputc( (w / 256) & 0xff, fp );
}

/*
 * Bump the 'curx' and 'cury' to point to the next pixel
 */

static void BumpPixel()
{
	/*
	 * Bump the current X position
	 */
	curx++;

	/*
	 * If we are at the end of a scan line, set curx back to the beginning
	 * If we are interlaced, bump the cury to the appropriate spot,
	 * otherwise, just increment it.
	 */
	if( curx == Width ) {
		curx = 0;

	        if( !Interlace )
			cury++;
		else {
		     switch( Pass ) {

	               case 0:
        	          cury += 8;
                	  if( cury >= Height ) {
		  		Pass++;
				cury = 4;
		  	  }
                          break;

	               case 1:
        	          cury += 8;
                	  if( cury >= Height ) {
		  		Pass++;
				cury = 2;
		  	  }
			  break;

	               case 2:
	                  cury += 4;
	                  if( cury >= Height ) {
	                     Pass++;
	                     cury = 1;
	                  }
	                  break;

	               case 3:
	                  cury += 2;
	                  break;
			}
		}
	}
}

/*
 * Return the next pixel from the image
 */

int GIFNextPixel(getpixel)
ifunptr getpixel;
{
	int r;

	if( CountDown == 0 )
		return EOF;

	CountDown--;

	r = getpixel( curx, cury );

	BumpPixel();

	return r;
}

/* public */

void GIFEncode( FName, GWidth, GHeight, GInterlace, Background,
	   			 BitsPerPixel, Red, Green, Blue, GetPixel )

char *	FName;
int 		GWidth, GHeight;
int 		GInterlace;
int 		Background;
int 		BitsPerPixel;
int 		Red[], Green[], Blue[];
ifunptr 	GetPixel;

{
	FILE *fp;
	int B;
	int RWidth, RHeight;
	int LeftOfs, TopOfs;
	int Resolution;
	int ColorMapSize;
	int InitCodeSize;
	int i;

	Interlace = GInterlace;

	ColorMapSize = 1 << BitsPerPixel;

	RWidth = Width = GWidth;
	RHeight = Height = GHeight;
	LeftOfs = TopOfs = 0;

	Resolution = BitsPerPixel;

	/*
	 * Calculate number of bits we are expecting
	 */
	CountDown = (long)Width * (long)Height;

	/*
	 * Indicate which pass we are on (if interlace)
	 */
	Pass = 0;

	/*
	 * The initial code size
	 */
	if( BitsPerPixel <= 1 )
		InitCodeSize = 2;
	else
		InitCodeSize = BitsPerPixel;

	/*
	 * Set up the current x and y position
	 */
	curx = cury = 0;

	/*
	 * Open the GIF file for binary write
	 */
	fp = fopen( FName, "wb" );

	if( fp == (FILE *)0 ) {
		printf( "error: could not open output file\n" );
		exit(1);
	}

	/*
	 * Write the Magic header
	 */
	fwrite( "GIF87a", 1, 6, fp );

	/*
	 * Write out the screen width and height
	 */
	Putword( RWidth, fp );
	Putword( RHeight, fp );

	/*
	 * Indicate that there is a global colour map
	 */
	B = 0x80;	/* Yes, there is a color map */

	/*
	 * OR in the resolution
	 */
	B |= (Resolution - 1) << 5;

	/*
	 * OR in the Bits per Pixel
	 */
	B |= (BitsPerPixel - 1);

	/*
	 * Write it out
	 */
	fputc( B, fp );

	/*
	 * Write out the Background colour
	 */
	fputc( Background, fp );

	/*
	 * Byte of 0's (future expansion)
	 */
	fputc( 0, fp );

	/*
	 * Write out the Global Colour Map
	 */
     	for( i=0; i<ColorMapSize; i++ ) {
		fputc( Red[i], fp );
		fputc( Green[i], fp );
		fputc( Blue[i], fp );
	}

	/*
	 * Write an Image separator
	 */
	fputc( ',', fp );

	/*
	 * Write the Image header
	 */

	Putword( LeftOfs, fp );
	Putword( TopOfs, fp );
	Putword( Width, fp );
	Putword( Height, fp );

	/*
	 * Write out whether or not the image is interlaced
	 */
	if( Interlace )
		fputc( 0x40, fp );
	else
		fputc( 0x00, fp );

	/*
	 * Write out the initial code size
	 */
	fputc( InitCodeSize, fp );

	/*
	 * Go and actually compress the data
	 */
	compress( InitCodeSize+1, fp, GetPixel );

	/*
	 * Write out a Zero-length packet (to end the series)
	 */
	fputc( 0, fp );

	/*
	 * Write the GIF file terminator
	 */
	fputc( ';', fp );

	/*
	 * And close the file
	 */
	fclose(fp);

}

/*																									*/
/*	GIF encoding routine																		*/
/*																									*/
/* This routine was borrowed from TGA2GIF with very little modification 	*/
/*	I must give credit to the original authors for their highly useful code */
/*																									*/
/* Based on :    	GIFENCODE.C - An LZW encoder for GIF							*/
/*						Written by Steven B. Coy											*/
/*						From code by David Rowley											*/
/*						And modified by Drew Wells and Chris Cason					*/
/* 																								*/
/*
 * GIF Image compression - modified 'compress'
 *
 * Based on: compress.c - File compression ala IEEE Computer, June 1984.
 *
 * By Authors:  Spencer W. Thomas       (decvax!harpo!utah-cs!utah-gr!thomas)
 *              Jim McKie               (decvax!mcvax!jim)
 *              Steve Davies            (decvax!vax135!petsd!peora!srd)
 *              Ken Turkowski           (decvax!decwrl!turtlevax!ken)
 *              James A. Woods          (decvax!ihnp4!ames!jaw)
 *              Joe Orost               (decvax!vax135!petsd!joe)
 *
 */

/*
 * compress stdin to stdout
 *
 * Algorithm:  use open addressing double hashing (no chaining) on the
 * prefix code / next character combination.  We do a variant of Knuth's
 * algorithm D (vol. 3, sec. 6.4) along with G. Knott's relatively-prime
 * secondary probe.  Here, the modular division first probe is gives way
 * to a faster exclusive-or manipulation.  Also do block compression with
 * an adaptive reset, whereby the code table is cleared when the compression
 * ratio decreases, but after the table fills.  The variable-length output
 * codes are re-sized at this point, and a special CLEAR code is generated
 * for the decompressor.  Late addition:  construct the table according to
 * file size for noticeable speed improvement on small files.  Please direct
 * questions about this implementation to ames!jaw.
 */

compress( init_bits, outfile, ReadValue )
int init_bits;
FILE *outfile;
ifunptr ReadValue;
{
    register long fcode;
    register code_int i = 0;
    register int c;
    register code_int ent;
    register code_int disp;
    register code_int hsize_reg;
    register int hshift;

    /*
     * Set up the globals:  g_init_bits - initial number of bits
     *                      g_outfile   - pointer to output file
     */
    g_init_bits = init_bits;
    g_outfile = outfile;

    /*
     * Set up the necessary values
     */
    offset = 0;
    out_count = 0;
    clear_flg = 0;
    in_count = 1;
    maxcode = MAXCODE(n_bits = g_init_bits);

    ClearCode = (1 << (init_bits - 1));
    EOFCode = ClearCode + 1;
    free_ent = ClearCode + 2;

    char_init();

    ent = GIFNextPixel( ReadValue );

    hshift = 0;
    for ( fcode = (long) hsize;  fcode < 65536L; fcode *= 2L )
	hshift++;
    hshift = 8 - hshift;                /* set hash code range bound */

    hsize_reg = hsize;
    cl_hash( (count_int) hsize_reg);            /* clear hash table */

    output( (code_int)ClearCode );

#ifdef SIGNED_COMPARE_SLOW
    while ( (c = GIFNextPixel( ReadValue )) != (unsigned) EOF ) {
#else
    while ( (c = GIFNextPixel( ReadValue )) != EOF ) {
#endif

	in_count++;

	fcode = (long) (((long) c << maxbits) + ent);
	i = (((code_int)c << hshift) ^ ent);    /* xor hashing */

	if ( HashTabOf (i) == fcode ) {
	    ent = CodeTabOf (i);
	    continue;
	} else if ( (long)HashTabOf (i) < 0 )      /* empty slot */
	    goto nomatch;
	disp = hsize_reg - i;           /* secondary hash (after G. Knott) */
	if ( i == 0 )
	    disp = 1;
probe:
	if ( (i -= disp) < 0 )
	    i += hsize_reg;

	if ( HashTabOf (i) == fcode ) {
	    ent = CodeTabOf (i);
	    continue;
	}
	if ( (long)HashTabOf (i) > 0 )
	    goto probe;
nomatch:
	output ( (code_int) ent );
	out_count++;
	ent = c;
#ifdef SIGNED_COMPARE_SLOW
	if ( (unsigned) free_ent < (unsigned) maxmaxcode) {
#else
	if ( free_ent < maxmaxcode ) {
#endif
	    CodeTabOf (i) = free_ent++; /* code -> hashtable */
	    HashTabOf (i) = fcode;
	} else
		cl_block();
    }
    /*
     * Put out the final code.
     */
    output( (code_int)ent );
    out_count++;
    output( (code_int) EOFCode );

    return 0;
}

/*****************************************************************
 * TAG( output )
 *
 * Output the given code.
 * Inputs:
 *      code:   A n_bits-bit integer.  If == -1, then EOF.  This assumes
 *              that n_bits =< (long)wordsize - 1.
 * Outputs:
 *      Outputs code to the file.
 * Assumptions:
 *      Chars are 8 bits long.
 * Algorithm:
 *      Maintain a BITS character long buffer (so that 8 codes will
 * fit in it exactly).  Use the VAX insv instruction to insert each
 * code in turn.  When the buffer fills up empty it and start over.
 */

void	output( code )
code_int  code;
{
    cur_accum &= code_mask[ cur_bits ];

    if( cur_bits > 0 )
	cur_accum |= ((long)code << cur_bits);
    else
	cur_accum = code;

    cur_bits += n_bits;

    while( cur_bits >= 8 ) {
	char_out( (unsigned int)(cur_accum & 0xff) );
	cur_accum >>= 8;
	cur_bits -= 8;
    }

    /*
     * If the next entry is going to be too big for the code size,
     * then increase it, if possible.
     */
   if ( free_ent > maxcode || clear_flg ) {

	    if( clear_flg ) {

		maxcode = MAXCODE (n_bits = g_init_bits);
		clear_flg = 0;

	    } else {

		n_bits++;
		if ( n_bits == maxbits )
		    maxcode = maxmaxcode;
		else
		    maxcode = MAXCODE(n_bits);
	    }
	}

    if( code == EOFCode ) {
	/*
	 * At EOF, write the rest of the buffer.
	 */
	while( cur_bits > 0 ) {
		char_out( (unsigned int)(cur_accum & 0xff) );
		cur_accum >>= 8;
		cur_bits -= 8;
	}

	flush_char();

	fflush( g_outfile );

	if( ferror( g_outfile ) )
		writeerr();
    }
}

/*
 * Clear out the hash table
 */

void	cl_block ()             /* table clear for block compress */
{

	cl_hash ( (count_int) hsize );
	free_ent = ClearCode + 2;
	clear_flg = 1;

	output( (code_int)ClearCode );
}

void	cl_hash(hsize)          /* reset code table */
register count_int hsize;
{

	register count_int *htab_p = htab+hsize;

	register long i;
	register long m1 = -1;

	i = hsize - 16;
	do {                            /* might use Sys V memset(3) here */
		*(htab_p-16) = m1;
		*(htab_p-15) = m1;
		*(htab_p-14) = m1;
		*(htab_p-13) = m1;
		*(htab_p-12) = m1;
		*(htab_p-11) = m1;
		*(htab_p-10) = m1;
		*(htab_p-9) = m1;
		*(htab_p-8) = m1;
		*(htab_p-7) = m1;
		*(htab_p-6) = m1;
		*(htab_p-5) = m1;
		*(htab_p-4) = m1;
		*(htab_p-3) = m1;
		*(htab_p-2) = m1;
		*(htab_p-1) = m1;
		htab_p -= 16;
	} while ((i -= 16) >= 0);

	for ( i += 16; i > 0; i-- )
		*--htab_p = m1;
}

void	writeerr()
{
	printf( "error writing output file\n" );
	exit(1);
}



