/* Application header file for user with 3D renderer */

/* Part of the REND386 package by Dave Stampe and Bernie Roehl */

/* Copyright 1992 by Dave Stampe and Bernie Roehl.
   May be freely used to write software for release into the public domain;
   all commercial endeavours MUST contact Bernie Roehl and Dave Stampe
   for permission to incorporate any part of this software into their
   products!
 */


#ifndef SCRIDEF
#define SCRIDEF 1
struct Screeninfo {
	int xmin, ymin, xmax, ymax, xcent, ycent, colors, pages, bw;
	long aspect;
	char id[80];
	};
#endif

extern struct Screeninfo *screeninfo;

#ifndef MATRIXDEF
typedef long MATRIX[4][3];        /* 3x3 rotate plus 3 translate para's */
			/* rotate matrix is <3.29>, translate is <32.0> */
#define MATRIXDEF 1
#endif

/* renderer viewpoint/screen control structure */
/* viewoint in X, Y, Z coords */
/* pan, tilt, roll in (float*65536) formats */
/* zoom is equiv. to magnification from 90 deg. FOV (also float*65536) */
/* aspect sets how much to magnify Y more than X to fix up displays */
/* light source point in world coordinates */
/* left, right, top, bottom set edges of screen */
/* hither sets closest point: keep >16 for best range of world coords */
/* yon sets max. distance: keep it 1<<26 if not used */
/* rest is renderer workspace */


#ifndef VIEWDEF
#define VIEWDEF 1
typedef struct {
	/* VIEWPOINT */
	long ex, ey, ez;        /* <25.0>  location of eyepoint         */
	long pan, tilt, roll;   /* <16.16> viewing angles (deg) +/- 128 */
	long zoom;              /* <16.16> 1/tan(H FOV/2) 0.5 to 16     */
	long lx,ly,lz;          /* <25.0>  location of light source     */
	int  directional;       /* 0 for point, 1 for normal (not unit) */
	int  ambient;           /* ambient light: 0-256 (72 recc.)      */

	/* SCREEN DATA */
	long left,right;        /* <25.0> clipping planes */
	long top, bottom;
	long hither, yon;       /* <25.0> near and far clipping planes   */
	long aspect;            /* <16.16> x:y fixup factor (magnify Y by...) */

	/* RENDERING CONTROL */
	unsigned flags;         /* 16 bits of flags */

	/* ADVANCED SCREEN DATA */
	long x_offset, y_offset;   /* amount to move screen center in pixels */
	unsigned orientation;      /* used to mirror screen image */
#define NOFLIP 0x0000      /* bits used in orientation field */
#define XFLIP  0x0001
#define YFLIP  0x0002

	MATRIX eye_xform;

	char spare[200];        /* 100 bytes needed, plus 100 bytes spare */
	} VIEW;
#endif


/* View flags: */

#define WIREFRAME           0x0001
#define HIDE_HIGHLIGHTED    0x0002
#define HIDE_UNHIGHLIGHTED  0x0004

/****** Only a stupid C programmer would do this...
#ifndef POLYDEF
#define POLYDEF 1
typedef void POLY;
#endif

#ifndef REPDEF
#define REPDEF 1
typedef void REP;
#endif


#ifndef OBJDEF
#define OBJDEF 1
typedef void *OBJECT;
#endif

#ifndef OBJLDEF
#define OBJLDEF 1
typedef void *OBJLIST;
#endif

typedef void SEGMENT;
**************/

#include "3dstruct.hpp"
#include "segment.hpp"

/* Function prototypes: */

OBJECT *new_obj(int type, int nv, int np);
void add_vertex(OBJECT *obj, long x, long y, long z);
POLY *add_poly(OBJECT *obj, unsigned color, int npoints);
void add_point(OBJECT *obj, POLY *p, int vertnum);
void delete_obj(OBJECT *obj);
long get_object_bounds(OBJECT *obj, long *x, long *y, long *z);
void compute_obj(OBJECT *obj);
void set_obj_flags(OBJECT *obj, unsigned value);
unsigned get_obj_flags(OBJECT *obj);
void highlight_obj(OBJECT *obj);
void unhighlight_obj(OBJECT *obj);
void first_rep(OBJECT *obj);
void next_rep(OBJECT *obj);
void set_rep_size(OBJECT *obj, long size);
long get_rep_size(OBJECT *obj);
REP  *add_representation(OBJECT *obj, long size, int nv, int np);
void select_representation(OBJECT *obj, long size);
void delete_rep(OBJECT *obj);

/* Depth-sorting control: */
#define DEEPEST 0x0000          /* sort polys by deepest point */
#define ATBACK  0x0001          /* push this object's poly's waaaay back */
#define AVERAGE 0x0002          /* sort polys by average depth */

#define BYOBJECT   0x0100       /* sort by object */
#define BYPOLY     0x0000       /* put polys in world before sort */

unsigned get_object_sorting(OBJECT *obj);
void set_object_sorting(OBJECT *obj, unsigned depth_type);

unsigned get_default_depth_sort();
void set_default_depth_sort(unsigned value);

/* Values for object flags */
#define OBJ_NONSEL       0x0800 /* can't be selected (i.e. pointer) */
#define OBJ_INVIS        0x1000
#define OBJ_HIGHLIGHTED  0x2000
#define OBJLIST_HEADER   0x4000
#define IS_OBJECT        0x8000 /* required by renderer: it will set */

OBJLIST *new_objlist();
void add_to_objlist(OBJLIST *list, OBJECT *obj);
void remove_from_objlist(OBJECT *obj);
void del_objlist(OBJLIST *list);
OBJLIST *on_objlist(OBJECT *obj);
OBJECT *first_in_objlist(OBJLIST *objlist);
OBJECT *next_in_objlist(OBJECT *obj);
OBJECT *prev_in_objlist(OBJECT *obj);
int is_first_in_objlist(OBJECT *obj);
int is_last_in_objlist(OBJECT *obj);
void walk_objlist(OBJLIST *objlist, void (*fn)(OBJECT *));
void get_obj_info(OBJECT *obj, int *nv, int *np);
void get_vertex_info(OBJECT *obj, int vertnum, long *x, long *y, long *z);
void get_vertex_world_info(OBJECT *obj, int vertnum, long *x, long *y, long *z);
void get_poly_info(OBJECT *obj, int polynum, unsigned *color, int *nverts, int *verts, int maxverts);
void set_poly_color(OBJECT *obj, int polynum, unsigned color);
void *get_object_owner(OBJECT *obj);
void set_object_owner(OBJECT *obj, void *owner);
void copy_world_to_object(OBJECT *obj);

void compute_view_factors(VIEW *v);

void fast_view_factors(VIEW *v);
		/* computes eye point/ angle movement factors only */

void initialize_screen_factors(VIEW *v);
		/* computes screen and viewport    */
		/* factors.  These stay constant   */
		/* over eye point changes          */
		/* setup viewwport data, do once   */
		/* unless offset, zoom etc changed */


void setup_render(unsigned K_of_mem, int maxpolys);
void reset_render();
void render(OBJLIST *objlist, VIEW *view);
void render_set_view(VIEW *view);
void subrender(OBJLIST *objlist);
int poly_cosine(POLY *poly);
void set_screen_monitor(int x, int y);
void clear_screen_monitor();
POLY *read_screen_monitor();

long where_pt(OBJLIST *objlist, long x, long y, long z, OBJECT **wobj, int *wvert);
OBJECT *best_collision(OBJLIST *objlist, long x, long y, long z);
int test_collision(OBJECT *obj, long x, long y, long z);
OBJECT *locate_screen_pt(int *pol, int *vert);
OBJECT *where_screen_pt(int *pol, int *vert, int x, int y);

SEGMENT *new_seg(SEGMENT *parent);
void seg_set_object(SEGMENT *s, OBJECT *app);
void *seg_get_object(SEGMENT *s);
char *seg_getname(SEGMENT *s);
void seg_setname(SEGMENT *s, char *name);
void seg_getposang(SEGMENT *s, long *rx, long *ry, long *rz);
void seg_getjointang(SEGMENT *s, long *rx, long *ry, long *rz);
void seg_getposxyz(SEGMENT *s, long *x, long *y, long *z);
void seg_getjointxyz(SEGMENT *s, long *x, long *y, long *z);
void abs_move_segment(SEGMENT *s, long tx, long ty, long tz);
void rel_move_segment(SEGMENT *s, long tx, long ty, long tz);
void abs_mat_segment(SEGMENT *s, MATRIX m);
void rel_mat_segment(SEGMENT *s, MATRIX m);
void abs_rotmat_segment(SEGMENT *s, MATRIX m);
void rel_rotmat_segment(SEGMENT *s, MATRIX m);
void abs_rot_segment(SEGMENT *s, long rx, long ry, long rz, int order);
void rel_rot_segment(SEGMENT *s, long rx, long ry, long rz, int order);
void move_rep(OBJECT *obj); /* move current rep of object  */
void set_move_handler(void (*move_handler_ptr)(OBJECT *));
void full_update_segment(SEGMENT *seg);
void update_segment(SEGMENT *seg); /* scan till update needed */
SEGMENT *find_root_segment(SEGMENT *s);
SEGMENT *parent_segment(SEGMENT *s);
SEGMENT *child_segment(SEGMENT *s);
SEGMENT *sibling_segment(SEGMENT *s);
MATRIX *get_seg_jmatrix(SEGMENT *s);
MATRIX *get_seg_pmatrix(SEGMENT *s);
void detach_segment(SEGMENT *s);    /* assumes segment is updated! */
void attach_segment(SEGMENT *s, SEGMENT *to); /* assumes parent is updated! */
void delete_segment(SEGMENT *s, void (*delapp_fn)(OBJECT *));
SEGMENT *find_segment_by_name(SEGMENT *s, char *name);
SEGMENT *copy_segment(SEGMENT *s, long dx, long dy, long dz, void *(*copy_fn)());

enter_graphics();
exit_graphics();
void clear_display(int page);
void vgabox(int left, int top, int right, int bottom, int color);

load_pcx(FILE *in, int page);
int save_pcx(FILE *out, int page);

typedef struct {   /* stereoscopic view factors */
	long phys_screen_dist;  /* eye-to-screen distance (mm); keep < 32000  */
	long phys_screen_width; /* viewport width on screen (mm)              */
	long pixel_width;       /* width of viewport in pixels                */
	long phys_eye_spacing;  /* spacing of eyes (mm) [usually 63]          */
	long phys_convergence;  /* convergence dist. (usually screen) (mm)    */
	long world_scaling;     /* world units per physical mm                */
	} STEREO ;

#define MONOSCOPIC 0 /* stereo types */
#define SWITCHED   1
#define SPLITLR    3
#define SEPARATE   5

#define LEFT_EYE   0
#define RIGHT_EYE  1

/* call fast_view_factors or view_from_matrix BEFORE these! */

	/* make a left or right eye view, cache */

void compute_stereo_data(STEREO *stereo, int eye, int xflip, int xdist, long yr,
			 long left, long top, long right, long bottom );

	/* create a new view from cyclopean view and cached data */

void make_stereo_view(VIEW *root, VIEW *nview, int eye);

	/* sufficient if only view point has changed and no twist or offset */
void update_stereo_view(VIEW *v, STEREO *s, int eye);

	/* makes a standard SWITCHED stereo screen */
void default_stereo_setup(VIEW *v, STEREO *s);

    /* high-resolution timer routines */
long current_time();
//void init_timer();
void set_current_time(long newtime);
int get_ticks_per_second();
extern volatile interrupts_occurred;

void *load_driver(char *dfile);

void matrix_view_factors(VIEW *v,MATRIX m); /* set up from matrix xform */
void view_to_matrix(VIEW *v,MATRIX m);      /* view matrix to xform matrix */

/* For other matrix routines, you should now include intmath.h */

/* End of rend386.h */
