//
// Hardware.hpp:
//	Interface Definition of classes parallelPort, machineTimer,
//	and gloveDriver.  Part of Object Glove library.
//
// Copyright 1992, 1993   Mark Thomas Pflaging
//
// Date:    5/15/92 - 6/4/92 (Release 1)
//	    8/9/92 - 9/21/92 (Release 2)
//	            11/20/92 (Release 2.1)
//                   3/22/93 (Release 3.0)     
//
#ifndef __HARDWARE_HPP
#define __HARDWARE_HPP

#include <dos.h>

#include "GloveDat.hpp"
#include "Fingers.hpp"
#include "AbsGest.hpp"

#ifndef _Windows
#include "t_types.h"
#include "htimer.h"
#endif

#include "isr.hpp"
#include "ini.hpp"

// bits for i/o ports
class subPort {

protected:
	unsigned char GDATA;	// PG data in
	unsigned char GSHIFT;	// Number of Bits to shift
	unsigned char GLATCH;	// PG latch out
	unsigned char GCLOCK;	// PG clock out
	unsigned char GCLOLAT;	// clock + latch

public:
	subPort(unsigned char gdata,
		unsigned char glatch,
		unsigned char gclock);
	subPort() {}
};

class parallelPort : subPort {
public:
	enum tPort {
		LPT1a = (int)'1',
		LPT1b,
		LPT2a,
		LPT2b,
		LPT3a,
		LPT3b
	};
	enum searchType { RR, RL, LL };

private:
	static char * title;
	static subPort primary;
	static subPort secondary;
	static searchType RightRight;
	int     InPort, OutPort;
	tPort which;
	Boolean polarity;
	static int userOutPort;
	void set_glove_inPort( int ip ) { InPort = ip; }
	void set_glove_outPort( int op ) { OutPort = op; }
	void set_user_port(InitFile & ini, int instance);

public:
	parallelPort(InitFile & ini, int instance);
	parallelPort(InitFile & ini, tPort p, int instance);
	setPort(tPort p);
	Boolean operator==(parallelPort & arg) {
		return ((arg.which == which) ? True : False);
	}
	parallelPort & operator++() {
		which = (tPort)(which + (((RightRight == RR) || (RightRight == LL)) ? 1 : 2));
		if (which > LPT3b) {
			if (polarity) which = LPT1a;
			else {
				if (RightRight == RL) which = LPT1b;
				else which = LPT1a;
			}
		}
		setPort((tPort)(which));
		return *this;
	}
	// In the following function, the return value indicates left-
	// or right-handedness.  True = right hand, False = left hand.
	// All gloves will return True if SearchingAlgorithm=Right-Right
	// in the .INI file.
	Boolean getPolarity() { return polarity; } 


protected:
	unsigned char BITREAD() { return ( ( inportb( InPort ) & GDATA ) >> GSHIFT ); }
	void C0L0() { outportb( OutPort, 0 ); }        // clock 0 latch 0.
	void C0L1() { outportb( OutPort, GLATCH ); }   // clock 0 latch 1.
	void C1L0() { outportb( OutPort, GCLOCK ); }   // clock 1 latch 0.
	void C1L1() { outportb( OutPort, GCLOLAT ); }  // clock 1 latch 1.
};

class machineTimer {
#ifndef _Windows
	HTimer * ptimer;
	int count_once();
	int get_count();
#endif
	// N and D will be calibrated.
	unsigned long int Num;
	unsigned long int Den;
	unsigned long int val;
	int test_value;

	// The following *function can be assigned to a
	// function which will be executed in place of the
	// instantiated delay loop.  Function execution time
	// should be approximately the same as the instantiated
	// delay.   -DPRC
	// (Frankly, I would be surprised if you actually
	// found a use for this. -mtp)
	void (*timing_func) (void);

public:
	machineTimer(unsigned int val, unsigned long N, unsigned long D)
		: Num(N), Den (D), val((unsigned long)val) { }
	// calibrates N and D
	machineTimer(unsigned int val, void (*func) (void),
		char * ascN = NULL, char * ascD = NULL);

#ifdef _Windows
	int isValid() { return test_value; }
#endif
	// The infamous delay loop!!
	void fdelay();
	void fdelay(int newdelay);
	// Now appearing at area theaters in static form!
	static void fdelay(unsigned long N, unsigned long D,
		unsigned long val);

	void multipliedDelay(int multiplier);
	void test();
	unsigned long getNum() { return Num; }
	unsigned long getDen() { return Den; }
	unsigned long getVal() { return val; }
	int getTestValue() { return test_value; }
};

// Set the maximum number of gloves!
#define MAXGLOVES	6

class gloveDriver : parallelPort, segaISR, public GloveData {
	// delay values for sending and sampling data.
	static machineTimer * D2BYTES, * D2BITS, * D2SLOW;
	static gloveDriver * forISR[MAXGLOVES];	// see hardware.cpp
	static int currentGlove;
	// # of interrupts to ignore glove so it
	// will recover from mode setup
	int glove_ignore;
	// number of times glove has been not ready
	int unready;
	// number of samples "missed" by the client program.
	unsigned nmissed;
	// Number of MILLISECONDS between samples.
	static int glove_tc;
	// Number of milliseconds to go until next sample.
	static int glove_pause;
	static char * title;

	unsigned char get_glove_byte();	// read byte from glove
	Boolean waitUntilReady();
	static void gloveISR();
	void intervalUpdate();
	Boolean polledUpdate();
	AbsGestureSystem * gestures;
	void calibrate(InitFile & ini);
	void setNextPort();
	static void next_glove();

public:
	typedef enum { EverythingOK, PolledModeWorking, DriverCoughing,
		PolledModeCoughing, TooManyInstances, UserHitKey,
		WindowsParametersMissing
	} statusType;
	gloveDriver(InitFile & ini, AbsGestureSystem * gs = NULL,
		void (*sega_switcher)(int) = NULL, Boolean (*quit_func)());
	virtual ~gloveDriver() {
		if (isRunning()) terminate();
		if (getInstantiations() == 1) {
			delete D2BYTES;
			delete D2BITS;
			delete D2SLOW;
		}
		C0L0();
	}
	void Start(Boolean IntervalOrPolled = True);
	virtual int updateGloveData();		// fill up the structure
	void slowDelay(int i) { D2SLOW->multipliedDelay(i); }

	Boolean yield();
	void waitForSample();
	void test();
	statusType driverStatus();
	static gloveDriver & indexedDriver(int i) {
		return *(forISR[i]);
	}
	static char * getTitle() { return title; }
	parallelPort::getPolarity;
	int whichGlove() { return getInstance(); }
	void checkGestures() { if (gestures) gestures->check(); }
	static size_t timeBetweenSamples() { return (getInstantiations() * glove_tc); }

protected:
	Boolean is_glove_ready();	// Delay repeat calls by 2-4 ms

	void set_glove_hiResMode();	// puts glove in hires mode
};

#endif
