/***********************************************************************/
/* rm.c   Program to emulate the unix "rm" command

Copyright (c) 1992, 1993 Richard C. Cochran  All rights reserved.

You may redistribute this entire package including all documentation,
provided no fee is charged, other than normal bulletin board access
time, for such redistribution.  You may use this code and modify it
for your own use, but you may not sell this code.

This program has been compiled and tested under Borland C++ 3.1 and
Borland Turbo C++ 1.0.  I've been using many times a day for over
a year, but nevertheless, I provide no warranty that it will work
for you.
************************************************************************/

#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>
#include <errno.h>
#include <sys\stat.h>
#include <io.h>

/* Function prototypes */

void usage (const char *commandname);

int rmpath(const char *name);
void removeffblk(const struct ffblk *fileblock);
void deny (const char *name);
int verify (const struct ffblk *fileblock);
void updatecwd(void);


/* global variables */
int recurse = 0;
int force = 0;
int interactive = 0;
char cwd[MAXPATH+1];

/***********************************************************************/
/*  Main program
      Parses input arguments, calls remaining routines.

*/

main (int argc, char **argv)
{
    int i, j, flags, bypassarg;

    /* check for options preceeded by '-' or '/' */
    bypassarg = 0;
    for(i=1; i < argc && !bypassarg;  i++){
	if(*(argv[i]) == '-' || *(argv[i]) == '/'){
            /* handle a single solitary '-', next args are filenames */
            if(!argv[i][1])
		bypassarg = 1;

	    for(j=1; j<strlen(argv[i]); j++){
		if(argv[i][j] == 'r'){
		    recurse = 1;
		} else if(argv[i][j] == 'i') {
		    interactive = 1;
		} else if(argv[i][j] == 'f') {
		    force = 1;
		} else {
		    usage(argv[0]);
		}
	    }
	    /* patch up argv & argc, so that it looks like the dash
	       argument wasn't there */
	    argc--;
	    for(j=i; j<argc; j++){
		argv[j] = argv[j+1];
	    }
	    i--;
	}
    }

    /* if we didn't get any filename arguments to remove, tell the
       user how to use us. */
    if(argc < 2){
	usage(argv[0]);
    }

    for (i = 1; i < argc; i++){
	/* convert filnames to upper case, especially needed for drive
	   letters. */
	for(j = 0; j < strlen(argv[i]); j++){
	    if(islower(argv[i][j]))
		argv[i][j] = toupper(argv[i][j]);
	}

	/* don't remove filenames that end in \.  Reason: this is most
	   likely an attempt to remove the root directory.  If the user
	   wants to remove a directory, it should not be postfixed with
	   a backslash */
	flags = fnsplit(argv[i], 0, 0, 0, 0);
	if((flags & DIRECTORY) && !(flags & FILENAME))
	    fprintf(stderr, "Can't remove %s\n", argv[i]);
	else
	    if(rmpath(argv[i]) == 2 && !force)
                fprintf(stderr, "%s - No such file or directory\n", argv[i]);
    }

    return(0);
} /* end main */

/***********************************************************************/
/* rmpath recursively removes a path given its name.  Name may include
   wildcards.  Previous drive/path are saved and restored */

int rmpath (const char *name){
    char oldpath[MAXPATH];
    int olddisk;
    struct ffblk fileblock;
    int fileattr;
    char drive[MAXDRIVE];
    char dir[MAXDIR];
    char file[MAXFILE];
    char ext[MAXEXT];
    int flags;
    char fullname[MAXFILE+MAXEXT];

    /* save current working drive and path */
    olddisk = getdisk();
    flags = fnsplit(name, drive, dir, file, ext);

    /* switch to correct drive */
    if (flags & DRIVE){
	setdisk(drive[0]-'A');
	if (drive[0]-'A' != getdisk()){
	    fprintf(stderr,"Drive %s is not available\n", drive);
	    return 1;
	}
    }

    updatecwd();


    /* save current directory on this drive */
    getcwd(oldpath, sizeof(oldpath));

    /* and switch to this directory */
    if (flags & DIRECTORY){
	/* eliminate trailing '\' if necessary */
	if(dir[strlen(dir)-1] == '\\')
	    if(strlen(dir) > 1)
		dir[strlen(dir)-1] = '\0';
	if(chdir(dir)){
	    fprintf(stderr,"Path not found: %s%s\n", drive, dir);
	    chdir(oldpath);
	    setdisk(olddisk);
	    updatecwd();
	    return 1;
	}
	updatecwd();
    }

    /* glue together file and extension */
    strcpy(fullname, file);
    strcat(fullname, ext);

    if (findfirst(fullname, &fileblock, FA_RDONLY|FA_HIDDEN|
		FA_SYSTEM|FA_DIREC )){
	/* file not found */
	chdir(oldpath);
	setdisk(olddisk);
	updatecwd();
	return 2;
    }

    removeffblk(&fileblock);

    while(!findnext(&fileblock)){
	removeffblk(&fileblock);
    }


    /* restore saved working directory and drive*/
    chdir(oldpath);
    setdisk(olddisk);
    updatecwd();

    return 0;
}
/***********************************************************************/
/* removeffblk removes a single file, given its fileblock */

void removeffblk(const struct ffblk *fileblock)
{
    char subfiles[20];

    if(fileblock->ff_attrib & FA_DIREC){
	/* don't try to remove . or .. */
	if(!strcmp(fileblock->ff_name, "."))
	    return;
	if(!strcmp(fileblock->ff_name, ".."))
	    return;
	if(recurse){
	    if(verify(fileblock)){
		/* remove subfiles before removing directory */
		strcpy(subfiles, fileblock->ff_name);
		strcat(subfiles, "\\*.*");
		rmpath(subfiles);
		if(rmdir(fileblock->ff_name))
		    if(errno==EACCES){
			deny(fileblock->ff_name);
		    }
	    }
	} else {
	    fprintf(stderr,"Cannot remove %s%s -- is a directory\n",
		    cwd, fileblock->ff_name);
	}

    } else {

	if(verify(fileblock))
	    if(unlink(fileblock->ff_name))
		if(errno==EACCES)
		    deny(fileblock->ff_name);
    }

}

/***********************************************************************/
/* deny simply tells the user tha permission was denied for a file */
void deny(const char *name){
    fprintf(stderr, "%s%s -- Permission Denied.\n", cwd, name);
}

/***********************************************************************/
/* verify asks the user to verify whether or not we want to remove a file.
   returns 0 if no, 1 if ok to delete */

int verify(const struct ffblk *fileblock){

    int readonly, hidden, system, direc, rc;
    char buffer[20];

    buffer[0]=18;


    readonly = (fileblock->ff_attrib & FA_RDONLY) && !force;
    hidden = (fileblock->ff_attrib & FA_HIDDEN) && !force;
    system = (fileblock->ff_attrib & FA_SYSTEM) && !force;
    direc = (fileblock->ff_attrib & FA_DIREC);

    if(interactive || readonly || hidden || system){
	printf("Remove %s%s? ", cwd, fileblock->ff_name);
	if(readonly)
	    printf("(ReadOnly) ");
	if(hidden)
	    printf("(Hidden) ");
	if(system)
	    printf("(System) ");
	if(interactive && direc)
	    printf("(Directory) ");
	printf("<Y/N> ");

	cgets(buffer);
	printf("\n");
	rc = (buffer[1] && (buffer[2] == 'y' || buffer[2] == 'Y'));

	/* change file permissions if necessary */
	if(rc && readonly)
	    chmod(fileblock->ff_name, S_IREAD|S_IWRITE);
	return rc;
    } else {
        /* we weren't interactive */
	/* change file permissions if asked to force a readonly file */
	if(force && (fileblock->ff_attrib & FA_RDONLY))
	    chmod(fileblock->ff_name, S_IREAD|S_IWRITE);
	return 1;
    }
}


/***********************************************************************/
/* update should be called after every chdir, to keep the
   global cwd variable up-to-date.  Keeping cwd as a global
   prevents having to litter the code with calls to getcwd */
void updatecwd(void)
{
    getcwd(cwd, sizeof(cwd)-1);
    if(cwd[strlen(cwd)-1] != '\\')
    strcat(cwd, "\\");
}


void usage (const char *commandname)
{
    char filename[MAXFILE];

    if(!(fnsplit(commandname, 0, 0, filename, 0) & FILENAME))
	strcpy(filename, "rm");

    fprintf(stderr," RM -- REMOVE FILES --   V3.0\n");
    fprintf(stderr,"usage: %s [-fir] [-] filename...\n", filename);
    fprintf(stderr,"Copyright (c) 1992, 1993 Richard Cochran, ");
    fprintf(stderr,"All Rights Reserved.\n\n");

    fprintf(stderr," -f  Force all files (hidden, system, read/only)\n");
    fprintf(stderr,"      to be removed without prompting or error display.\n");
    fprintf(stderr," -i  Interactive.  Prompt before removing each file.\n");
    fprintf(stderr," -r  Recursively remove subdirectories and contents.\n");
    fprintf(stderr," -   Means the following arguments are filenames.\n");
    fprintf(stderr,"      (so you can rm files beginning with '-')\n");
    fprintf(stderr,"\nThis program may be given away and used by\n");
    fprintf(stderr,"individuals in its unaltered form.  It may not\n");
    fprintf(stderr,"be sold without written permission of the author.\n");
    fprintf(stderr,"\tRichard Cochran\n\tCompuServe: 73040,263\n\t");
    fprintf(stderr,"Internet: 73040.263@compuserve.com\n");
    exit(1);
}