/*=========================================================*
 |                                                         |
 | ACKPCX_W.C                                              |
 |                                                         |
 *=========================================================*
 | PCX Graphics File Format output routines.               |
 *=========================================================*
 | Copyright (c) 1991, 1994 kgraham,  All Rights Reserved  |
 | These routines are being included with the ACK 3D       |
 | Construction Kit and are subject to the licensing       |
 | restrictions of said software.                          |
 *=========================================================*/
#include <stdio.h>
#include "ack3d.h"
#include "ackpcx.h"

#define     OK      0
#define     ERROR  -1

int writepcx(char *fname, char *buffer, char *palette, int xsize, int ysize);
int pcx_putc(unsigned int, FILE *);
int pcx_outc(unsigned int, unsigned int, FILE *);

/*====================================*
 |                                    |
 | writepcx()                         |
 |                                    |
 +------------------------------------+
 | Requires:                          |
 |  fname   - pointer to file name    |
 |  buffer  - pointer to the image    |
 |  palette - pointer to palette      |
 |  xsize   - horizontal size of image|
 |  ysize   - vertical size of image  |
 |                                    |
 | Returns:                           |
 |  OK    - file written correctly    |
 |  ERROR - a file error occurred     |
 *====================================*/
int writepcx(char  *fname, char  *buffer, char  *palette, int xsize, int ysize)
{
    int      x;
    int      y;
    FILE    *fp;
    PCXHDR   hdr;
    char    *pbuff = buffer;

    if (fname == NULL) {
        return(ERROR);
    }
    /*
     * clear header to 0
     */
    memset(&hdr, 0, sizeof(PCXHDR));
    /*
     * fill out header info
     */
    hdr.pcxflag      = PCXFLAG;
    hdr.version      = VERSION30;
    hdr.encoding     = 1;           /* use RLE encoding */
    hdr.bitsperpixel = 8;           /* 8 bits/pixel     */
    hdr.x1           = 0;           /* upper left x     */
    hdr.y1           = 0;           /* upper left y     */
    hdr.x2           = xsize-1;     /* lower right x    */
    hdr.y2           = ysize-1;     /* lower right y    */
    hdr.hres         = 80;          
    hdr.vres         = 80;
    memset(&hdr.triple, 0, 3*16);
    hdr.videomode    = 0;           /* obsolete (use 0) */
    hdr.nplanes      = 1;           /* video planes = 1 */
    hdr.bytesperline = hdr.x2+1;    /* bytes per line   */
    hdr.palinfo      = 1;           /* color palette    */
    hdr.hscreensize  = 640;         /* screen res       */
    hdr.vscreensize  = 480;
    /*
     * open the image file and write out the header
     */
    if ((fp = fopen(fname, "wb")) == NULL) {
        return(ERROR);
    }
    if (fwrite(&hdr, sizeof(PCXHDR), 1, fp) != 1) {
        return(ERROR);
    }
    /*
     * output the image in RLE encoded format
     */
    for (y=0;y<(hdr.y2+1);y++) {
        for (x=0;x<(hdr.bytesperline);x++) {   /* read a line          */
            if (pcx_putc(*pbuff++, fp) != OK) { /* encode and save it   */
                fclose(fp);
                return(ERROR);
            }
        }
        pcx_putc(-1,fp);                    /* flush buffer            */
    }
    pcx_putc(-1,fp);                        /* flush buffer            */
    /*
     * output the palette "magic cookie"
     */
    fputc(PALFLAG, fp);
    /*
     * output the palette
     */
    if (fwrite(palette, 3*256, 1, fp) != 1) {
        return(ERROR);
    }
    /*
     * flush and close the file
     */
    fflush(fp);
    fclose(fp);
    return(OK);
}

/*==================================*
 |                                  |
 | pcx_putc()                       |
 |                                  |
 +----------------------------------+
 | Requires:                        |
 |  c     - character to output     |
 |  fp    - file pointer to output  |
 |          file                    |
 |                                  |
 | Returns:                         |
 |  OK    - file written correctly  |
 |  ERROR - a file error occurred   |
 *==================================*/
int pcx_putc( unsigned int c, FILE *fp )
{
    static int    lastc = 0;    /* last character sent              */
    static int    count = 0;    /* count of chars for repeat count  */

    if ( c == -1 )    {         /* flush char & rep count (if any)  */
        if ( count != 0 ) {     /* if no count then no output       */
            if ( pcx_outc( lastc, count, fp ) ) {    /* output char */
                return(ERROR);
            }
        }
        lastc = count = 0;      /* reset last char and count        */
        return(OK);
    }

    if ( c == lastc )    {      /* if same char again up the count  */
        count++;                /* c==0 will up count to 1 same as  */
        return(OK);             /* if it fell through               */
    }

    if ( count != 0 ) {         /* if c!=lastc and count>0 then     */
        if ( pcx_outc( lastc, count, fp ) ) {   /* output c & count */
            return(ERROR);
        }
    }

    lastc = c;                  /* if it gets here it must be first */
    count = 1;                  /* time for this char               */
    return(OK);
}

/*==================================*
 |                                  |
 | pcx_outc()                       |
 |                                  |
 +----------------------------------+
 | Requires:                        |
 |  c     - character to output     |
 |  n     - number of times to write|
 |          character to file       |
 |  fp    - file pointer to output  |
 |          file                    |
 |                                  |
 | Returns:                         |
 |  OK    - file written correctly  |
 |  ERROR - a file error occurred   |
 *==================================*/
int pcx_outc( unsigned int c, unsigned int n, FILE *fp )
{
    if ( (n > 1) || ( (c & 0xc0) == 0xc0 ) ) {
        while ( n > 0x3f ) {
            if (pcx_outc(c, 0x3f, fp)) {    /* call recursively until n < 0x3f    */
                return(ERROR);                        
            }
            else {
                n -= 0x3f;
            }
        }
        if ( !n ) {                     /* if n == 0 all bytes are out  */
            return(OK);
        }
        if (fputc(0xc0 | n, fp) == EOF) {   /* output encoded byte count*/
            return(ERROR);
        }
    }
    if ( fputc( c, fp ) == EOF ) {      /* output byte value        */
        return(ERROR);
    }
    return(OK);
}

