/*
 * This utility dumps the topic file for a Windows Help File.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <memory.h>
#include "helpfile.h"

#define DEBUG            //Define if debug info displayed
//#define SHOWPHRASES      //Define if phrase table to be displayed

//--- HELP File support structures and routines ---

typedef struct {
  DWORD pFile;
  char szFilename[32];
  } FILETABLEENTRY;

void TraverseIfsBtree(FILE *fp,
  DWORD dwFirstBtreePage,
  WORD wCurrentLevel, WORD wLeafLevel, WORD wPage);
DWORD FindSubfile(char * szFilename);


//--- IFS subfile open file structure --
//    This structure is filled in whenever a subfile is open
//    It controls reading/writing, etc.

typedef struct {
  FILE *FilePointer;                    //File pointer for help file
  DWORD dwFileStart;                    //Starting position for file
  DWORD dwFileSize;                     //Size for file
  DWORD dwFilePos;                      //Current read/write pos in file
  } IFS_INFO;


//--- TOPIC sufile IFS file structure ---
//    This structure is filled in whenever a TOPIC subfile is open.
//    It controls reading/writing, etc.

#define RAWSIZE 0x1000                  //Size of raw data
#define MAXSIZE 0x4000                  //Size of xlated data (max)

typedef struct {
  IFS_INFO ifs;                         //IFS info for TOPIC subfile
  DWORD dwPageNumber;                   //Desired page number
  DWORD dwPagePos;                      //Desired position within page
  DWORD dwCurrentPage;                  //Current page number
  DWORD dwPageSize;                     //Size of current page
  } TOPICIFS_INFO;


//This data is used solely for TOPIC file, so can be defined global.
//However, this means that only one topic subfile can be open, and
// that if the ifs is duplicated, it must not go past the page boundary!
//This is okay since we are only using duplicate ifs to read topic
// text and attribs from the same record!

char cRawData[RAWSIZE];                 //Raw data buffer
char cTopicData[MAXSIZE];               //Translated topic data buffer



int topic_fopen(FILE *fp, TOPICIFS_INFO *ifs, char *pszFilename);
int topic_fclose(TOPICIFS_INFO *ifs);
DWORD topic_fseek(TOPICIFS_INFO *ifs, DWORD dwPos);
DWORD topic_ftell(TOPICIFS_INFO *ifs);
DWORD topic_fread(TOPICIFS_INFO *ifs, void *pDest, DWORD dwSize);
DWORD topic_readpage(TOPICIFS_INFO *ifs);

int ifs_fopen(FILE *fp, IFS_INFO *ifs, char *pszFilename);
int ifs_fclose(IFS_INFO *ifs);
DWORD ifs_fseek(IFS_INFO *ifs, DWORD dwPos);
DWORD ifs_ftell(IFS_INFO *ifs);
int ifs_feof(IFS_INFO *ifs);
DWORD ifs_fread(IFS_INFO *ifs, void *pDest, DWORD dwSize);

int ReadPhrasesFile(FILE *fp, WORD fCompressed);
int ReleasePhraseTable(void);
int TranslatePhrase(unsigned char *pDest, int iDstLen, unsigned char *pSource,
 int iSrcLen);
int Decompress(BYTE *pDest, WORD wDestLen, BYTE *pSource, WORD wSourceLen);
int HexDump(unsigned char *ptr, int blen);

int DumpTopicFile(FILE *fp);


static FILETABLEENTRY FileTable[128];           //File table
static WORD wFileCount;                         //# of files

static BYTE fCompressed = 0;                    //Compression flag


void ReadParAttribs(TOPICIFS_INFO *ifs);
void DisplayText(TOPICIFS_INFO *pTopicIfs, TOPICIFS_INFO *pAttribIfs);


static PHRASESHEADER PhraseHeader;      //Phrases header
static WORD *pPhraseTable;		//Pointer to table of phrases
static BYTE *pRawPhrases;		//Pointer to raw phrases
static char *pPhrases;			//Pointer to uncompressed phrases



int main(int argc, char * argv[1])
{
  FILE *fp;                                     //File pointer
  int bytesread;                                //# of bytes read

  HELPHEADER HelpHeader;                        //File header
  FILESYSTEMHEADER FileSystem;                  //File system header
  WORD i;                                       //Temp counter
  WORD page;                                    //Page counter
  WORD entry;                                   //Entry counter
  DWORD pFile;                                  //File pointer
  FILEHEADER FileHeader;                        //Subfile header
  SYSTEMFILE SystemFile;                        //System file contents
  SYSTEMDATA SystemData;                        //System file data
  int j;
  char Filename[64];
  char Sectionname[64];
  int fError;

  *Filename='\0';
  *Sectionname='\0';

  if (argc==2)
  {
    strcpy(Filename, argv[1]);
  }
  else
  {
    printf("Program usage:\n  DTOPIC filename\n");
    exit(1);
  };

  if (!strchr(Filename,'.'))
    strcat(Filename,".HLP");

  printf("\nDump of TOPIC contents for file %s\n\n", Filename);

  //--------------------------------------------------------------------
  //Open the help file
  //--------------------------------------------------------------------

  fp = fopen(Filename,"rb");
  if (fp==NULL) {
    printf("Error openning file %s\n", Filename);
    exit(1);
    };

  //--------------------------------------------------------------------
  //This section reads in the standard Help file header
  //--------------------------------------------------------------------

  bytesread = fread(&HelpHeader, sizeof(HELPHEADER), 1, fp);
  if (bytesread==0) {
    printf("Error reading file header\n");
    exit(1);
    };

  //--------------------------------------------------------------------
  //This section reads in the internal file system
  //The internal file table is stored in global FileTable[]
  //--------------------------------------------------------------------

  //-- Read the internal file system header --

  fseek(fp, (long)HelpHeader.pDirectory, SEEK_SET);
  bytesread = fread(&FileSystem, sizeof(FILESYSTEMHEADER), 1, fp);
  if (bytesread==0) {
    printf("Error reading internal file system header\n");
    exit(1);
    };

  //-- Read the internal file system file table --

  wFileCount=0;
  TraverseIfsBtree(fp,ftell(fp),1,
     FileSystem.wNoLevels,FileSystem.wRootPage);

  //--------------------------------------------------------------------
  //This section reads in the information from the SYSTEM subfile.
  //It calls FindSubfile to find the index of the subfile in the FileTable
  //then positions the read/write pointer there and reads the data
  //and displays it to the user.
  //--------------------------------------------------------------------

  //GROTS: CHECK WITH PDAVIS HELPDUMP RE DIFF FROM V3 & V3.1 FILES!

  pFile = FindSubfile("SYSTEM");
  if (pFile == 0xFFFFFFFF)
  {
    printf("*** Error: \"SYSTEM\" subfile not found! ***\n\n");
    exit(1);
  };

  //-- Position to system file --
  fseek(fp, (long)pFile, SEEK_SET);

  //-- Read standard file header --
  bytesread = fread(&FileHeader, sizeof(FILEHEADER), 1, fp);

  //-- Read system file header --
  bytesread = fread(&SystemFile, sizeof(SYSTEMFILE), 1, fp);

  if (SystemFile.wCompressionFlag) fCompressed = 1;

  //--------------------------------------------------------------------
  //Read in the information from the Phrases subfile
  //--------------------------------------------------------------------

  ReadPhrasesFile(fp, SystemFile.wCompressionFlag);

  //--------------------------------------------------------------------
  //Dump contents of Topic file.
  //--------------------------------------------------------------------

  DumpTopicFile(fp);


  //--------------------------------------------------------------------
  //--------------------------------------------------------------------

  ReleasePhraseTable();

  fclose(fp);
  return 0;

}




//--------------------------------------------------------------------
//Recursive routine to traverse the File System B-tree, starting at the
// first page.
//
// Each 2K page consists of either internal nodes or leaf nodes in
// the B-tree.
//
// A page of internal nodes consists of a IFSBTREENODEPAGEHEADER,
// followed by a number of node records, each of which consists of
// alternating acisz filename, and word page number.  In a page
// with n items, there are n file names and n+1 page numbers.  For
// example, in a page with 2 entries, you'd have:
//
//  IFSBTREENODEPAGEHEADER
//  page0
//  asciz0
//  page1
//  asciz1
//  page2
// where asciz0 is the name of the first file on page1, and asciz1
// is the name of the first file on page2.
//
// To search the B-tree, compare the desired filename with asciz0.
// If it is <, goto page0; otherwise compare with asciz1.  If less,
// goto page1 else goto page2
//
// To traverse the B-tree, recurse through page0, page1, ..., pagen+1
// in order.  The pages may be further nodes or leafs, depending on the
// level.
//
// A page of leaf nodes consists of a IFSBTREELEAFPAGEHEADER,
// followed by a number of leaf records, each of which consists of
// an asciz filename and a dword pointer to the file.
//
//--------------------------------------------------------------------


void TraverseIfsBtree(FILE *fp,
 DWORD dwFirstBtreePage,
 WORD wCurrentLevel, WORD wLeafLevel, WORD wPage)
{
  DWORD dwPage;
  IFSBTREENODEPAGEHEADER IfsBtreeNodePageHdr;
  IFSBTREELEAFPAGEHEADER IfsBtreeLeafPageHdr;
  char tmp;
  char *pFilename;
  int i,j;
  WORD wNextPage;
  int bytesread;
  DWORD dwCurPos;

  dwCurPos = ftell(fp);                         //Save original fpos

  dwPage = dwFirstBtreePage + (wPage*2048);     //Calc page ptr
  fseek(fp,dwPage,SEEK_SET);                    //Seek to page

  if (wCurrentLevel == wLeafLevel)
  {
    //------------------------------------------------------------
    //-- this is a leaf page --
    //------------------------------------------------------------
    bytesread = fread(&IfsBtreeLeafPageHdr, sizeof(IFSBTREELEAFPAGEHEADER), 1,
 fp);

    for (i=0; i<IfsBtreeLeafPageHdr.wNoEntries; i++)
    {
      fread(&tmp, 1, 1, fp);                    //Skip the 7C ()
      pFilename = FileTable[wFileCount].szFilename;
      fread(pFilename,1,1,fp);                  //Read first byte of name
      while (*pFilename) {                      //While still more to do
        pFilename++;                            // point to next byte
        fread(pFilename,1,1,fp);                // read next byte
        };
      fread(&FileTable[wFileCount].pFile,sizeof(DWORD),1,fp); //Read file pointer
      wFileCount++;
    };
  }
  else
  {
    //------------------------------------------------------------
    //-- this is a node page --
    //------------------------------------------------------------
    bytesread = fread(&IfsBtreeNodePageHdr, sizeof(IFSBTREENODEPAGEHEADER), 1, fp);

    //-- go through entries in b-tree --
    for (i=0; i<IfsBtreeNodePageHdr.wNoEntries; i++)
    {
      bytesread = fread(&wNextPage,sizeof(WORD),1,fp);
      j=0;
      tmp='*';
      while (tmp) {
       bytesread = fread(&tmp,1,1,fp);          //read char
       };

      TraverseIfsBtree(fp,dwFirstBtreePage,
        wCurrentLevel+1,wLeafLevel,wNextPage);

    };
    //-- do the final page# entry --
    bytesread = fread(&wNextPage,sizeof(WORD),1,fp);
    TraverseIfsBtree(fp,dwFirstBtreePage,
      wCurrentLevel+1,wLeafLevel,wNextPage);


  };

  fseek(fp,dwCurPos,SEEK_SET);                  //Restore original file pos

}


//--------------------------------------------------------------------
//This routine is used to read in a single file table entry from
// the internal file header in the help file.
//The entry is stored in the FileTable[wFileCount] entry.
//wFileCount and FileTable[] are global.
//wFileCount is incremented after each file entry is read.
//--------------------------------------------------------------------

int ReadFileEntry(FILE * fp)
{
  BYTE tmp;
  char * pFilename;

  fread(&tmp, 1, 1, fp);                        //Skip the 7C ()
  pFilename = FileTable[wFileCount].szFilename;
  fread(pFilename,1,1,fp);                      //Read first byte of name
  while (*pFilename) {                          //While still more to do
    pFilename++;                                // point to next byte
    fread(pFilename,1,1,fp);                    // read next byte
    };
  fread(&FileTable[wFileCount].pFile,4,1,fp);   //Read file pointer
  return 0;
}

//--------------------------------------------------------------------
//This procedure is called to search the file table for a particular
//subfile and to return the pointer to that file's data in the help file.
//It searches through the FileTable[] structure for a matching filename,
// and returns the file pointer stored there.
//It returns 0xFFFFFFFF if no match is found.
//The file name comparison IS NOT CASE SENSITIVE!
//--------------------------------------------------------------------

DWORD FindSubfile(char * szFilename)
{
  WORD i;
  for (i=0; i<wFileCount; i++)
    if (strcmpi(FileTable[i].szFilename, szFilename) == 0)
      return FileTable[i].pFile;
  return 0xFFFFFFFF;
}











//********************************************************************
//
// Dump the contents of the TOPIC file
//
//bla bla bla
//
//The data structures in helpfile.h are:
//
// -- Topic file header record ---
//
//  typedef struct {
//    DWORD dwPrevRecord;    //Prev record (last rec in prev page)
//    DWORD dwFirstRecord;   //First record (in this page)
//    DWORD dwPrevTopic;     //Start of prev topic (1st rec of last topic
//                           //      in prev page)
//    } TOPICHEADER;
//
// //-- Define generic topic record header structure --
//
// typedef struct {
//   WORD wRecordSize;     //Size of record
//   WORD wDataSize;       //Size of data
//   WORD wPrev;           //Previous record pointer
//   WORD wNext;           //Next record pointer
//   WORD wDataOffset;     //Offset to data
//   BYTE bRecordType;     //Type of record
//   } TOPICRECHEADER;
//
// //-- Define record types --
//
// #define TOPICTITLEID 0x02  //Id for topic title type record
// #define TOPICDATAID 0x20   //Id for topic data type record
//
// //-- Define header for topic title record --
//
// typedef struct {
//   WORD wNextTopicOffset;  //Offset to next topic
//   WORD wReserved1;        //Reserved
//   WORD wReserved2;        //Reserved
//   WORD wTopicId;          //Topic id
//   WORD wReserved3;        //Reserved
//   WORD wTopicData;        //Pointer to topic data
//   WORD wNextTopicPointer; //Pointer to next topic
//   } TOPICTITLERECHEADER;
//
//-- Define header for topic data record --
//
//typedef struct {
//  BYTE bUnknown;			//Unknown
//  BYTE bReserved80;			//Reserved: 0x80
//  WORD wTwiceData;			//Twice data size
//  } TOPICDATARECHEADER;
//
//typedef struct {
//  BYTE bReserved80also;		//Reserved: 0x80
//  DWORD dwParAttribs;			//Paragraph attributes
//  } TOPICDATARECHEADER2;
//
// //-- Define paragraph attribute bits --
//
//
// //-- Define segment attribute ids --
//
// #define SEG_FONT 0x80     //Font change identifier
// #define SEG_NEWLINE 0x81  //New line indicator
// #define SEG_PAR 0x82      //New paragraph indicator
// #define SEG_TAB 0x83      //Tab indicator
// #define SEG_ENDLINK 0x89  //End of hotlink
// #define SEG_POPLINK 0xE2  //Popup hot-link indicator
// #define SEG_HOTLINK 0xE3  //Hot link indicator
// #define SEG_END 0xFF      //End of segment list
//
//
//
//********************************************************************

int DumpTopicFile(FILE *fp)
{
  static char title_indent[] = "";
  static char data_indent[]  = "    ";
  char *indent;

  TOPICIFS_INFO ifs;
  TOPICIFS_INFO ifs_dup;

  TOPICHEADER TopicHeader;
  TOPICHEADER TmpHeader;
  TOPICRECHEADER TopicRecHeader;


  TOPICTITLERECHEADER TopicTitleRecHeader;

  BYTE bUnknown0;			//Unknown
  BYTE bUnknown1;			//Unknown
  BYTE bTwiceData;			//Twice data size
  BYTE bTwiceDataHi;
  WORD wTwiceData;
  BYTE bColumnCount;                  //# of columns
  DWORD dwColumnWidth;                //Column widths
  WORD wColumnNo;                     //Column number
  WORD wUnknown;                      //Unknown word value (or 2 bytes)
  WORD wUnknown1;                     //Unknown word value (or 2 bytes)
  BYTE bUnknown2;                     //Unknown, usually 80.
  DWORD dwParAttribs;                 //Paragraph attributes


  static char Tmp[0x1000];
  static char Title[0x1000];
  #define TMPLEN 56
  static char Tmp2[TMPLEN+1];
  char *ptmp;
  char *ptmp2;

  int bytesread;
  int i;
  int fMore;
  DWORD pNext;
  DWORD pPrev;
  DWORD dwi;
  WORD wDataSize;

  DWORD dwTmp;

  WORD wParam;
  DWORD dwParam;
  BYTE bAttrib;
  int segid, segcount, seglen;

  DWORD dwHotspotPage;
  DWORD dwHotspotPointer;

  if (!topic_fopen(fp,&ifs,"TOPIC"))
  {
    printf("*** Error openning file TOPIC! ***\n");
    return 0;
  };

  //-- Display generic topics found every 1000h bytes --

  #ifdef DEBUG
  printf("\n  Offset      PrevRec  FirstTop PrevTopic\n");
  printf("\n  --------    -------- -------- --------\n");
  for (dwi=0; dwi<ifs.ifs.dwFileSize;dwi+=0x1000)
  {
    ifs_fseek(&ifs.ifs, dwi);
    ifs_fread(&ifs.ifs, &TmpHeader, sizeof(TOPICHEADER));
    printf("  %08lX    %08lX %08lX %08lX\n",
      dwi,
      TmpHeader.dwPrevRecord, TmpHeader.dwFirstRecord, TmpHeader.dwPrevTopic);
  };
  printf("\n\n");
  #endif

  //-- Now, go through records and display info for them --

  topic_fseek(&ifs, 0);
  topic_fread(&ifs, &TopicHeader, sizeof(TOPICHEADER));

  pNext = TopicHeader.dwFirstRecord;
  dwHotspotPage = dwHotspotPointer = 0;

  while (pNext != 0xFFFFFFFF)
  {

    topic_fseek(&ifs,pNext);
    topic_fread(&ifs,&TopicRecHeader,sizeof(TOPICRECHEADER));
    ifs_dup = ifs;
    topic_fseek(&ifs_dup, pNext + TopicRecHeader.dwDataOffset);

    if (TopicRecHeader.bRecordType == TOPICTITLEID)
    {
      printf("\n");
      indent = title_indent;
    }
    else
      indent = data_indent;

    //-- Note that topic page is shifted 3 bits, not 2! --

    printf("\n");
    printf("%sHotspot Address = %08lX\n", indent, ((dwHotspotPage<<3)+dwHotspotPointer));
    printf("\n");
    printf("%sAddress         = %08lX\n", indent, pNext);
    printf("%sdwNext          = %08lX\n", indent, TopicRecHeader.dwNext);
    printf("%sdwPrev          = %08lX\n", indent, TopicRecHeader.dwPrev);
    printf("%sdwRecordSize    = %08lX\n", indent, TopicRecHeader.dwRecordSize);
    printf("%sdwDataSize      = %08lX\n", indent, TopicRecHeader.dwDataSize);
    printf("%sdwDataOffset    = %08lX\n", indent, TopicRecHeader.dwDataOffset);
    printf("%sbRecordType     = %02X\n", indent, TopicRecHeader.bRecordType);

    if (TopicRecHeader.bRecordType == TOPICTITLEID)
    {
      //-----------------------------------------------
      //-- This is a type 0x02 topic title record
      //-- Display record info and topic title.
      //-----------------------------------------------

      //-- Report on title record header --
      topic_fread(&ifs, &TopicTitleRecHeader, sizeof(TOPICTITLERECHEADER));
      printf("%sdwTopicId       = %08lX\n",
        indent, TopicTitleRecHeader.dwTopicId);
      printf("%sdwReserved1     = %08lX\n",
        indent, TopicTitleRecHeader.dwReserved1);
      printf("%sdwReserved2     = %08lX\n",
        indent, TopicTitleRecHeader.dwReserved2);
      printf("%sdwNextTopicOffs = %08lX\n",
        indent, TopicTitleRecHeader.dwNextTopicOffset);
      printf("%sdwNextTopicPtr  = %08lX\n",
        indent, TopicTitleRecHeader.dwNextTopicPointer);
      printf("%sdwTopicNonScroll= %08lX\n",
        indent, TopicTitleRecHeader.dwTopicNonScroll);
      printf("%sdwTopicData     = %08lX\n",
        indent, TopicTitleRecHeader.dwTopicData);

      //-- Report on title --
      wDataSize = (WORD)(TopicRecHeader.dwRecordSize-TopicRecHeader.dwDataOffset);
      topic_fseek(&ifs, pNext+TopicRecHeader.dwDataOffset);
      topic_fread(&ifs, Tmp, wDataSize);
      TranslatePhrase(Title,0x1000,Tmp,wDataSize);
      Title[TopicRecHeader.dwDataSize] = '\0';
      printf("%s<title>         = \"%s\"\n", indent, Title);
    }

    else
    if (TopicRecHeader.bRecordType == TOPICDATAID || TopicRecHeader.bRecordType == TOPICTABLEID )
    {
      //-----------------------------------------------
      //-- This is a type 0x20 or 0x23 topic data record.
      //-- Display segment attribute table, then
      //--  segments themselves.
      //-- Be sure to report on any unusual attributes!
      //-----------------------------------------------

      //-- Read in unknown bytes --
      topic_fread(&ifs, &bUnknown0, sizeof(BYTE));
      topic_fread(&ifs, &bUnknown1, sizeof(BYTE));
      printf("%sbUnknown0       = %02X\n",
        indent, bUnknown0);

      //-- Read in twice data --
      //-- If lsb is set, read in a word, else read a byte --

      topic_fread(&ifs, &bTwiceData, sizeof(BYTE));
      bTwiceDataHi = 0;
      if (bTwiceData & 0x01)
        topic_fread(&ifs, &bTwiceDataHi, sizeof(BYTE));
      wTwiceData = (bTwiceDataHi << 8) + bTwiceData;
      printf("%sbUnknown1       = %02X\n",
        indent, bUnknown1);
      printf("%swTwiceData      = %04X\n",
        indent, wTwiceData);

      //-- Read column count.  For type 0x20, should be 0 --
      topic_fread(&ifs, &bColumnCount, sizeof(BYTE));
      printf("%sbColumnCount    = %02X\n",
        indent, bColumnCount);

      //-- Read unknown byte --
      topic_fread(&ifs, &bUnknown2, sizeof(BYTE));
      printf("%sbUnknown2       = %02X\n",
        indent, bUnknown2);



      //-- If this is a type 0x23, read table of column widths --
      if (TopicRecHeader.bRecordType == TOPICTABLEID)
      {
        for (i=0; i<bColumnCount; i++)
        {
          topic_fread(&ifs, &dwColumnWidth, sizeof(DWORD));
          printf("%sdwColWidth[%02X]  = %08lX\n",
            indent, i, dwColumnWidth);
        };
      };

      //-- For each column, display text data --


      fMore = 1;
      while (fMore)
      {
        if (TopicRecHeader.bRecordType == TOPICTABLEID)
        {
          //-- Read type 0x23 column header --
          //-- A column number of -1 indicates end of list --

          topic_fread(&ifs, &wColumnNo, sizeof(WORD));
          if (wColumnNo != 0xFFFF)
          {
            topic_fread(&ifs, &wUnknown, sizeof(WORD));
            topic_fread(&ifs, &wUnknown1, sizeof(WORD));
            topic_fread(&ifs, &bUnknown1, sizeof(BYTE));
            printf("%swColumnNo       = %04X\n",
              indent, wColumnNo);
            printf("%s  wUnknown      = %04X\n",
              indent, wUnknown );
            printf("%s  wUnknown1     = %04X\n",
              indent, wUnknown1 );
            printf("%s  bUnknown1     = %02X\n",
              indent, bUnknown1);
          }
          else
            fMore = 0;
        };

        if (fMore)
        {
          //-- Display paragraph attribs --
          ReadParAttribs(&ifs);


          //-- Report on topic text with attributes --
          //-- At this point:
          //    Ifs is pointing to start of topic attribute data
          //    Ifs_dup is pointing to start of textual data.
          printf("%sText:\n        ", indent);
          DisplayText(&ifs_dup, &ifs);

        };

        //-- If this is a type 0x20, then only 1 column --
        if (TopicRecHeader.bRecordType == TOPICDATAID)
          fMore = 0;

      }; //end of while

      dwHotspotPointer += wTwiceData/2;

    }

    else

    {
      //-----------------------------------------------
      //-- This is an unknown data record.
      //-----------------------------------------------

      printf("%s**** UNKNOWN TOPIC TYPE %02X ****\n\n",
        TopicRecHeader.bRecordType);

    };

    pPrev = pNext;
    pNext = TopicRecHeader.dwNext;
    if (pPrev == pNext) {
      printf("**** ERROR: Infinite loop at %08lX! ****\n", pPrev);
      pNext = 0xFFFFFFFF;
      };

    //-- Check if on a new page and set hotspot headers --
    if ((pNext & 0xFFFFC000)>>2 != dwHotspotPage)
    {
      dwHotspotPage = (pNext & 0xFFFFC000) >> 2;
      dwHotspotPointer = 0;
    };

    //GROTS: DEBUG CODE TO STOP RUNAWAY POINTERS!
    //if (pNext > 0x00FFFFFF) {
    //  printf("*** ERROR: Suspicious pointer detected at %08lX! ***\n", pNext);
    //  pNext = 0xFFFFFFFF;
    //  };

  };

  return 0;

}


//--------------------------------------------------------------------
//--------------------------------------------------------------------

//Read in paragraph attributes and display them,
// along with any additional parameters for the attributes.

void ReadParAttribs(TOPICIFS_INFO *ifs)
{

  DWORD dwParAttribs;
  BYTE bLftMar;
  BYTE bRgtMar;
  BYTE bParInd;
  BYTE bLineSpacing;
  BYTE bSpaceBefore;
  BYTE bSpaceAfter;
  BYTE bBorder;
  WORD wBorderParam;

  topic_fread(ifs, &dwParAttribs, sizeof(DWORD));

  printf("    dwParAttribs    = %08lX\n", dwParAttribs);

  //-- Read extra parameters based on bits set in dwParAttribs --
  //-- NOTE THAT THE ORDER OF THESE IS VERY IMPORTANT! --

  if (dwParAttribs & PAR_SPACEBEFORE)
  {
    topic_fread(ifs, &bSpaceBefore, sizeof(BYTE));
    printf("      Par SpcBefore = %02X\n", bSpaceBefore);
  };

  if (dwParAttribs & PAR_SPACEAFTER)
  {
    topic_fread(ifs, &bSpaceAfter, sizeof(BYTE));
    printf("      Par SpcAfter  = %02X\n", bSpaceAfter);
  };

  if (dwParAttribs & PAR_LINESPACING)
  {
     topic_fread(ifs, &bLineSpacing, sizeof(BYTE));
    printf("      Par LnSpacing = %02X\n", bLineSpacing);
  };

  if (dwParAttribs & PAR_LEFTINDENT)
  {
    topic_fread(ifs, &bLftMar, sizeof(BYTE));
    printf("      Par LftMar    = %02X\n", bLftMar);
  };

  if (dwParAttribs & PAR_RIGHTINDENT)
  {
    topic_fread(ifs, &bRgtMar, sizeof(BYTE));
    printf("      Par RgtMar    = %02X\n", bRgtMar);
  };

  if (dwParAttribs & PAR_FIRSTINDENT)
  {
    topic_fread(ifs, &bLftMar, sizeof(BYTE));
    printf("      Par Indent    = %02X\n", bParInd);
  };


  if (dwParAttribs & PAR_BORDERED)
  {
    topic_fread(ifs, &bBorder, sizeof(BYTE));
    topic_fread(ifs, &wBorderParam, sizeof(WORD));
    printf("      Border        = %02X [", bBorder);
    if (bBorder & BORDER_DOT)       printf(" Dot ");
    if (bBorder & BORDER_DOUBLE)    printf(" Dbl ");
    if (bBorder & BORDER_THICK)     printf(" Thck ");
    if (bBorder & BORDER_RIGHT)     printf(" Rgt ");
    if (bBorder & BORDER_BOTTOM)    printf(" Bot ");
    if (bBorder & BORDER_LEFT)      printf(" Lft ");
    if (bBorder & BORDER_TOP)       printf(" Top ");
    if (bBorder & BORDER_BOX)       printf(" Box ");
    printf("]\n      Border par    = %04X\n", wBorderParam);
  };

}



//--------------------------------------------------------------------
//--------------------------------------------------------------------

//Display text with attributes
//Pass: IFS pointers to duplicate IFSs.
//      TopicIfs contains the IFS that is pointing to the start of the
//        topic data to dump.
//      AttribIfs contains the IFS that is pointing to the start of the
//        attributes to apply to that topic.
//Also does phrase expansion on the text.

void DisplayText(TOPICIFS_INFO *pTopicIfs, TOPICIFS_INFO *pAttribIfs)
{
  BYTE bText;
  BYTE bAttrib;
  int fMore = 1;
  int seglen = 0;
  WORD wParam;
  DWORD dwParam;
  BYTE bPhrase;
  unsigned int phrase;
  unsigned char *pphrase;
  int plen;

  #define MAXWIDTH 60

  while (fMore)
  {
    topic_fread(pTopicIfs, &bText, sizeof(unsigned char));

    if (bText == '\0')
    {
      topic_fread(pAttribIfs, &bAttrib, sizeof(BYTE));
      switch(bAttrib)
      {

        case SEG_FONT:
          topic_fread(pAttribIfs, &wParam, sizeof(WORD));
          printf("\n        <Font %d>", wParam);
          break;

        case SEG_NEWLINE:
          printf("\n        <NewLine>");
          break;

        case SEG_PAR:
          printf("\n        <NewPar>");
          break;

        case SEG_TAB:
          printf("\n        <Tab>");
          break;

        case SEG_ENDLINK:
          printf("\n        <End of hotlink>");
          break;

        case SEG_POPLINK:
          topic_fread(pAttribIfs, &dwParam, sizeof(DWORD));
          printf("\n        <Popup hotlink; hash = %08lX>", dwParam);
          break;

        case SEG_HOTLINK:
          topic_fread(pAttribIfs, &dwParam, sizeof(DWORD));
          printf("\n        <Standard hotlink; hash = %08lX>", dwParam);
          break;

        case SEG_END:
          printf("\n        <**End**>\n\n");
          fMore = 0;
          break;

        default:
          printf("\n        <****  WARNING: UNKNOWN ATTRIBUTE %02X  ****>", bAttrib);

        };

      seglen=MAXWIDTH;

    }
    else if (pPhrases && bText < ' ')
    {
      topic_fread(pTopicIfs, &bPhrase, sizeof(BYTE));

      phrase = ((bText-1)*256 + bPhrase ) / 2;
      pphrase = pPhrases+pPhraseTable[phrase];
      plen = pPhraseTable[phrase+1]-pPhraseTable[phrase];
      while (plen--)
      {
        if (seglen == MAXWIDTH)
        {
          seglen = 0;
          printf("\n      ");
        };
        printf("%c", *pphrase++);
        seglen++;
      };

      if (bPhrase & 0x01)
      {
        if (seglen == MAXWIDTH)
        {
          seglen = 0;
          printf("\n      ");
        };
        printf(" ");
        seglen++;
      };
    }
    else
    {
      if (seglen == MAXWIDTH)
      {
        seglen = 0;
        printf("\n        ");
      };

      printf("%c", bText);
      seglen++;
    };


  };


};



//********************************************************************
//
//This procedure is used to read the contents of the Phrases subfile
//
// The Phrases file consists of a simple list of phrases, indexed by
//  the phrase number.
// It consists of:
//	the PHRASESHEADER structure, followed by
//	a long value indicating the desired size of the uncompressed
//	  phrases (used only if phrases are compressed), followed by
//	a table of phrase pointers (offsets to the phrases from the
//	  start of the table -- note that is 1 extra entry pointing to
//        the end of the phrases), followed by
//	the phrases themselves, which may be compressed.
//
//The structures used are defined in helpfile.h:
//
//typedef struct {
//  WORD wPhraseCount;			//# of phrases in table
//  WORD wReserved100;			//Reserved: 0x0100
//  } PHRASESHEADER;
//
//********************************************************************

int ReadPhrasesFile(FILE *fp, WORD fCompressed)
{
  DWORD pFile;
  int bytesread;

  DWORD dwRawSize;			//Size of raw phrases
  DWORD dwRequiredSize;			//Size required by phrases
  WORD wPhraseTableSize;		//Size required for phrase table
  IFS_INFO ifs;
  int i;

  char *pPhrase;
  static char szPhrase[512];		//Individual phrase
  WORD wPhraseSize;			//Size of individual phrase

  #ifdef SHOWPHRASES
  printf("Reading phrases file...\n");
  #endif

  fCompressed &= 0x04;			//Mask compression flag.


  pPhraseTable = NULL; pRawPhrases = NULL; pPhrases = NULL;
  if (!ifs_fopen(fp, &ifs, "Phrases"))
  {
    printf("No Phrases file found...\n");
    return 0;
  };

  bytesread = ifs_fread(&ifs, &PhraseHeader, sizeof(PHRASESHEADER));
  #ifdef SHOWPHRASES
  printf("  Phrase count:         %04X\n", PhraseHeader.wPhraseCount);
  printf("  Reserved:             %04X\n", PhraseHeader.wReserved100);
  #endif

  //-- If this is a compressed file, read in the size of the uncompressed
  //	phrases, otherwise calculate it.
  //	Also, calculate size of compressed phrases --

  dwRequiredSize = dwRawSize = ifs.dwFileSize - sizeof(PHRASESHEADER);
  if (fCompressed)
    bytesread = ifs_fread(&ifs,&dwRequiredSize, sizeof(DWORD));
  wPhraseTableSize = (PhraseHeader.wPhraseCount+1) * sizeof(WORD);
  #ifdef SHOWPHRASES
  printf("  Raw data size:        %08lX\n", dwRawSize);
  printf("  Uncompressed size:    %08lX\n", dwRequiredSize);
  #endif

  //-- Allocate space for and read in phrase index table --

  pPhraseTable = (WORD *)malloc(wPhraseTableSize);
  if (pPhraseTable == NULL) {
    printf("Not enough memory to read phrase table!\n");
    ifs_fclose(&ifs);
    return 1;
    };
  bytesread = ifs_fread(&ifs, pPhraseTable, wPhraseTableSize);
  #ifdef SHOWPHRASES
  printf("  Phrase Table at:      %p\n", pPhraseTable);
  printf("  Phrase Table size:    %04X\n", wPhraseTableSize);
  #endif

  //-- Because the phrase table contains pointers to phrases as offset
  //	from start of table itself, we have to subtract wPhraseTableSize
  //	from each entry to get an offset from the start of the phrases --

  for (i=0; i<PhraseHeader.wPhraseCount+1; i++)
    pPhraseTable[i] -= wPhraseTableSize;


  //-- Now, allocate space for raw data and read it in --

  pRawPhrases = (BYTE *)malloc((WORD)dwRawSize);
  if (pRawPhrases == NULL) {
    printf("Error -- not enough memory to read in Phrases!\n");
    free(pPhraseTable);
    ifs_fclose(&ifs);
    return 1;
    };
  bytesread = ifs_fread(&ifs, pRawPhrases,(WORD)dwRawSize);
  #ifdef SHOWPHRASES
  printf("  Raw phrase data at:   %p\n", pRawPhrases);
  #endif

  ifs_fclose(&ifs);

  //-- If we are compressed, allocate space for uncompressed phrases
  //    and do the decompression, otherwise just use the same pointer --

  if (fCompressed)
  {
    pPhrases = (char *)malloc((WORD)dwRequiredSize);
    if (pPhrases == NULL)
    {
      printf ("Error -- not enough memory to convert Phrases!\n");
      free(pRawPhrases);
      free(pPhraseTable);
      return 1;
    };
    if (Decompress(pPhrases,(WORD)dwRequiredSize,
		   pRawPhrases, (WORD)dwRawSize) == 0)
    {
      printf ("Error decompressing Phrases!\n");
      free(pPhrases);
      free(pRawPhrases);
      free(pPhraseTable);
      pPhrases = NULL; pRawPhrases = NULL; pPhraseTable = NULL;
      return 1;
    };
  }
  else
  {
    pPhrases = pRawPhrases;
  }


  //-- Now display the phrases --
  #ifdef SHOWPHRASES
  for (i=0; i<PhraseHeader.wPhraseCount; i++)
  {
    //-- get pointer to and size of the phrase --
    pPhrase = pPhrases+pPhraseTable[i];
    wPhraseSize = pPhraseTable[i+1]-pPhraseTable[i];
    //-- copy the phrase --
    strncpy(szPhrase,pPhrase,wPhraseSize);
    //-- 0 terminate the phrase --
    szPhrase[wPhraseSize]='\0';
    //-- now display the phrase --
    printf("  Phrase[%04X] = \"%s\"\n", i, szPhrase);
  };
  #endif

  return 0;

}



//--------------------------------------------------------------------
//--------------------------------------------------------------------

//Release the memory allocated for the Phrase substitution.

int ReleasePhraseTable(void)
{

  //-- Free allocated memory --
  if (!fCompressed)
    free(pPhrases);
  free(pRawPhrases);
  free(pPhraseTable);
  return 0;
}


//--------------------------------------------------------------------
//--------------------------------------------------------------------

//Translate memory to do phrase expansion.
//Pass the pointer to the source buffer and the source length,
// and the pointer to the destination buffer and the destination length.
//
//Phrase substitution is done by detecting a 0x01 to 0x05 and translating
// it and the following byte thus:
//
//    [aa][bb] -->  ((aa-1)*256 + bb)/2
//
//This gives the index into pPhraseTable[]
//
//If the second byte [bb] has a 1 in the low order bit, this is an
// indication that the phrase is followed by a space character.

int TranslatePhrase(unsigned char *pDest, int iDstLen, unsigned char *pSource, int iSrcLen)
{
  unsigned int phrase;
  unsigned char *pphrase;
  int plen;
  int fspace;

  while (iSrcLen)
  {

    if (*pSource == 0x00 || *pSource > 0x05 || pPhrases == NULL)
    {
      if (iDstLen)
      {
        *pDest++=*pSource;
        iDstLen--;
      };
      pSource++;
      iSrcLen--;
    }
    else
    {
      phrase = ( ((*pSource) -1 ) * 256 + *(pSource+1) ) / 2;
      pSource++; iSrcLen--;
      if (phrase < PhraseHeader.wPhraseCount)
      {
        fspace = *pSource&0x01;
        if (iSrcLen) { pSource++; iSrcLen--; };

        pphrase = pPhrases+pPhraseTable[phrase];
        plen = pPhraseTable[phrase+1]-pPhraseTable[phrase];

        while (iDstLen && plen)
        {
          *pDest++ = *pphrase++;
          iDstLen--;
          plen--;
        };

        if (fspace && iDstLen)
        {
          *pDest++ = ' ';
          iDstLen--;
        };
      };
    };

  } /* while */

}






//********************************************************************
//This routine is used to decompress compressed information.
//Pass the pointer to the compressed data and the length of it,
// and the pointer to a buffer to store uncompressed data and length.
//Uses the compression algorigthm described by Pete Davis.
//I.e., source consists of an 8-bit map, followed by 8 codes.  A 0 bit in
// the map indicates that the corresponding code is an unencrypted byte.
// A 1 bit in the map indicates that the corresponding code is a word value
// that indicates a 12-bit offset and 4-bit length.  This points into the
// text that has already been decrypted and is copied to the output.
//
//This looks alittle complicated only because I'm taking care to do
// error checking which probably isn't necessary since all pointers and
// sizes should be valid.
//
//Return 0 if there is an error, otherwise returns the size of the
// decompressed data.
//
//********************************************************************

int Decompress(BYTE *pDest, WORD wDestLen, BYTE *pSource, WORD wSourceLen)
{
  BYTE fCodeByte;
  int fError = 0;
  WORD wCode;
  BYTE *pSrc;
  WORD lSrc;
  int i;
  int DstSize = 0;

  while (wSourceLen && wDestLen)
  {

    fCodeByte = *pSource++;                     //Get code byte

    for (i=0; i<8 && wSourceLen && wDestLen; i++)
    { //-- 8 codes follow code byte --

      if (fCodeByte & 0x01)                     //Get low bit of code byte
      {

        //-- Bit==1 indicates this is a compressed code --
        wCode = *((WORD *)pSource);             //Get code
        pSource+=2;
        if (wSourceLen) wSourceLen--;           //Adjust source pointers
        if (wSourceLen) wSourceLen--;           //Adjust source pointers
        pSrc = pDest - (wCode & 0x0FFF) - 1;    //Decode pointer to stuff
        lSrc = ((wCode>>12) & 0x0F) + 3;        //Decode length of stuff

        while (lSrc)                            //while stuff to copy,
          if (wDestLen) {                       //if room in dest,
            *pDest++=*pSrc++;                   // copy one byte
            wDestLen--;                         // dec dest length
            DstSize++;                          // inc dest size
            lSrc--;                             // dec source length
            }
         else fError++;                         //else mark error!
      } //-- Bit==1 --

      else

      {
        //-- Bit==0 indicates an uncompressed byte code --
        if (wDestLen) {                         //if room in dest,
          *pDest++=*pSource++;                  // copy byte
          wDestLen--;                           // dec dest length
          DstSize++;                            // inc dest size
          if (wSourceLen) wSourceLen--;         //Adjust source pointers
          }
        else fError++;                          //else mark error!
      }; //-- Bit==0 --

      fCodeByte >>= 1;                          //get next coding bit

    }; /* for */

  }; /* while */

  return fError ? 0 : DstSize;

}





//--------------------------------------------------------------------
//This procedure dumps the hex contents of a subfile.
//Pass the file pointer, and the name of the subfile to dump.
//--------------------------------------------------------------------

int HexDump(unsigned char *data, int blen)
{

  int ioff;
  unsigned char *ptr;
  int btmp,i;

  ioff = 0;

  printf("\n");
  while (blen)
  {
    printf("    %04X: ", ioff);
    ptr=data; btmp=blen;
    for (i=0; i<16;i++)
    {
      if (btmp)
      {
        printf("%02X ", *ptr++);
        btmp--;
      }
      else
        printf("-- ");
    };
    printf("    ");
    for (i=0;i<16;i++)
    {
      if (blen && *data>=' ' && *data < 128)
        printf("%c", *data);
      else
        printf(".");

      if (blen)
      {
        data++;
        blen--;
      }
    };
    printf("\n");
    ioff += 16;
  };
  printf("\n");
  return 0;
}






//********************************************************************
//
//TOPIC Read/Write routines
//
//These routines are used to read data from the TOPIC subfile.
//They mirror the ifs_xxx routines except that they read in from the
// file in 4K pages and do decompression of the pages if necessary.
//
//
//********************************************************************

//--- TOPIC subfile IFS file structure ---
//    This structure is filled in whenever a TOPIC subfile is open.
//    It controls reading/writing, etc.
//
//#define RAWSIZE 0x1000                  //Size of raw data
//#define MAXSIZE 0x4000                  //Size of xlated data (max)
//
//typedef struct {
//  IFS_INFO ifs;                         //IFS info for TOPIC subfile
//  DWORD dwPageNumber;                   //Desired page number
//  DWORD dwPagePos;                      //Desired position within page
//  DWORD dwCurrentPage;                  //Current page number
//  DWORD dwPageSize;                     //Size of current page
//  } TOPICIFS_INFO;
//
// cRawData and cTopicData are the buffers for the raw and translated
// topic data and are globally defined.  This means that only a single
// page can be read at a time, even if there are duplicate IFSs.  This is
// okay since the only instance where we do this is when reading text
// and attributes at the same time and these are in the same record.


//--------------------------------------------------------------------
//Open a TOPIC IFS file.
//Opens the file and fills in the IFS_INFO structure.
//Returns FALSE if the file does not exist, TRUE otherwise.
//--------------------------------------------------------------------

int topic_fopen(FILE *fp, TOPICIFS_INFO *topifs, char *pszFilename)
{

  topifs->dwCurrentPage = 0xFFFFFFFF;
  topifs->dwPagePos = topifs->dwPageNumber = 0;
  return ifs_fopen(fp,&topifs->ifs,pszFilename);
}


//--------------------------------------------------------------------
//Close the TOPIC IFS file
//--------------------------------------------------------------------

int topic_fclose(TOPICIFS_INFO *topifs)
{
  return ifs_fclose(&topifs->ifs);
}

//--------------------------------------------------------------------
//Position within the TOPIC file.
//Pass the encoded position which must be translated first.
//Return the new file position.
//We do this whenever we are positioning to a new topic or to a record
// within a topic so that the page can be checked against the current
// page and read in if necessary.  The page actually doesn't get read
// until topic_fread() is called, however, topic_fseek must be called
// to change the page number in memory, otherwise the next page will
// not get read in.
//--------------------------------------------------------------------

DWORD topic_fseek(TOPICIFS_INFO *topifs, DWORD dwPos)
{
  topifs->dwPageNumber = (dwPos & 0xFFFFC000) >> 2;
  topifs->dwPagePos = dwPos & 0x00003FFF;
  return topic_ftell(topifs);
}

//--------------------------------------------------------------------
//Return the current position within the TOPIC file
//--------------------------------------------------------------------

DWORD topic_ftell(TOPICIFS_INFO *topifs)
{
  return (topifs->dwPageNumber << 2) + topifs->dwPagePos;
}

//--------------------------------------------------------------------
//Read data from the IFS file.
//Return the number of bytes of data actually read.
//--------------------------------------------------------------------

DWORD topic_fread(TOPICIFS_INFO *topifs, void *pDest, DWORD dwSize)
{
  DWORD dwrest;
  DWORD dwsize;

  //-- If the page has not been read in, do it now --

  if (topifs->dwPageNumber != topifs->dwCurrentPage)
    topic_readpage(topifs);


  //-- Now copy desired memory from that page --
  //-- be sure to check for record going past end of 4K page --

  dwsize = dwSize;
  dwrest = topifs->dwPageSize - topifs->dwPagePos;
  while (dwsize >= dwrest)
  {
    //-- copy memory to end of page --
    memcpy(pDest, cTopicData+(int)topifs->dwPagePos, (int)dwrest);
    (BYTE *)pDest += dwrest;
    //-- read in next page --
    topifs->dwPageNumber+=0x1000;
    topifs->dwPagePos = sizeof(TOPICHEADER);
    topic_readpage(topifs);
    //-- adjust sizes to read in rest --
    dwsize -= dwrest;
    dwrest = topifs->dwPageSize - topifs->dwPagePos;
  }

  //-- if there is anything left on page, read it --
  if (dwsize)
  {
    memcpy(pDest,cTopicData+(int)topifs->dwPagePos,(int)dwsize);
    topifs->dwPagePos += dwsize;
  };

  return dwSize;

}



//--------------------------------------------------------------------
//Read a page from the TOPIC file.
//Reads the page and decompresses it.
//The page gets stored in cTopicData[]
//--------------------------------------------------------------------

DWORD topic_readpage(TOPICIFS_INFO *topifs)
{
  DWORD bytesread;

  //printf("*** READING PAGE %08lX ***\n", topifs->dwPageNumber);

  //-- Seek to desired page --
  ifs_fseek(&topifs->ifs, topifs->dwPageNumber);

  //-- Read in the 4K page --
  //-- If we are compressed, then we read the page header into cTopicData,
  //    read the rest of the 4K page into cRawData and decompress it into
  //    cTopicData.
  //   If we are not compressed, then we can read the entire 4K page into
  //    cTopicData at once.

  if (fCompressed)
  {
    //---- Read in from compressed file ----

    //-- Read in the page header --
    bytesread = ifs_fread(&topifs->ifs, cTopicData, sizeof(TOPICHEADER));


    //-- Now read in the rest of the 4K page --
    bytesread = ifs_fread(&topifs->ifs, cRawData, RAWSIZE-sizeof(TOPICHEADER));

    //-- Now it must be decompressed --
    topifs->dwPageSize = (DWORD)Decompress(
      cTopicData + sizeof(TOPICHEADER), MAXSIZE - sizeof(TOPICHEADER),
      cRawData, (WORD)bytesread );
  }
  else
  {
    //---- Read in from uncompressed file ----

    bytesread = topifs->dwPageSize
      = ifs_fread(&topifs->ifs, cTopicData, RAWSIZE);
  };


  topifs->dwCurrentPage = topifs->dwPageNumber;
  return (bytesread);

}






//********************************************************************
//
//IFS Read/Write routines
//
//********************************************************************


//--- IFS subfile open file structure --
//    This structure is filled in whenever a subfile is open
//    It controls reading/writing, etc.
//
//  typedef struct {
//    FILE *FilePointer;                    //File pointer for help file
//    DWORD dwFileStart;                    //Starting position for file
//    DWORD dwFileSize;                     //Size for file
//    DWORD dwFilePos;                      //Current read/write pos in file
//    } IFS_INFO;


//--------------------------------------------------------------------
//Open an IFS file.
//Opens the file and fills in the IFS_INFO structure.
//Returns FALSE if the file does not exist, TRUE otherwise.
//--------------------------------------------------------------------

int ifs_fopen(FILE *fp, IFS_INFO *ifs, char *pszFilename)
{
  DWORD pFile;
  FILEHEADER FileHeader;
  int bytesread;

  pFile = FindSubfile(pszFilename);
  if (pFile == 0xFFFFFFFF)
    return 0;

  //-- Position to file --
  fseek(fp, (long)pFile, SEEK_SET);

  //-- Read standard file header --
  bytesread = fread(&FileHeader, sizeof(FILEHEADER), 1, fp);
  pFile+=sizeof(FILEHEADER);

  //-- Fill in the ifs structure --

  ifs->FilePointer = fp;
  ifs->dwFileStart = pFile;
  ifs->dwFileSize = FileHeader.dwFileSize;
  ifs->dwFilePos = 0;
  return 1;
}


//--------------------------------------------------------------------
//Close the IFS file
//--------------------------------------------------------------------

int ifs_fclose(IFS_INFO *ifs)
{
  ifs->FilePointer = NULL;
}

//--------------------------------------------------------------------
//Position within the IFS file.
//If an attempt is made to position past the end of the file, the
// position is set to 1 more than the end of the file (and thus any reads
// will fail).
//--------------------------------------------------------------------

DWORD ifs_fseek(IFS_INFO *ifs, DWORD dwPos)
{
  return (ifs->dwFilePos =
    (dwPos < ifs->dwFileSize) ? dwPos : ifs->dwFileSize );
}

//--------------------------------------------------------------------
//Return the current position within the IFS file
//--------------------------------------------------------------------

DWORD ifs_ftell(IFS_INFO *ifs)
{
  return ifs->dwFilePos;
}


//--------------------------------------------------------------------
//Determine if we are at or past the end of an IFS file
//--------------------------------------------------------------------

int ifs_feof(IFS_INFO *ifs)
{
  return (ifs->dwFilePos >= ifs->dwFileSize);
}

//--------------------------------------------------------------------
//Read data from the IFS file.
//Return the number of bytes of data actually read.
//The maximum we can read is from ifs->dwFilePos to ifs->dwFileSize-1
//--------------------------------------------------------------------

DWORD ifs_fread(IFS_INFO *ifs, void *pDest, DWORD dwSize)
{
  DWORD bytesread;

  if (ifs->FilePointer == NULL) return 0;

  if (ifs->dwFilePos >= ifs->dwFileSize) return 0;

  if (ifs->dwFilePos + dwSize > ifs->dwFileSize)
    dwSize = ifs->dwFileSize - ifs->dwFilePos;

  if (dwSize <= 0) return 0;

  fseek(ifs->FilePointer, ifs->dwFileStart+ifs->dwFilePos, SEEK_SET);
  bytesread = fread(pDest, 1, (int)dwSize, ifs->FilePointer);
  ifs->dwFilePos += bytesread;

  return bytesread;
}



