;-------------------------------- MEMWATCH ---------------------------------
; A TSR to "watch" for all the ways a program might try to allocate memory.
;
; This includes EMS, XMS(UMB's and EMB's), Int 15, DPMI, and VCPI.
; Note: Programs that use higher level methods (like DPMI) may show a 
; "hit" on several different methods at the same time.
;---------------------------------------------------------------------------
; To effectively watch for EMS allocations, we need to do some trickery
; when we hook Int67.  Programs often look for an EMS driver by assuming
; Int67 is aimed into a device driver's code segment and then look into 
; what would be the device header at DDSeg:devicename.  We need to setup
; a "fake" device header they can scan through.
;---------------------------------------------------------------------------
; Things to do:
;
;  - Relocate some code/data into the PSP to reduce resident footprint.
;  - Think about adding usage counts
;  - Think about active PSP "scoped" reporting
;---------------------------------------------------------------------------
; Author: Tony Ingenoso 
;---------------------------------------------------------------------------

;---------------------------------------------------------------------------
; We'll be monitoring these types of memory activities...
;---------------------------------------------------------------------------
MemFlags struc
fEMS32    db    0                       ; EMS 3.2 flag
fEMS40    db    0                       ; EMS 4.0 flag
fEEMS     db    0                       ; EEMS flag
fHMA      db    0                       ; HMA flag
fEMB      db    0                       ; XMS extended memory flag
fUMB      db    0                       ; XMS UMB memory flag
fDOSSTRAT db    0                       ; DOS UMB strategy flag
fDOSLINK  db    0                       ; DOS UMB link flag
fDOSHMA   db    0                       ; DOS managed HMA flag (slack space)
fINT15    db    0                       ; Int 15 memory flag
fDPMI     db    0                       ; DPMI memory flag
fVCPI     db    0                       ; VCPI memory flag
        ends

ShowMessage macro msg
        mov     dx, offset msg
        mov     ah, 9
        int     21h
        endm

memwatch segment para public 'code'
        assume  cs:memwatch
        org     100h                    ; This is a COM file.
start   label   near
        jmp     THROW_AWAY_CODE

;--------------------------------------------------------------------------
; Old vectors/pointers to chain to
;--------------------------------------------------------------------------
        align   4
OldInt15 dd     0                       ; Saved Int 15 vector
OldInt21 dd     0                       ; Saved Int 21 vector
OldInt2F dd     0                       ; Saved Int 2F vector
OldInt67 dd     0                       ; Saved Int 67 vector
OldXMS   dd     0                       ; Saved XMS entry point

;--------------------------------------------------------------------------
; Working flags......
;--------------------------------------------------------------------------
        db      "MEMWATCH"              ; A signature
MemVars MemFlags <>                     ; Flags structure
MemVarsLen = $-MemVars

;--------------------------------------------------------------------------
; The XMS hook to "watch" for XMS calls.
;--------------------------------------------------------------------------
; By taking over the Int 2F that returns the XMS entry point rather than
; patching into the existing driver, nice things happen.  All prior loaded
; TSR's and device drivers that call XMS will go directly to the original
; driver.  This means we won't see a bunch of "noise" that looks like it's
; comming from the app(s) being monitored.  We'll be seeing only his actions.
; We're really not interested in XMS activity from caches, VDISK's, etc.
;--------------------------------------------------------------------------
        align   2
XMShooker proc far
        jmp     short XMS_hookability   ; So we can subsequently be hooked
        nop                             ; out by some other program...  
        nop                             ;
        nop                             ;

XMS_hookability:
        cmp     ah, 1                   ; Trying to allocate HMA?
        jne     CheckAllocEMB           ;
        mov     MemVars.fHMA, 1         ;
        jmp     short ChainXMS          ;

CheckAllocEMB:
        cmp     ah, 9                   ; Trying to allocate EMB?
        jne     CheckAllocUMB           ;
        mov     MemVars.fEMB, 1         ;
        jmp     short ChainXMS          ;

CheckAllocUMB:
        cmp     ah, 10h                 ; Trying to allocate UMB?
        jne     ChainXMS                ;
        mov     MemVars.fUMB, 1         ;

ChainXMS: jmp     [OldXMS]
XMShooker endp

;--------------------------------------------------------------------------
; The 15 hook to "watch" for extended memory moves.
;--------------------------------------------------------------------------
        align   2
Int15hooker proc far
        cmp     ah, 87h                 ; Extended memory move?
        jne     No15                    ;
        mov     MemVars.fINT15, 1      ; Yes, we've seen one
No15:   jmp     [OldInt15]              ; Chain it forward
Int15hooker endp

;--------------------------------------------------------------------------
; The 21 hook to "watch" for UMB activity
;--------------------------------------------------------------------------
        align   2
Int21hooker proc far
        cmp     ax, 5801                ; Set strategy?
        jne     LookForLink             ;

        ;-------------------------------------------------------------
        ; Check the "set" strateg to see if it includes UMB's
        ;-------------------------------------------------------------
        cmp     bl, 40h                 ; Is the allocation
        je      YesStrat                ;  strategy being set
        cmp     bl, 41h                 ;   to something that
        je      YesStrat                ;    will cause an
        cmp     bl, 42h                 ;     allocation to come
        je      YesStrat                ;      from a UMB?
        cmp     bl, 80h                 ;
        je      YesStrat                ;
        cmp     bl, 81h                 ;
        je      YesStrat                ;
        cmp     bl, 82h                 ;
        je      YesStrat                ;

        ;-------------------------------------------------------------
        ; Check the "link" option to see if it's "linking in" UMB's
        ;-------------------------------------------------------------
LookForLink:                            ;
        cmp     ax, 5803h               ; Is he trying to (un)link UMB's?
        jne     No21                    ; Nope...
        cmp     bx, 0001h               ; Add UMB's to chain?
        je      YesLink                 ; Yes,...
        jmp     short No21              ; No,...

YesStrat:
        mov     MemVars.fDOSSTRAT, 1    ; Yes, we've seen one
        jmp     short   No21

YesLink:
        mov     MemVars.fDOSLINK, 1     ; Yes, we've seen one

No21:   jmp     [OldInt21]              ; Chain it forward
Int21hooker endp

;--------------------------------------------------------------------------
;     The 2F hook responds to our presence check and other things....
;--------------------------------------------------------------------------
        align   2
Int2Fhooker proc far
        cmp     ax, 'EM'                ; Int2F presence check?
        jne     TryDOSmanagedHMA        ; We're
        cmp     bx, 'WM'                ;   looking
        jne     TryDOSmanagedHMA        ;     for
        cmp     cx, 'TA'                ;       "MEMWATCH"
        jne     TryDOSmanagedHMA        ;         in the
        cmp     dx, 'HC'                ;           AX:BX:CX:DX
        jne     TryDOSmanagedHMA        ;             registers

        ;------------------------------------------------------------------
        ; Int2F was called with all the right signature bytes...
        ; We will return the address of the memory flags in ES:DI and flip
        ; all the bits in the signature that was passed to us to verify
        ; that it was indeed us who responded.
        ;------------------------------------------------------------------
        not     ax                      ; NOT the signature, so
        not     bx                      ;   the caller can have a 
        not     cx                      ;     better chance of knowing it
        not     dx                      ;       was us who responded...
        mov     di, offset MemVars      ; ES:DI--> addr of flags
        push    cs                      ;
        pop     es                      ;
        iret                            ;

        ;------------------------------------------------------------------
        ;              Check for DOS managed HMA allocation
        ;------------------------------------------------------------------
TryDOSmanagedHMA:
        cmp     ax, 4A02h               ; Allocate?
        jne     TryXMS                  ;
        mov     MemVars.fDOSHMA, 1      ; Set the DOS managed HMA flag
        jmp     short Chain2F           ; Chain forward...

        ;------------------------------------------------------------------
        ;      Get driver address API?  (we return ours instead...)
        ;------------------------------------------------------------------
TryXMS:
        cmp     ax, 4310h               ; DOS managed HMA alloc?
        jne     TryDPMI                 ; No,...

        push    cs                      ; Return our monitor's address
        pop     es                      ;  rather than the original.
        mov     bx, offset XMShooker    ;
        iret

        ;------------------------------------------------------------------
        ;                   DPMI installation check?
        ;------------------------------------------------------------------
TryDPMI:
        cmp     ax, 1687h               ; DPMI installed check?
        jne     Chain2F
        mov     MemVars.fDPMI, 1        ; Set the DPMI flag

Chain2F:jmp     [OldInt2F]              ; Chain forward...
Int2Fhooker endp

;--------------------------------------------------------------------------
;               Monitor for EMS32/EMS40/EEMS/VCPI activity
;--------------------------------------------------------------------------
        align   2
Int67hooker proc far
        cmp     ah, 40h                 ; <= EMS/EEMS/VCPI range?
        jb      Chain67                 ;
        cmp     ah, 6Ah                 ; > EEMS range?
        ja      CheckVCPI               ;

CheckEMS32:
        cmp     ah, 4Eh                 ; EMS 3.2 range?
        ja      CheckEMS40              ; No,...
        mov     MemVars.fEMS32, 1       ; Set EMS 3.2 flag
        jmp     short Chain67           ;

CheckEMS40:
        cmp     ah, 5Dh                 ; EMS 4.0 range?
        ja      CheckEEMS               ; No,...
        mov     MemVars.fEMS40, 1       ; Set EMS 4.0 flag
        jmp     short Chain67           ;

CheckEEMS:
        cmp     ah, 60h                 ; Start of EEMS range
        jb      Chain67                 ; There are 2 unspec'd function #'s
        mov     MemVars.fEEMS, 1        ; Set EEMS flag
        jmp     short Chain67           ;
        
CheckVCPI:
        cmp     ax, 0DE00h              ; Min VCPI function number
        jb      Chain67                 ; Below?
        cmp     ax, 0DE0Ch              ; Max VCPI function number
        ja      Chain67                 ; above?
        mov     MemVars.fVCPI, 1        ; Set VCPI flag

Chain67:jmp     [OldInt67]
Int67hooker endp

;--------------------------------------------------------------------------
; Fake EMS device driver header and JMP into this TSR's space.
;--------------------------------------------------------------------------
        align   16
FakeEMSheader   db      18 dup (0)
FakeJMPtoTSR label far
        .errnz  (FakeJMPtoTSR - FakeEMSheader - 18)
        db      0EAh                    ; Direct inter-seg JMP opcode
        dw      offset Int67hooker      ;
CSpatch dw      ?                       ; Filled in with CS at init time
        .errnz  (CSpatch - FakeJMPtoTSR - 3)

        align   16
THROW_AWAY_CODE label   near
        jmp     Initialize

;---------------------------------------------------------------------
; Zero out all the working variables.  ES:DI-->variables
;---------------------------------------------------------------------
ClearMemVars proc near
        push    di                      ; Save some
        push    cx                      ;   registers
        push    ax                      ;     we'll
        pushf                           ;       clobber

        xor     ax, ax                  ; Zero for REP STOS
        mov     cx, MemVarsLen          ; Byte count
        cld                             ; Just to be sure...
        rep     stosb                   ; Wham!

        popf                            ; Restore
        pop     ax                      ;   the 
        pop     cx                      ;     registers
        pop     di                      ;       we clobbered
        ret                             ; Bye.
ClearMemVars endp

;---------------------------------------------------------------------
; Save old vectors and XMS entry point
;---------------------------------------------------------------------
SaveVectors proc near
        mov     ax, 3515h               ; Save Int 15 vector
        int     21h                     ;
        mov     word ptr OldInt15, bx   ;
        mov     word ptr OldInt15+2, es ;

        mov     ax, 3521h               ; Save Int 21 vector
        int     21h                     ;
        mov     word ptr OldInt21, bx   ;
        mov     word ptr OldInt21+2, es ;

        mov     ax, 352Fh               ; Save Int 2F vector
        int     21h                     ;
        mov     word ptr OldInt2F, bx   ;
        mov     word ptr OldInt2F+2, es ;

        mov     ax, 3567h               ; Save Int 67 vector
        int     21h                     ;
        mov     word ptr OldInt67, bx   ;
        mov     word ptr OldInt67+2, es ;

        mov     ax, 4300h               ; XMS install check
        int     2Fh                     ;
        cmp     al, 80h                 ; XMS driver here?
        jne     NoXMSdriver             ;

        mov     ax, 4310h               ; Get XMS driver entry point
        int     2Fh                     ;
        mov     word ptr OldXMS, bx     ; Save XMS driver entry point
        mov     word ptr OldXMS+2, es   ;
NoXMSdriver:
        ret
SaveVectors endp

;---------------------------------------------------------------------
; "dummy up" the fake EMS device driver header and take Int 67
;---------------------------------------------------------------------

EMSname	db	'EMMXXXX0'

SpliceInFakeEMSdriver proc near
        mov     ds, word ptr OldInt67+2 ; DS:SI --> EMS device header
        xor     si, si                  ;
        mov     cx, 18                  ; 18 bytes in DD header
        push    cs                      ; ES:DI --> Our "fake" header
        pop     es                      ;
        mov     di, offset FakeEMSheader;
        cld                             ;
        rep     movsb                   ; Copy the header.

TakeInt67Vector label near
        mov     CSpatch, cs             ; Fixup this address now

        ;------------------------------------------------------------------
        ; Construct a new pointer for Int 67 such that when the segment
        ; value is considered as the base of a device driver, SEG:10d will
        ; point to the device name.  This allows programs groping in the
        ; device header for 'EMMXXXX0' to find it!
        ;------------------------------------------------------------------
        mov     dx, offset FakeEMSheader; DX == offset
        mov     cl, 4                   ; Para normalize
        shr     dx, cl                  ;
        mov     bx, cs                  ; Add paras to current CS
        add     dx, bx                  ; DX == SEG of fake DD header
        mov     ds, dx                  ; DS == SEG of fake DD header
        mov     dx, (FakeJMPtoTSR - FakeEMSheader)

        mov     ax, 2567h               ; DS:DX == rationalized addr of DD
        int     21h                     ; Take Int67

        ret
SpliceInFakeEMSdriver endp

;---------------------------------------------------------------------
; Take over a bunch of vectors...
;---------------------------------------------------------------------
HookVectors proc near
        les     bx, OldInt67            ; Int 67 vector non-zero?
        mov     ax, es                  ;
        or      ax, bx                  ; Null?
        jz      Null67                  ; Yes, do nothing...

        cmp     byte ptr es:[bx], 0CFh  ; Aimed at IRET?
        je      Null67                  ; Yes, do nothing...

        ;-------------------------------------------------------------
        ; Is EMS driver present?
        ;-------------------------------------------------------------
        mov     di, 10                  ; ES:DI --> dev name in header
        push	cs                      ;
        pop     ds                      ;
        mov     si, offset EMSname      ;
        cld                             ;
        mov     cx, 4                   ; 4 words
        rep     cmpsw                   ;
        je      SpliceEMS               ; EMS driver is present

        call    TakeInt67Vector         ; No EMS driver present.  VCPI only.
        jmp     short Null67            ;

SpliceEMS:
        call    SpliceInFakeEMSdriver   ;

Null67:
        push    cs                      ; Grab Int 15
        pop     ds                      ;
        mov     dx, offset Int15hooker  ;
        mov     ax, 2515h               ;
        int     21h                     ;

        push    cs                      ; Grab Int 21
        pop     ds                      ;
        mov     dx, offset Int21hooker  ;
        mov     ax, 2521h               ;
        int     21h                     ;

        push    cs                      ; Grab Int 2F
        pop     ds                      ;
        mov     dx, offset Int2Fhooker  ;
        mov     ax, 252Fh               ;
        int     21h                     ;

        ret
HookVectors endp

;---------------------------------------------------------------------------
; Am I already loaded?  If so, report activity so far and reset.
; If not, then go resident and start watching.
;---------------------------------------------------------------------------
ToImNotLoaded: jmp ImNotLoaded          ; Ignore this...(bridge JMP)

MsgFlag db      0

Initialize label near
        xor     di, di                  ;
        mov     es, di                  ;
        mov     ax, 'EM'                ;
        mov     bx, 'WM'                ;
        mov     cx, 'TA'                ;
        mov     dx, 'HC'                ;
        int     2Fh                     ;
        cmp     ax, NOT 'EM'            ;
        jne     ToImNotLoaded           ;
        cmp     bx, NOT 'WM'            ;
        jne     ToImNotLoaded           ;
        cmp     cx, NOT 'TA'            ;
        jne     ToImNotLoaded           ;
        cmp     dx, NOT 'HC'            ;
        jne     ToImNotLoaded           ;

;----------------------------------------------------------------------
; Report on what has been seen so far... ES:DI-->usage flags
;----------------------------------------------------------------------
        push    es                      ; Save pointer to flags...
        push    di                      ;

        push    cs                      ; DS == CS
        pop     ds                      ;

        ShowMessage     ReportMsg       ; Show report title

        cmp     es:[di].fHMA, 0         ; HMA activity?
        je      TryEMS32
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgHMA

TryEMS32:
        cmp     es:[di].fEMS32, 0       ; EMS 3.2 activity?
        je      TryEMS40
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgEMS32

TryEMS40:
        cmp     es:[di].fEMS40, 0       ; EMS 4.0 activity?
        je      TryEEMS
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgEMS40

TryEEMS:
        cmp     es:[di].fEEMS, 0        ; EEMS activity?
        je      TryEMB
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgEEMS

TryEMB:
        cmp     es:[di].fEMB, 0         ; EMB activity?
        je      TryUMB
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgEMB

TryUMB:
        cmp     es:[di].fUMB, 0         ; XMS UMB activity?
        je      TryDosStrat
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgXUMB

TryDosStrat:
        cmp     es:[di].fDOSSTRAT, 0    ; DOS alloc strat UBM activity?
        je      TryDosLink
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgDSTRAT

TryDosLink:
        cmp     es:[di].fDOSLINK, 0     ; DOS UMB link activity?
        je      TryInt15
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgDLINK

TryInt15:
        cmp     es:[di].fINT15, 0       ; Int15 extended memory activity?
        je      TryDPMIx
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgINT15

TryDPMIx:
        cmp     es:[di].fDPMI, 0        ; DPMI activity?
        je      TryVCPI
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgDPMI

TryVCPI:
        cmp     es:[di].fVCPI, 0        ; VCPI activity?
        je      Terminate
        inc     MsgFlag                 ; Make a note.
        ShowMessage     MsgVCPI

Terminate: 
        cmp     MsgFlag, 0              ; Any non-low activity seen?
        jne     SkipDefaultMessage      ; Nope,...
        ShowMessage     MsgNothing      ; Yes.

SkipDefaultMessage:
        pop     di                      ; Restore pointer to flags...
        pop     es                      ;
        call    ClearMemVars            ; Reset flags

        mov     ax, 4C00h               ; Terminate (RC==0)
        int     21h                     ;
        jmp     $                       ; Should never get here...

;----------------------------------------------------------------------
; Setup to go resident.
;----------------------------------------------------------------------
ImNotLoaded:
        ;
        ; Do we need to free the environment?
        ;
        mov     cx, cs:[2Ch]           	; CX == Env pointer
        jcxz    NoEnvironment          	;

        mov     es, cx                 	; ES == Env seg
        mov     ah, 49h                	; Free the environment
        int     21h                    	;				    
NoEnvironment:

        ;
        ; Clear the monitor flags.
        ;
        push    cs                      ;
        pop     es                      ;
        mov     di, offset MemVars      ;
        call    ClearMemVars            ; Zero all the flags

        ;
        ; Hook in...
        ;
        call    SaveVectors             ;
        call    HookVectors             ;

        ;
        ; Say that we're loaded... 
        ;
        push    cs                      ; Show "loaded" message
        pop     ds                      ;
        ShowMessage     LoadedMsg       ;

        ;
        ; Go resident
        ;
        mov     dx, offset THROW_AWAY_CODE
        int     27h                     ;
        jmp     $                       ; Should never get here...

;-----------------------------------------------------------------------
;                        Translatable text
;-----------------------------------------------------------------------
LoadedMsg db    "MEMWATCH V0.1(beta) - Memory Allocation Monitor",13,10
        db      "Run MEMWATCH again to see usage report(this resets monitor flags too)",13,10,'$'

ReportMsg db    "------------------------ Memory Activity Report -----------------------",13,10,'$'

MsgEMS32 db     "Expanded Memory (EMS) 3.2 activity noted",13,10,'$'
MsgEMS40 db     "Expanded Memory (EMS) 4.0 activity noted",13,10,'$'
MsgEEMS  db     "Enhanced Expanded Memory (EEMS) activity noted",13,10,'$'
MsgEMB  db      "XMS Extended Memory Block (EMB) activity noted",13,10,'$'
MsgHMA  db      "XMS High Memory Area (HMA) activity noted",13,10,'$'
MsgXUMB db      "XMS Upper Memory (UMB) activity noted",13,10,'$'
MsgDOSHMA db    "DOS managed HMA slack space activity noted",13,10,'$'
MsgDSTRAT db    "DOS allocation strategy was altered to include UMBs in allocations",13,10,'$'
MsgDLINK db     "DOS allocation strategy was altered to link in UMBs",13,10,'$'
MsgINT15 db     "BIOS Int 15 extended memory activity noted",13,10,'$'
MsgDPMI db      "App checked for DPMI server",13,10,'$'
MsgVCPI db      "App did VCPI activity",13,10,'$'
MsgNothing db   "*** Nothing but normal DOS low memory activity has been noted ***",13,10,'$'

memwatch ends
        end     start
