// Pre-packaged DDE Client class that encapsulates DDE and combines some functions
// See "ddemgr.h" for instructions for use
// Intellectual property of Lance Klein 11/93

#include "ddemgr.h"

DDEManager::DDEManager(HINSTANCE Inst/* = NULL */, void * clientthis/* = NULL */,
				char * DDEService/* = NULL */, char * DDETopic/* = NULL */)
{
	hConv = 0;
	hInst = NULL;
	idInst = 0;
	RetData = NULL;
	LenRetData = 0;
	lpCallBack = NULL;
	hszService = hszTopic = hszItem = 0;
	pClientReceiveProc = NULL;
	ClientThis = NULL;
	IsInAdviseLoop = IsAdviseData = FALSE;
	errortext[0] = '\0';
	if( Inst && Initialize(Inst, clientthis) && DDEService && DDETopic )
		Connect(DDEService, DDETopic);
}


DDEManager::~DDEManager()
{
	if( hConv )
		Disconnect();
	if( idInst )
		DdeUninitialize(idInst);
}


// Get a single piece of data from the server
char * DDEManager::GetDDEItem(char * DDEService, char * DDETopic, char * DDEItem,
				char * retdata/* = NULL */, int lenretdata/* = 0 */,
				HINSTANCE Inst/* = NULL */, void * clientthis/* = NULL */)
{
	// Perform all functions to start conversation and get one item of data
	if( Inst && !Initialize(Inst, clientthis) )
		return NULL;  
	if( !Connect(DDEService, DDETopic) )
		return NULL;

	RetData = Request(DDEItem, retdata, lenretdata);

	Disconnect();

	return RetData;
}


// Initialize DDEML library
BOOL DDEManager::Initialize(HINSTANCE Inst, void * clientthis/* = NULL */)
{
	if( idInst )
		return TRUE;

	hInst = Inst;
	if( clientthis )
		ClientThis = clientthis;
	idInst = 0;		// Must be 0 first call to DdeInitialize

	if( !(lpCallBack = MakeProcInstance( (FARPROC) DDEManager::CallBack, Inst)) )
	{
		strcpy(errortext, "MakeProcInstance");
		return FALSE;
	}

	if( DdeInitialize( &idInst, (PFNCALLBACK) lpCallBack, APPCMD_CLIENTONLY, 0) !=
		DMLERR_NO_ERROR )
	{
		strcpy(errortext, "DdeInitialize");
		return FALSE;
	}

	return TRUE;
}


// Initiate conversation with the server
BOOL DDEManager::Connect(char * DDEService, char * DDETopic,
				void * clientthis/* = NULL */, HINSTANCE Inst/* = NULL */)
{
	if( Inst && !Initialize(Inst) )
		return FALSE;

	if( clientthis )
		ClientThis = clientthis;

	// Must call Initialize first
	if( !idInst )
	{
		strcpy(errortext, "Not Initialized");
		return FALSE;
	}

	// Must Disconnect previous connection first
	if( hConv )
	{
		strcpy(errortext, "Already Connected");
		return FALSE;
	}

	// Get string handles for service and topic
	if( !(hszService = DdeCreateStringHandle(idInst, DDEService, CP_WINANSI)) )
	{
		strcpy(errortext, "DdeCreateStringHandle");
		return FALSE;
	}
	if( !(hszTopic = DdeCreateStringHandle(idInst, DDETopic, CP_WINANSI)) )
	{
		strcpy(errortext, "DdeCreateStringHandle");
		return FALSE;
	}

	// Connect to DDE Server
	if( !(hConv = DdeConnect(idInst, hszService, hszTopic, NULL)) )
	{
		strcpy(errortext, "DdeConnect");
		return FALSE;
	}

	// Register static "this" pointer for callback function
	if( !TServer )
		TServer = new ThisServer;
	if( TServer )
		TServer->RegisterThis(this, hConv);

	return TRUE;
}


// Request data from the server
char * DDEManager::Request(char * DDEItem, char * retdata/* = NULL */,
				int lenretdata/* = 0 */)
{
	if( !hConv )
	{
		strcpy(errortext, "Not Connected");
		return NULL;
	}

	strcpy(errortext, "");
	RetData = retdata;
	LenRetData = lenretdata;

	// Get string handle for item (free old one first)
	if( hszItem )
	{
		DdeFreeStringHandle(idInst, hszItem);
		hszItem = 0;
	}
	if( !(hszItem = DdeCreateStringHandle(idInst, DDEItem, CP_WINANSI)) )
	{
		strcpy(errortext, "DdeCreateStringHandle");
		return NULL;
	}

	// Initiate client transaction to get data item
	HDDEDATA hData = DdeClientTransaction(NULL, 0, hConv, hszItem, CF_TEXT,
						XTYP_REQUEST, 0, NULL);
	if( hData )
	{
		ReceivedData(hData);
		DdeFreeDataHandle(hData);
	}

	return RetData;
}


// Send data to the server
BOOL DDEManager::Poke(char * DDEItem, char * data)
{
	if( !hConv )
	{
		strcpy(errortext, "Not Connected");
		return FALSE;
	}

	// Get string handle for item
	if( hszItem )
	{
		DdeFreeStringHandle(idInst, hszItem);
		hszItem = 0;
	}
	if( !(hszItem = DdeCreateStringHandle(idInst, DDEItem, CP_WINANSI)) )
		return FALSE;

	// Send data
	return( (DdeClientTransaction((LPBYTE) data, strlen(data)+1, hConv,
    			hszItem, CF_TEXT, XTYP_POKE, 1000, NULL) != FALSE) );
}


// Tell the server to execute a command
BOOL DDEManager::Execute(char * command)
{
	if( !hConv )
	{
		strcpy(errortext, "Not Connected");
		return FALSE;
	}

	// Send data
	return( (DdeClientTransaction((LPBYTE) command, strlen(command),
			hConv, NULL, CF_TEXT, XTYP_EXECUTE, 1000, NULL) != FALSE) );
}


// Register a client advise procedure and optionally start advise loop
BOOL DDEManager::RegisterAdviseProc(adviseproc * clientproc, char * DDEItem/* = NULL */)
{
	if( DDEItem && !hConv )
	{
		strcpy(errortext, "Not Connected");
		return FALSE;
	}
	if( !clientproc )
	{
		strcpy(errortext, "Not Registered");
		return FALSE;
	}

	pClientReceiveProc = clientproc;

	if( DDEItem )
		return BeginAdvise(DDEItem);
	else
		return TRUE;
}


// Notify server to begin advise loop
BOOL DDEManager::BeginAdvise(char * DDEItem)
{
	if( !hConv )
	{
		strcpy(errortext, "Not Connected");
		return FALSE;
	}
	if( IsInAdviseLoop )
	{
		strcpy(errortext, "Already in Advise Loop");
		return FALSE;
	}
	if( !pClientReceiveProc )
	{
		strcpy(errortext, "Not Registered");
		return FALSE;
	}

	if( hszItem )
	{
		DdeFreeStringHandle(idInst, hszItem);
		hszItem = 0;
	}
	if( !(hszItem = DdeCreateStringHandle(idInst, DDEItem, CP_WINANSI)) )
		return FALSE;

	DWORD dwTempResult;
	if( !(BOOL)DdeClientTransaction( NULL, 0, hConv, hszItem, CF_TEXT,
			XTYP_ADVSTART | XTYPF_ACKREQ, 1000, &dwTempResult ) )
		return FALSE;

	return IsInAdviseLoop = TRUE;
}


// Notify Server to end advise loop
BOOL DDEManager::EndAdvise()
{
	if( !hConv )
	{
		strcpy(errortext, "Not Connected");
		return FALSE;
	}
	if( !IsInAdviseLoop )
	{
		strcpy(errortext, "Not in Advise Loop");
		return FALSE;
	}

	DWORD dwTempResult;
	if( !(BOOL)DdeClientTransaction( NULL, 0, hConv, hszItem, CF_TEXT,
			XTYP_ADVSTOP, 1000, &dwTempResult ) )
		return FALSE;

	IsInAdviseLoop = FALSE;

	return TRUE;
}


// Disconnect DDE Conversation 
BOOL DDEManager::Disconnect()
{
	if( !hConv )
	{
		strcpy(errortext, "Not Connected");
		return FALSE;
	}

	if( IsInAdviseLoop )
		EndAdvise();

	// Disconnect from DDE server
	DdeDisconnect( hConv );

	// Free string handles
	if( hszService )
	{
		DdeFreeStringHandle(idInst, hszService);
		hszService = 0;
	}
	if( hszTopic )
	{
		DdeFreeStringHandle(idInst, hszTopic);
		hszTopic = 0;
	}
	if( hszItem )
	{
		DdeFreeStringHandle(idInst, hszItem);
		hszItem = 0;
	}

	// Free callback function pointer
	if( lpCallBack )
	{
		FreeProcInstance((FARPROC) lpCallBack);
		lpCallBack = NULL;
	}
	
	// Un-register pointer from "This" server
	if( TServer )
	{
		TServer->UnRegisterThis(hConv);
		hConv = 0;
		if( !TServer->NumThis() )
		{
			delete TServer;
			TServer = NULL;
		}
	}
	return TRUE;
}


BOOL DDEManager::IsConnected()
{
	if( hConv )
		return TRUE;
	else
		return FALSE;
}


BOOL DDEManager::IsRegistered()
{
	return IsInAdviseLoop;
}


char * DDEManager::GetErrorText()
{
	return errortext;
}


// Procedure to handle received data
void DDEManager::ReceivedData(HDDEDATA hData)
{
	DWORD len;

	if( hData )
	{
		len = DdeGetData( hData, NULL, 0, 0);
		if( RetData )
		{
			// User-created string: check for overflow
			if( len > LenRetData )
				strcpy(errortext, "data overflow");
		}
		else
		{
			// Dynamically allocate return string
			RetData = new char[len];
			LenRetData = len;
		}
		DdeGetData( hData, RetData, LenRetData, 0);
	}

	// If this data came from advise loop, call client procedure to process it
	if( IsAdviseData && pClientReceiveProc )
	{
		pClientReceiveProc(RetData, ClientThis);
		IsAdviseData = FALSE;
		RetData = NULL;
	}
}


// This is the static callback funtion
HDDEDATA FAR PASCAL _export DDEManager::CallBack(WORD Type, WORD,
	HCONV Conv, HSZ, HSZ, HDDEDATA Data, DWORD, DWORD)
{
	// "this" pointer must be retrieved to receive data
	if( !TServer )
		return NULL;
		
	DDEManager *This = TServer->GetThis(Conv);
	if( !This )
		return NULL;

	switch( Type )
	{
		case XTYP_ADVDATA:
			This->IsAdviseData = TRUE;
			This->ReceivedData(Data);
			return (HDDEDATA) DDE_FACK;

		case XTYP_XACT_COMPLETE:
			This->IsAdviseData = TRUE;
			This->ReceivedData(Data);
			break;

		case XTYP_ERROR:
			strcpy(This->errortext, "critical DDE error");
			// fall-through

		case XTYP_DISCONNECT:
			This->Disconnect();

		default:
			break;
	}
            
	return NULL;
}



