/* cputzc.c  -  Decoding image in ZCompact format to HRAM buffer,
 *              in form of 4 planes. (for UNCHAINED mode).
 *
 * Uses following coding method:
 *
 * Line    ends by a 0, 0 (color code = 0, counter = 0)
 * Picture ends by rows counter.
 *
 *     +7-------------------6+5----------------------0+
 *     | counter (1 till 3)  |  color code (0 - 63)   |
 *     +---------------------+------------------------+
 * or
 *     +7-6+5------------0+    +7---------------------0+
 *     | 0 | color code   |    |  counter              |
 *     +---+--------------+    +-----------------------+
 *
 * Functions:
 *      MputZCompact().
 *
 * See Also: uputc.c, cput.c
 * Portability: BORLANDC
 *                                      (c) erdy 1992
 * $Header: $
 */
#pragma inline
#include "far.h"
#include "vgaprefx.h"
#include "vgadrv.h"
#include "screen.h"

void MputZCompact(char far *image,    /* image data ptr, seg:0 points to colormap */
                  int rows,           /* rows to process */
                  unsigned planewidth,/* width of plane buffer, in paragraphs */
                  unsigned planeintl, /* plane buffer interleaving */
                  unsigned segment    /* address segment of planes buffer */
                  )
{
        char near *index[NPLANES]; /* Reserve space for indeces */
        /* This code cause compiler to produce the following instructions:
         *
         *  push    bp     ; save old bp
         *  mov     bp,sp  ; save old sp
         *  sub     sp,8   ; space for array
         *
         *  So bp points to itself in stack. Then we pushed ds and bp.
         *
         *   | old bp   |    <-- primary bp position    |
         *   ------------                               | Decreasing
         *   | index[0] |    <-- new bp position        | address
         *   ------------                               V
         *   | index[2] |
         *   ------------
         *   | index[3] |
         *   ------------
         *   | index[4] |
         *   ------------
         *   |   ds     |
         *   ------------
         *   | saved bp |    <-- sp
         *   ------------
         *  In order to make bp to points to array, just substract 2.
         */

        /* Charge the offsets, we will use it in back order */
        index[3] = 0;
        index[2] = planeintl;
        index[1] = index[2] + planeintl;
        index[0] = index[1] + planeintl;

        /* Charge the registers */
	_CX = rows;
        _ES = segment;
        _DX = planewidth;
        asm xor di, di;        /* Start from 0 */

	asm push ds;
	asm lds si, image;
	asm cld;
        asm push bp;               /* Save bp */
        asm sub bp, sizeof(int);   /* bp now points to index[0] */
	/*
	 * Scanline loop. CX contains nrows.
	 */
Loop:
                asm push bp;                    /* Save starting bp */
                asm push cx;                    /* Loop counter */

                /* Loop until EOL */
		for (;;) {
			asm lodsb;              /* al <- ds:[si++]; get next byte */
			asm mov bl, al;         /* Save color code */

			asm rol al, 1;          /* Get counter */
                        asm rol al, 1;          /* = shr al, 6 */
                        asm and ax, 3;          /* ah = 0, al &= 3 */
			asm jne short OKCounter;
				asm lodsb;      /* Long counter */
				if (_AL == 0) { /* End of this scanline */
					break;
				}
OKCounter:
			if (_BL == 0) {         /* Zero color code -> skipping */
				asm add di, ax; /* Advance the index */
				continue;
			}
			asm mov cx, ax;         /* Save length */
			/*
			 * Color map table must reside at segment:0.
			 */
                        asm and bx, 0x3F;               /* Mask off color code */
                        asm mov al, [bx];               /* Map color code */
Lplane:
                                asm mov di, [bp];
                                asm stosb;
                                asm mov [bp], di;
                                asm dec bp;
                                asm dec bp;
                                asm jge C1;
                                        asm add bp, NPLANES*sizeof(void near *);
C1:
                                asm loop Lplane;
			continue;
		}
                asm mov ax, es;
                asm add ax, dx;  /* Advance es to next row */
                asm mov es, ax;
                asm xor di, di;

                asm pop cx;
                asm pop di;

        asm loop Loop;

        asm pop bp;
	asm pop ds;
	return;
}