/********************************************************
 * SafeHeap -- Heap handler with safety net		*
 *							*
 * Link this with it's associated routines to trap all	*
 * calls to the following routines:			*
 *							*
 *   	malloc						*
 *	farmalloc					*
 *	free						*
 *	farfree						*
 *							*
 * And if compiled with -P (force C++ compile)		*
 *	new						*
 *	delete						*
 *							*
 * Writes out errors to the log file: MEM.LOG		*
 *							*
 * Note: Only works in the large model.			*
 ********************************************************/
#include <stdio.h>
#include <mem.h>
#include <alloc.h>
#include <dos.h> 
#include <string.h>

#define MAGIC (0xA55A)	/* Magic number used to fill free space */

/*
 * Real versions of the heap routines
 */
void        _Cdecl r_free(void _FAR *__block);
void  _FAR *_Cdecl r_malloc(size_t __size);

void        _Cdecl r_farfree(void far *__block);
void far  * _Cdecl r_farmalloc(unsigned long __nbytes);

FILE *heap_log_file = NULL;	/* File to write the data to */
#define LOG_FILE_NAME "MEM.LOG"	/* Name of the log file */

/********************************************************
 * log -- send a message to the log file		*
 *							*
 * Parameters						*
 *	what -- what we are doing			*
 *	ret_ptr -- address of the thing calling us	*
 *	ptr -- the pointer that we're doing it to	*
 ********************************************************/
static void log(char *what, char **ret_ptr, void *ptr)
{
    unsigned int ret_seg;	/* Segment of the return address */
    static in_log = 0;		/* True if we're in the log routine */

    if (in_log != 0)
        return;

    in_log = 1;
    if (heap_log_file == NULL)
        heap_log_file = fopen(LOG_FILE_NAME, "w");

    if (heap_log_file == NULL)
        return;

    ret_seg = FP_SEG(*ret_ptr);
    fprintf(heap_log_file, "heap error(%s) %04X:%04X %p (%d,%d)\n", what, 
    	ret_seg - _psp - 0x10, FP_OFF(*ret_ptr),
	ptr, heapcheck(), heapchecknode(ptr));
    fflush(heap_log_file);
    in_log = 0;
}
#define NULL_PTR ((long int *)NULL)
extern far long int far _Int0Vector;
/********************************************************
 * null_check -- check for null pointer assignment.	*
 *		If NULL changed, log the problem.	*
 *							*
 * Parameter						*
 *	ret_ptr -- the location of who called us	*
 ********************************************************/
static void null_check(char **ret_ptr)
{
    /* The value of the *NULL */
    static long int null_value = 0L;

    if (null_value == 0L) {
	null_value = *NULL_PTR;
    } else {
	if ((null_value != *NULL_PTR) && (*NULL_PTR != _Int0Vector)) {
	    log("null pointer assignment", ret_ptr, (void *)(*NULL_PTR));
	}
    }
}

/********************************************************
 * malloc -- log and malloc				*
 ********************************************************/
void *malloc(unsigned int size)
{
    /*
     * This little bit of magic gets us a pointer to the return
     * address.
     */
    char **ret_ptr = MK_FP(_SS, _BP+2);
    void *ptr;	/* Pointer to the returned data */

    null_check(ret_ptr);
    ptr = r_malloc(size);
    if ((ptr == NULL) || (heapcheck() == _HEAPCORRUPT))
	log("malloc", ret_ptr, ptr);
    return (ptr);
}

/********************************************************/
void far *farmalloc(unsigned long size)
{
    char **ret_ptr = MK_FP(_SS, _BP+2);
    void *ptr = r_farmalloc(size);

    null_check(ret_ptr);
    if ((ptr == NULL) || (heapcheck() == _HEAPCORRUPT))
	log("farmalloc", ret_ptr, ptr);
    return (ptr);
}

/********************************************************/
void free(void *ptr)
{
    char **ret_ptr = MK_FP(_SS, _BP+2);

    if (heapcheckfree(MAGIC) < 0)
        log("free", ret_ptr, ptr);
    else {
	null_check(ret_ptr);
	if ((heapchecknode(ptr) != _USEDENTRY) || (heapcheck() == _HEAPCORRUPT))
	    log("free", ret_ptr, ptr);
    }
    r_free(ptr);
    heapfillfree(MAGIC);
}

/********************************************************/
void farfree(void *ptr)
{
    char **ret_ptr = MK_FP(_SS, _BP+2);

    if (heapcheckfree(MAGIC) < 0)
        log("farfree", ret_ptr, ptr);
    else {
	null_check(ret_ptr);
	if ((heapchecknode(ptr) != _USEDENTRY) || (heapcheck() == _HEAPCORRUPT))
	    log("farfree", ret_ptr, ptr);
    }

    r_farfree(ptr);
    heapfillfree(MAGIC);
}

#ifdef __BCPLUSPLUS__
#include <stddef.h>
#include <stdlib.h>

void *operator new( size_t size )
{
    char **ret_ptr = MK_FP(_SS, _BP+2);
    void *ptr;	/* Pointer to the returned data */

    if (size == 0)
        size = 1;

    null_check(ret_ptr);
    ptr = r_malloc(size);

    if ((ptr == NULL) || (heapcheck() == _HEAPCORRUPT))
	log("new", ret_ptr, ptr);
    return (ptr);
}

void operator delete( void _FAR *ptr )
{
    char **ret_ptr = MK_FP(_SS, _BP+2);

    if (heapcheckfree(MAGIC) < 0)
        log("delete", ret_ptr, ptr);
    else {
	null_check(ret_ptr);
	if ((heapchecknode(ptr) != _USEDENTRY) || (heapcheck() == _HEAPCORRUPT))
	    log("delete", ret_ptr, ptr);
    }
    r_free(ptr);
    heapfillfree(MAGIC);
}
#endif __BCPLUSPLUS__
