/*
    ST2DOS - Make Atari ST written disks readable under DOS
    Module st2dos.c

    Copyright (C) 1993 Arno Schaefer

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <stdio.h>
#include <dos.h>
#include <bios.h>
#include <alloc.h>

typedef unsigned char byte;
typedef unsigned int word;
typedef unsigned long dword;

typedef enum {false,true} boolean;

#define DISK_INT 0x13

#define RESET_DISK 0
#define READ_SECTOR 2
#define WRITE_SECTOR 3
#define VERIFY_SECTOR 4
#define GET_DISK_TYPE 0x15

/* ----------------------------------------------------------------------- */
/* Copyright notice and version number                                     */
/* ----------------------------------------------------------------------- */

void notice (void)
{
	printf ("\nST2DOS version 1.0, Copyright (C) 1993 Arno Schaefer\n");
	printf ("Makes Atari ST written disks readable under DOS\n");
	printf ("ST2DOS comes with ABSOLUTELY NO WARRANTY, see file COPYING for details\n");
	printf ("This is free software, and you are welcome to redistribute it\n");
	printf ("under certain conditions; again see file COPYING for details.\n\n");
}

/* ----------------------------------------------------------------------- */
/* Error Handling                                                          */
/* ----------------------------------------------------------------------- */

int getx (void)
{
	int character = getch();
	if (character == 3)
	{
		printf ("\n");
		exit (0);
	}
	return (character);
}

void error (char *message)
{
	fprintf (stderr,"\nError: %s!\n",message);
	exit (-1);
}

void warning (char *message)
{
	fprintf (stderr,"\nWarning: %s\n",message);
	fprintf (stderr,"Press any key\n");
	getx();
}

/* ----------------------------------------------------------------------- */
/* BIOS calls                                                              */
/* ----------------------------------------------------------------------- */

int reset_drive (int drive_number)
{
	union REGS regs;

	regs.h.ah = RESET_DISK;
	regs.h.dl = drive_number;
	int86 (DISK_INT,&regs,&regs);
	if (regs.x.cflag) return (-1);
	return 0;
}

int get_disk_type (int drive_number)
{
	union REGS regs;
	regs.h.ah = GET_DISK_TYPE;
	regs.h.dl = drive_number;
	int86 (DISK_INT,&regs,&regs);
	if (regs.x.cflag) return 0;
	return (regs.h.ah);			/* disk type */
}

/* ----------------------------------------------------------------------- */
/* read / write sectors                                                    */
/* ----------------------------------------------------------------------- */

int read_sector (int drive_number,dword head,dword cylinder,dword sector,byte *buffer)
{
	int i;
	boolean done=false;
	for (i=0;i<3;i++)
	{
		if (!biosdisk (READ_SECTOR,drive_number,head,cylinder,sector,1,buffer))
		{
			done=true;
			break;
		}
		reset_drive(drive_number);
	}
	if (!done) return (-1);
	return 0;
}

int verify_sector (int drive_number,dword head,dword cylinder,dword sector,byte *buffer)
{
	if (biosdisk (VERIFY_SECTOR,drive_number,head,cylinder,sector,1,buffer)) return (-1);
	return 0;
}

int write_sector (int drive_number,dword head,dword cylinder,dword sector,byte *buffer)
{
	int i;
	boolean done=false;
	for (i=0;i<3;i++)
	{
		if (!biosdisk (WRITE_SECTOR,drive_number,head,cylinder,sector,1,buffer))
		{
			done=true;
			break;
		}
		reset_drive(drive_number);
	}
	if (!done) return (-1);
	return (verify_sector (drive_number,head,cylinder,sector,buffer));
}

/* ----------------------------------------------------------------------- */
/* Main                                                                    */
/* ----------------------------------------------------------------------- */

void main (void)
{
	int drive_number = 0;
	byte media_byte;
	byte buffer[512];

	int sides,sectors_per_track,sectors_per_fat;
	int fat2side,fat2track,fat2sector;

	notice ();

	if (get_disk_type (1))
	{
		int character='x';
		printf ("Which drive (A/B): ");
		while ((character != 'a') && (character != 'b')) character = getx();
		printf ("%c\n\n",character);
		drive_number = character - 'a';
	}

	while (true)
	{
		printf ("Insert Disk to be converted and press any key\n");

		getx();

		if (read_sector (drive_number,0,0,1,buffer))
			error ("Error reading Bootsector");

		sectors_per_fat = buffer[22];
		sectors_per_track = buffer[24];
		sides = buffer[26];

		if (sectors_per_track > 12)
		{
			warning ("Can't convert High Density Disks");
			continue;
		}

		if (sides == 1) media_byte = 0xf8;
		else if (sides == 2) media_byte = 0xf9;
		else error ("Invalid Bootsector");

		buffer[0] = 0xeb;
		buffer[1] = 0x3c;
		buffer[2] = 0x90;

		buffer[21] = media_byte;

		if (write_sector (drive_number,0,0,1,buffer))
			error ("Error writing Bootsector");

		if (read_sector (drive_number,0,0,2,buffer))
			error ("Error reading FAT 1");
		buffer[0] = media_byte;
		if (write_sector (drive_number,0,0,2,buffer))
			error ("Error writing FAT 1");

		fat2track = (1 + sectors_per_fat) / (sides * sectors_per_track);
		fat2side = (1 + sectors_per_fat - (fat2track * sides * sectors_per_track)) / sectors_per_track;
		fat2sector = 1 + sectors_per_fat - (fat2track * sides * sectors_per_track) - (fat2side * sectors_per_track) + 1;

		if (read_sector (drive_number,fat2side,fat2track,fat2sector,buffer))
			error ("Error reading FAT 2");
		buffer[0] = media_byte;
		if (write_sector (drive_number,fat2side,fat2track,fat2sector,buffer))
			error ("Error writing FAT 2");
	}
}
