/* 

   EXTERN.C example for building a DLL for newLISP, note that all parameters
   and the return value must be 32 bits !

   COMPILE/LINK when using Borland C++ 3.1 or 4.0
   ----------------------------------------------
   bcc -WD -c -ml windir.c
   tlink /Twd /n /c c0dl extern, extern,, mathwl cwl import, extern.def
   rc extern.dll

   
*/


#include <windows.h>
#include <stdlib.h>
#include <string.h>


/* cell types */
#define CELL_NIL 0
#define CELL_TRUE 1
#define CELL_NUMBER 2
#define CELL_FLOAT 3
#define CELL_STRING 4
#define CELL_SYMBOL 5
#define CELL_PRIMITIVE 6
#define CELL_IMPORT_DLL 7
#define CELL_QUOTE 8
#define CELL_LIST 9 
#define CELL_LAMBDA 10
#define CELL_MACRO 11
#define CELL_FREE 0xFF


typedef struct
	{
	WORD type;        /* see above table */
	DWORD aux;        /* string size or lower 4 bytes of double */
	DWORD contents;   /* long integer or high 4 bytes of double or CELL* */
	void * next;      /* pointer to next cell in list or to nilCell */
	} CELL;


/* initialize DLL gets called by Windows */

#pragma argsused
int FAR PASCAL LibMain(
	HANDLE hModule, WORD wDataSeg, WORD cbHeapSize, LPSTR lpszCmdLine)
{
return(1);
}


/* called from Windows when unloading DLL */

#pragma argsused
int FAR PASCAL WEP (int bSystemExit)
{
/* do any cleanup here */
return(1);
}


/* copies the path name of the windows directory and returns its length 
   this is the exported function called from the application 
   
   USAGE:
   
   (dll-import "extern.dll" "getWindowsDir")  ; import function
   (set 'str "                    ")          ; reserve string space   
   (getWindowsDir str (length str))

*/

DWORD FAR PASCAL getWindowsDir(LPSTR pathName, DWORD maxLength)
{
GetWindowsDirectory(pathName, (WORD)maxLength);
return(strlen(pathName));
}



/* 

   This examples shows how to access cells in a list in case
   neither an integer or a string is passed as parameter.
   
   the function (product listA listB) returns the sum of products
   of integers stored in the cells of listA and listB
   
   example in newLISP:
   
   (product '(1 2 3 4) '(5 6 7 8)) ; 1*5 + 2*6 + 3*7 + 4*8
   > 70 
 
*/   


DWORD FAR PASCAL product(CELL * listA, CELL * listB)
{
CELL * cellA;
CELL * cellB;
long sum;

/* check if parameters are really lists, if not return */
if(listA->type != CELL_LIST || listB->type != CELL_LIST)
	return(-1);

/* get the first cell in each list */
cellA = (CELL *)listA->contents;
cellB = (CELL *)listB->contents;
	

/* go through the two lists and sum cells which hold integers */
sum = 0;
while(cellA->type != CELL_NIL && cellB->type != CELL_NIL)
	{
	if(cellA->type != CELL_NUMBER || cellB->type != CELL_NUMBER)
		continue;
		
	sum += (long)cellA->contents * (long)cellB->contents;
	
	
	cellA = cellA->next;
	cellB = cellB->next;
	}

return(sum);
}
	

/*
  same as product() but for floating point numbers. As the result is
  8 bytes (sizeof(double)) we need to pass a float cell for the result.
  
  example in newLISP:
  
  (set 'result 0.0)
  (productFloats '(0.1 0.2 0.3 0.4) '(0.5 0.6 0.7 0.8) result)
  (print result)
  > 0.7
  
*/


DWORD FAR PASCAL productFloats(CELL * listA, CELL * listB, CELL * result)
{
CELL * cellA;
CELL * cellB;
double sum, floatA, floatB;

/* check if parameters are really lists, if not return */
if(listA->type != CELL_LIST || listB->type != CELL_LIST)
	return(-1);

/* get the first cell in each list */
cellA = (CELL *)listA->contents;
cellB = (CELL *)listB->contents;
	

/* go through the two lists and sum cells which hold integers */
sum = 0;
while(cellA->type != CELL_NIL && cellB->type != CELL_NIL)
	{
	if(cellA->type != CELL_FLOAT || cellB->type != CELL_FLOAT)
		continue;
		
	memcpy(&floatA, &cellA->aux, sizeof(double));
	memcpy(&floatB, &cellB->aux, sizeof(double));

	sum += floatA * floatB;
	
	cellA = cellA->next;
	cellB = cellB->next;
	}

/* check if the result cell is a float */
if(result->type != CELL_FLOAT) return(-1);
memcpy(&result->aux, &sum, sizeof(double));

return(0);
}
	
/* end of file */
