;;;
;;;                       oop.lsp
;;;
;;; Object based programming and demonstration how to use
;;; local state variables and optional variables in newLISP
;;;
;;; All 'selectors' (methods) are shared between different
;;; accounts created with 'new'.  The dynamic scoping of newLISP
;;; guarantees that 'arg1 .. arg3', 'name' and 'balance' always refer
;;; to the calling account.
;;;
;;;

;;
;; This is the generic "class" account.
;; It is the only part which gets copied for each new account.
;; All other code is shared between all accounts.
;; Some function args are optional (arg2, arg3). Args arg1 to arg3 
;; have different meaning in different selectors. Variables
;; after the "|" are local state variables which hold their
;; contents even after the function returned.
;;
(define (account selector arg1 arg2 arg3 | name balance)
 (case selector
   ('new (account-new))
   ('cancel (account-cancel))
   ('deposit (account-deposit))
   ('withdraw (account-withdraw))
   ('get-name (account-get-name))
   ('get-balance (account-get-balance))))


;;
;; (account 'new 'account# name value)
;;
;; this makes a new account instance copying the generic account
;; to 'number and initializing 'balance and 'owner.
;;
(define (account-new)
  (set arg1 account)
  (set-state (eval arg1) (list arg2 arg3)) arg3)


;;
;; (account 'cancel 'account#)
;; deletes the account
;;
(define (account-cancel) (set arg1 nil))


;;
;; (account# 'deposit amount)  ; deposit 'amount on 'account#
;;
(define (account-deposit)
  (set 'balance (+ balance arg1)))


;;
;; (account# 'withdraw amount) ; withdraw 'amount from 'account#
;;
(define (account-withdraw)
  (set 'balance (- balance arg1)))


;;
;; (account# 'get-balance)  ; return the accounts balance
;;
(define (account-get-balance) balance)

;;
;; (account# 'get-name)  ; return the owners name
;;
(define (account-get-name) name)

;;
;;
;; this is a sample session:  opening an account and making
;; a deposit and a withdraw then canceling the account
;;


;   > (load "oop.lsp")                        ; loading the account demo
;
;   true                                      ; loading successful
;
;   > (account 'new 'a123 "John Doe" 100)     ; creating a new account
;
;   ()                                        ; previous internal state of a123
;
;   > (a123 'deposit 200)                     ; deposit of 100$
;
;   300                                       ; new balance
;
;   > (a123 'withdraw 50)                     ; withdraw of 50$
;
;   250                                       ; new balance
;
;   > (a123 'get-name)                        ; retrieving internal variable
;
;   "John Doe"                                ; contents of state var 'name
;
;   > (a123 'get-balance)                     ; retrieving internal variable
;
;   250                                       ; contents of state var 'balance
;
;   > (get-state a123)                        ; retrieving internal state 'a123
;
;   ("John Doe" 250)                          ; list of contents of state vars
;
;   > (account 'cancel 'a123 nil)             ; cancel the account
;


;;; end of file ;;;


