/* Process.c -- implementation of processes


Class Process serves as a base class for user processes (co-routines).
*/

/* MSDOS revisions - Michael Murphy 1993
 * 8.3 filenames
 * undo include <nihcl..
 */        
#include "nihclstd.h"
#pragma hdrstop 
 
#pragma -vi
#include "NIHProc.h"
#include "Schedule.h"
#include "ExceptAc.h"
#include "nihclIO.h"

#define  THIS  Process
#define  BASE  Link
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES String::desc()
#define VIRTUAL_BASE_CLASSES

DEFINE_ABSTRACT_CLASS(Process,1,"",NULL);

extern const int NIHCL_STACKOV,NIHCL_BADPRI,NIHCL_RESRUN,NIHCL_RESTERM,NIHCL_INVALIDPS,
	NIHCL_SUSTERM,NIHCL_NOTINIT;

#define DUFF
#define DEBUG_PROCESS
void Process::copyStack(const stackTy* src, stackTy* dst, unsigned /*long*/ count)
{
#ifdef DEBUG_PROCESS
cerr << Scheduler::activeProcess().name()
	<< " : " << hex << (void*)&Scheduler::activeProcess()
	<< "\nProcess::copyStack()\n"
	<< " src=" << hex << /*(unsigned)*/(void*)src
	<< " dst=" << hex << /*(unsigned)*/(void*)dst
	<< " count=" << dec << count << " count=0x" << hex << count
	<< " bytes(0x" << hex << (unsigned)(count*sizeof(stackTy)) << ")"
	<< endl;
#endif
#ifndef DUFF
	while (count--) *dst++ = *src++;
#else
//  Unrolled loop using Duff's Device:
	unsigned long n = (count+7)>>3;
	switch (count & 7) {
		case 0:  do {  *dst++ = *src++;
		case 7:     *dst++ = *src++;
		case 6:     *dst++ = *src++;
		case 5:     *dst++ = *src++;
		case 4:     *dst++ = *src++;
		case 3:     *dst++ = *src++;
		case 2:     *dst++ = *src++;
		case 1:     *dst++ = *src++;
		} while (--n > 0);
	}
#endif
}

Process::Process(const char* name, stackTy* bottom, int priority)
	: process_name(name)
{
	stack_bottom = bottom;
	process_priority = priority;
	process_state = SUSPENDED;
	saved_exception_env_stack_top = ExceptionEnv::lastResort;
	saved_exception_action = new ExceptionActionTbl;
}

Process::Process(stackTy* bottom, int priority)
	: process_name("MAIN")
// MAIN Process constructor
{
	stack_bottom = bottom;
	process_priority = priority;
	process_state = SUSPENDED;
	saved_exception_env_stack_top = /*NULL*/0UL;
	saved_exception_action = /*NULL*/0UL;
	saved_catch_stack_top = /*NULL*/0UL;
	SETPC(0UL);
}

Process::~Process()
{
	terminate();
	delete saved_exception_action;
}

void Process::add()  { Scheduler::addProcess(*this); }

unsigned char Process::priority() const   { return process_priority; }

unsigned char Process::priority(unsigned char newPriority)
{
	unsigned char oldPriority =process_priority;
	if (newPriority > MAXPRIORITY)
		setError(NIHCL_BADPRI,DEFAULT,newPriority,MAXPRIORITY);
	process_priority = newPriority;
	if (newPriority != oldPriority && process_state == RUNNING) {
		Scheduler::runList[oldPriority].remove(*this);
		Scheduler::runList[newPriority].addLast(*this);
	}
	return oldPriority;
}

unsigned Process::capacity() const  { return size(); }

void Process::resume()
{
	switch (process_state) {
		case SUSPENDED: {
			Scheduler::addProcess(*this);
			Scheduler::schedule();
			break;
		}
		case RUNNING: setError(NIHCL_RESRUN,DEFAULT,name(),this);
		case TERMINATED: setError(NIHCL_RESTERM,DEFAULT,name(),this);
		default: setError(NIHCL_INVALIDPS,DEFAULT,name(),this,className(),process_state);
	}
}

void Process::suspend()
{
	switch (process_state) {
		case SUSPENDED: break;
		case RUNNING: {
			process_state = SUSPENDED;
			Scheduler::runList[process_priority].remove(*this);
			Scheduler::runCount--;
			break;
		}
		case TERMINATED: {
			setError(NIHCL_SUSTERM,DEFAULT,name(),this);
		}
		default: {
			setError(NIHCL_INVALIDPS, DEFAULT, name(), this, className(), process_state);
		}
	}
}

void Process::terminate()
{
	switch (process_state) {
		case SUSPENDED: {
			process_state =   TERMINATED;
			return;
		}
		case RUNNING: {
			suspend();
			process_state = TERMINATED;
			Scheduler::schedule();
			return;
		}
		case TERMINATED: break;
		default: {
			setError(NIHCL_INVALIDPS, DEFAULT, name(), this, className(), process_state);
		}
	}
}

int Process::compare(const Object& ob) const // compare process priorities
{
	assertArgSpecies(ob,classDesc,"compare");
	return process_priority - castdown(ob).process_priority;
}

Process::Process(const Process&) { shouldNotImplement("Process"); }

Object* Process::copy() const
{
	shouldNotImplement("copy");
	return 0;
}

void Process::deepenShallowCopy()
{
	shouldNotImplement("deepCopy");
}

unsigned long Process::hash() const { return (unsigned long)this; }

bool Process::isEqual(const Object& ob) const   { return isSame(ob); }
	
void Process::dumpOn(ostream& strm) const
{
	strm  << className() << '[' << process_name
		<< " " << hex << (void*)this
		<< "  pri: " << (int)process_priority
		<< "  state: ";
	switch (process_state) {
		case SUSPENDED: strm << "SUSPENDED"; break;
		case RUNNING:  strm << "RUNNING"; break;
		case TERMINATED: strm << "TERMINATED"; break;
		default: strm << "INVALID";
	}
	strm << endl;
	if (PC() != 0UL) {
		strm  << "  PC: " << hex << (void*)PC()
				<< "  SP: " << hex << (void*)SP()
				<< "  FP: " << hex << (void*)FP() << endl;
	}
	else
		strm  << "  PC: 0x0  SP: ?  FP: ?" << endl;
	strm  << "  stack_bottom: " << hex << /*(unsigned)*/(void*)stack_bottom
			<< "  size:0x"        << hex << size() ;
	strm  << "]\n";
}

void Process::printOn(ostream& strm) const
{
	strm << process_name;
}

void Process::restore()
{
	ExceptionEnv::stackTop = saved_exception_env_stack_top;
	ExceptionActionTbl::active = saved_exception_action;
	Catch::stackTop = saved_catch_stack_top;
}

void Process::save()
{
	saved_exception_env_stack_top = ExceptionEnv::stackTop;
	saved_exception_action = ExceptionActionTbl::active;
	saved_catch_stack_top = Catch::stackTop;
}

Process::Process(OIOifd&) {}

Process::Process(OIOin&) {}

void Process::storer(OIOout&) const
{
	shouldNotImplement("storeOn");
}

void Process::storer(OIOofd&) const
{
	shouldNotImplement("storeOn");
}
