/* MS-DOS version of NIH class library - Michael F. Murphy 4/93 */
/* Unix source: Time.c,v 3.15 92/12/19 15:53:37 */

/*
Function:
	
Provides an object that represents a Time, stored as the number of
seconds since January 1, 1901, GMT.
*/
#include "nihclstd.h"
#pragma hdrstop
                  
#include <iomanip.h>
                  
#include "Date.h"
#include "NIHTime.h"
#include "nihclcon.h"
#include "nihclIO.h"

const unsigned VERSION =2;

#define	THIS	Time
#define	BASE	Object


#include <time.h>
#define TIME_ZONE timezone
#define DST_OBSERVED daylight
#define BASE_CLASSES BASE::desc()
#define MEMBER_CLASSES
#define VIRTUAL_BASE_CLASSES Object::desc()

DEFINE_CLASS(Time,VERSION,"$"__FILE__" "__DATE__" "__TIME__"$",NULL)

extern const int NIHCL_DATERANGE,NIHCL_BADTIME;

static const unsigned long seconds_in_day = 24*60*60L;
static const Date refDate(0L);
static const Date maxDate(31,"Dec",2036);

#if defined(MIBADDEFAULTFCNS) 
Time::Time(const Time& copyfrom): BASE(copyfrom), sec(copyfrom.sec)
{
}

Time& Time::operator=(const Time& rhs)
{
	if(this == &rhs)
		return *this;

	BASE::operator=(rhs);

	sec = rhs.sec;
	return *this;
}
#endif

Time Time::localTime(const Date& date, hourTy h, minuteTy m, secondTy s)
/*
	Return a local Time for the specified Standard Time date, hour, minute,
	and second.
*/
{
	clockTy t = (seconds_in_day*(date-refDate) + 60L*60L*h + 60L*m + s);
	if ( !date.between(refDate,maxDate) || (TIME_ZONE < 0 && (long)t < -TIME_ZONE) )    //(long) MFM
		setError(NIHCL_DATERANGE,DEFAULT,
			date.dayOfMonth(),date.nameOfMonth(),date.year());
	return Time(t);
}

Time::Time()
/*
	Construct a Time for this instant.
*/
{
	sec = time(0);
	sec += 2177452800L;	/* seconds from 1/1/01 to 1/1/70 */
}

Time::Time(hourTy h, minuteTy m, secondTy s, bool dst)
/*
	Construct a Time for today at the specified (local) hour, minute, and
	second.
*/
{
	sec = Time(Date(),h,m,s,dst).sec;
}


Time::Time(const Date& date, hourTy h, minuteTy m, secondTy s, bool dst)
/*
	Construct a Time for the specified (local) Date, hour, minute, and
	second.
*/
{
	sec = localTime(date,h,m,s).sec-3600L;
	if (isDST()) {
		sec += 3600L;
		if (isDST() || dst) sec -= 3600L;
	}
	else {
		sec += 3600L;
		if (isDST()) setError(NIHCL_BADTIME,DEFAULT,
			date.dayOfMonth(),date.nameOfMonth(),date.year(),
			h,m,s,(dst?"DST":""));
	}
	sec += TIME_ZONE;				// adjust to GMT 
}

Time::operator Date() const
/*
	Convert a Time to a local Date
*/
{
//	return Date((int)(localTime().sec/seconds_in_day));	4.2 cc bug
	long daycount = (long)(localTime().sec/seconds_in_day);
	return Date(daycount);
}

bool Time::between(const Time& a, const Time& b) const
{
	return *this >= a && *this <= b;
}

Time::hourTy Time::hour() const
/*
	Return the hour of this Time in local time; i.e., adjust for
	time zone and Daylight Savings Time.
*/
{
	return localTime().hourGMT();
}

Time::hourTy Time::hourGMT() const
/*
	Return the hour of this Time in GMT.
*/
{
	return (Time::hourTy)((sec % 86400L) / 3600L);       // (Time::hourTy) MFM
}

Time Time::beginDST(unsigned year)
/*
	Return the local Standard Time at which Daylight Savings Time
	begins in the specified year.
*/
{
	Time DSTtime(localTime(Date(31,"Mar",year).previous("Sun")+7,2));      
	if (year<=1986) {
		DSTtime = localTime(Date(30,"Apr",year).previous("Sun"),2);
		if (year==1974) DSTtime = localTime(Date(6,"Jan",1974),2);
		if (year==1975) DSTtime = localTime(Date(23,"Feb",1975),2);
	}
	return DSTtime;
}

Time Time::endDST(unsigned year)
/*
	Return the local Standard Time at which Daylight Savings Time
	ends in the specified year.
*/
{
	Time STDtime(localTime(Date(31,"Oct",year).previous("Sun"),2-1));
	return STDtime;
}

bool Time::isDST() const
/*
	Return YES if this local Standard Time should be adjusted
	for Daylight Savings Time.
*/
{
//	unsigned year = Date((unsigned)(this->sec/seconds_in_day)).year();  4.2 cc bug
	long daycount = (long)(this->sec/seconds_in_day);
	unsigned year = Date(daycount).year();
	if (DST_OBSERVED)
		if (*this >= beginDST(year))
			if (*this < endDST(year)) return YES;
	return NO;
}

Time Time::localTime() const
/*
	Adjusts this GM Time for local time zone and Daylight Savings Time.
*/
{
	Time local_time(sec-TIME_ZONE);
	if (local_time.isDST()) local_time.sec += 3600L;
	return local_time;
}

Time::minuteTy Time::minute() const
/*
	Return the minute of this Time in local time; i.e., adjust
	for time zone and Daylight Savings Time.
*/
{
	return localTime().minuteGMT();
}

Time::minuteTy Time::minuteGMT() const
/*
	Return the minute of this Time in GMT.
*/
{
	return (Time::minuteTy)(((sec % 86400L) % 3600L) / 60L);		// (Time::minuteTy) MFM
}

Time::secondTy Time::second() const
/*
	Return the second of this Time.
*/
{
	return (Time::secondTy)(((sec % 86400L) % 3600L) % 60L);		// (Time::secondTy ) MFM
}

Time Time::max(const Time& t) const
{
	if (t < *this) return *this;
	return t;
}

Time Time::min(const Time& t) const
{
	if (t > *this) return *this;
	return t;
}

int Time::compare(const Object& ob) const
{
	assertArgSpecies(ob,classDesc,"compare");
	register clockTy t = castdown(ob).sec;
	if (sec < t) return -1;
	if (sec > t) return 1;
	return 0;
}

void Time::deepenShallowCopy()	{}

unsigned long Time::hash() const	{ return sec; }

bool Time::isEqual(const Object& ob) const
{
	return ob.isSpecies(classDesc) && *this==castdown(ob);
}

const Class* Time::species() const { return &classDesc; }
                      
#pragma warning (disable: 4270)   
//warning C4270: 'initializing' : do not initialize a non-const 'class ::__SMANIP_int __far &' with a non-lvalue 'class ::__SMANIP_int ' function return
                   
void Time::printOn(ostream& strm) const
{
	register unsigned hh = hour();
	Date(*this).printOn(strm);
 	strm << ' ' << ((hh <= 12) ? hh : hh-12) << ':';
 	strm << setfill('0') << setw(2) << minute() << ':';
 	strm << setfill('0') << setw(2) << second() << ' ';
	if (hh < 12) strm << "am";
	else strm << "pm";
}
#pragma warning(default: 4270)

Time::Time(OIOin& strm)
	: BASE(strm)
{
	unsigned long usec;
	strm >> sec >> usec;
}

void Time::storer(OIOout& strm) const
{
	BASE::storer(strm);
	strm << sec << 0l /* for future usec */;
}


Time::Time(OIOifd& fd)
	: BASE(fd)
{
	unsigned long usec;
	fd >> sec >> usec;
}

void Time::storer(OIOofd& fd) const
{
	BASE::storer(fd);
	fd << sec << 0l /* for future usec */;
}
