/*
   Play/R interface module.
   Copyright 1992, Kevin Weiner, All rights reserved. 
*/

#include <dos.h>
#include <string.h>

#if defined(__BORLANDC__)
#include <mem.h>
#else
#define asm _asm
#endif

#define LOCAL
#include "mlib.h"

typedef char *cptr;

#define fcnPause     0
#define fcnPlay      1
#define fcnPopup     2
#define fcnRewind    3
#define fcnLoadPlay  4
#define fcnPlayStat  5
#define fcnLoadStat  6
#define fcnDoneStat  7
#define fcnLoad      8
#define fcnQuiet     9
#define fcnPopEna    10
#define fcnVolume    11
#define fcnReset     12
#define fcnTimeMode  13
#define fcnGetChan   14
#define fcnSetChan   15
#define fcnSetPos    16
#define fcnSkipSong  17
#define fcnLoopMode  18

#define fcnGetName   20
#define fcnSendShort 21
#define fcnSendLong  22
#define fcnCheckInp  23
#define fcnGetInput  24
#define fcnGetTime   25
#define fcnGetCID    26
#define fcnGetEntry  27
#define fcnRemove    99

char MidiDriverLoaded;
int MID;

ftype2 midiPutByte;
ftype4 midiInputReady;
ftype5 midiGetByte;
ftype1 midiClearInput;
ftype3 msTimer;
ftype6 midiGetMessage;
ftype7 midiResend;
ftype8 midiPutMessage;

int tessmpx = 0x5453;                   /* TesSeRact multiplex id */
int idnum;                              /* Play/R id number */
#if defined(__BORLANDC__)
char playrid[8] = "Play/R  ";           /* PlayR id string */
char param [64];                        /* Parameter block */
#else
char far playrid[8] = "Play/R  ";       /* PlayR id string */
char far param [64];                    /* Parameter block */
#endif
union REGS reg;                         /* Machine registers */
struct SREGS sreg;


void CallPlayR(char fcn)

  /*  Call Play/R using the contents of the byte array "param".
      Byte 0, passed in fcn, the function code, and the rest is
      any required data.  */


  {
    long temp;

    param[0] = fcn;
    reg.x.ax = tessmpx;                /* ax = Tess multiplex id */
#if defined(__BORLANDC__)
    sreg.es = FP_SEG(param);           /* es = parameter segment */
    reg.x.di = FP_OFF(param);          /* di = parameter offset */
#else
    temp = (long) param;
    sreg.es = temp >> 16;              /* es = parameter segment */
    reg.x.di = temp & 0xffff;          /* di = parameter offset */
#endif
    reg.x.cx = idnum;                  /* cx = PlayR id number */
    reg.x.bx = 0x20;                   /* bx = Tess call user function */

    asm {
      push di
      push si
    }

    int86x(0x2f, &reg, &reg, &sreg);   /* Call int 2fh */

    asm {
      pop si
      pop di
    }
  }


void mfPause()

  {
    CallPlayR(fcnPause);
  }


void mfContinue()

  {
    CallPlayR(fcnPlay);
  }


void mfPopup()

  {
    CallPlayR(fcnPopup);
  }


void mfRewind()

  {
    CallPlayR(fcnRewind);
  }


byte mfPlay(char *name)

  {
    strcpy(&param[1], name);
    CallPlayR(fcnLoadPlay);
    return param[0];
  }


void mfSongStat(char *playing, char *done, long *position,
                byte *songcount, byte *cursong)

  {
    CallPlayR(fcnPlayStat);
    *playing = param[0];
    *done = param[1];
    memcpy(position, &param[2],  4);
    *songcount = param[6];
    *cursong = param[7];
  }


void mfFileStat(byte *stat, char *name)

  {
    CallPlayR(fcnLoadStat);
    *stat = param[0];
    strcpy(name, &param[1]);
  }


byte mfLoad(char *name)

  {
    strcpy(&param[1], name);
    CallPlayR(fcnLoad);
    return param[0];
  }


void mfQuiet()

  {
    CallPlayR(fcnQuiet);
  }


void mfPopEnable(char stat)

  {
    param[1] = stat;
    CallPlayR(fcnPopEna);
  }


void mfVolume(int adjust)

  {
    param[1] = (char) adjust;
    CallPlayR(fcnVolume);
  }


void midiReset(int dev)

  {
    param[1] = dev;
    CallPlayR(fcnReset);
  }


void mfTimeMode(byte mode)

  {
    param[1] = mode;
    CallPlayR(fcnTimeMode);
  }


void mfGetChan(int datatype, channels *chan)

  {
    param[1] = datatype;
    CallPlayR(fcnGetChan);
    memcpy(chan, &param[2], 16);
  }


void mfSetChan(int datatype, channels *chan)

  {
    param[1] = datatype;
    memcpy(&param[2], chan, 16);
    CallPlayR(fcnSetChan);
  }

void mfSetPos(long time)

  {
    memcpy(&param[1], &time, 4);
    CallPlayR(fcnSetPos);
  }

void mfSkipSong(byte n)

  {
    param[1] = n;
    CallPlayR(fcnSkipSong);
  }

void mfLoopMode(byte n)

  {
    param[1] = n;
    CallPlayR(fcnLoopMode);
  }

int midiDevName(int dev, char *devname, char *devdesc)

  {
    param[1] = dev;
    CallPlayR(fcnGetName);
    if (param[1] == 0)
      {
	*devname = NULL;
	*devdesc = NULL;
      }
    else
      {
	memcpy(devname, &param[2], 3);
	devname[3] = 0;
	memcpy(devdesc, &param[5], 20);
	devdesc[20] = 0;
      }
    return param[1];
  }


void midiPutBuffer(int dev, char far *buf, unsigned int len)

  {
    int i;

    param[1] = dev;
    memcpy(&param[2], &len, 2);
    memcpy(&param[4], &buf, 4);
    for (i = 8; i < 12; i++) param[i] = 0;
    CallPlayR(fcnSendLong);
  }


void midiPutBuffer1(int dev, char *buf, unsigned int len)

  {
    word  i;
    int   x;

    for (i = 0; i < len; i++)
    {
      x = buf[i];
      midiPutByte(dev, x);
    };
  }


word midiGetBuffer(int dev, char *buf, word max)

  {
    word  i;
    int   x;

    for (i=0; midiInputReady(dev) && (i < max); i++)
    {
      midiGetByte(dev, &x);
      buf[i] = x;
    };
    return i;
  }


void midiRemove()

  {
    CallPlayR(fcnRemove);
  }


void BindDriver()

  {
    CallPlayR(fcnGetEntry);
    memcpy(&midiPutByte, &param[1], 4);
    memcpy(&midiInputReady, &param[5], 4);
    memcpy(&midiGetByte, &param[9], 4);
    memcpy(&midiClearInput, &param[13], 4);
    memcpy(&msTimer, &param[17], 4);
    memcpy(&midiGetMessage, &param[21], 4);
    memcpy(&midiResend, &param[25], 4);
    memcpy(&midiPutMessage, &param[29], 4);
  }


int CheckRes()

/*  Check Play/R is loaded - returns id number if found, else -1  */

  {
    long temp;

    reg.x.ax = tessmpx;                /* ax = Tess int 2fh muliplex id */
#if defined(__BORLANDC__)
    sreg.ds = FP_SEG(playrid);         /* ds = id string segment */
    reg.x.si = FP_OFF(playrid);        /* si = id string offset */
#else
    temp = (long) playrid;
    sreg.ds = temp >> 16;              /* es = parameter segment */
    reg.x.si = temp & 0xffff;          /* di = parameter offset */
#endif
    reg.x.cx = 0;                      /* cx = Tess id counter - must be 0 */
    reg.x.bx = 0;                      /* bx = Tess check resident function */

    asm {
      push di
      push si
    }

    int86x(0x2f, &reg, &reg, &sreg);   /* Call int 2fh */

    asm {
      pop si
      pop di
    }

    if (reg.x.ax == 0xffff)
      idnum = reg.x.cx;                /* Found - return tsr id */
    else
      idnum = -1;                      /* Not loaded */
    return idnum;
  }


char midiInit()

  {
    MidiDriverLoaded = CheckRes() >= 0;
    if (MidiDriverLoaded)
      {
	CallPlayR(fcnGetCID);
	MID = param[1];
	BindDriver();
      }
    return MidiDriverLoaded;
  }
