// ===================================================================
// algs.cpp
//	Illustrate a variety of algebraic surfaces.
// Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================

#include "oort.h"
#include "world.h"
#include "colors.h"
#include "utils.h"

void
PopulateWorld(World& world)
{
    world.SetAmbientLight(RGBColor(1/3.));
    world.SetDepthLimit(3);
    world.SetOutputFile("algs.raw");

    // Important: set threshold of no-intersection to a
    // much smaller value than the default.
    Limits::Threshold = 0.01;
    Limits::Small = 0.001;

    world.SetViewerParameters(Vector3D(0), Vector3D(0, 0, -8), Vector3D(0, 1, 0));
    world.SetScreenWidth(4);
    world.SetScreenHeight(3);
    world.SetScreenDistance(2);

    world.SetBackgroundColor(RGBColor(0));

    world.AddLight(new PointLight(Vector3D(2, 0, -2.5), RGBColor(1/3.), 1, 0, 0));
    world.AddLight(new PointLight(Vector3D(-2, 2, -2), RGBColor(1/3.), 1, 0, 0));

    HallSurface *surf = new HallSurface;
    surf->SetBumpMap(new TwoSided);
    surf->SetAmbient(new PureColor(0, 0.5, 1));
    surf->SetDiffuse(new PureColor(0, 0.5, 1));
    surf->SetSpecular(new PureColor(0.8), 100);

    Object3D *addme = new CSGIntersection(MakePiriform(surf), new Plane(-1, 0, 0, -0.1, surf));
    addme->ApplyTransform(RotationZMatrix(-90*M_PI/180));
    addme->ApplyTransform(RotationXMatrix(30*M_PI/180));
    addme->ApplyTransform(TranslationMatrix(Vector3D(-3, 0, 0)));
    world.AddObject(addme);

    addme = new CSGIntersection(new Sphere(Vector3D(0), 2, surf), MakeLemniscate(surf));
    addme->ApplyTransform(RotationYMatrix(-25*M_PI/180));
    addme->ApplyTransform(RotationZMatrix(-40*M_PI/180));
    addme->ApplyTransform(TranslationMatrix(Vector3D(-6, 0, 0)));
    world.AddObject(addme);

    addme = new CSGIntersection(new Sphere(Vector3D(0), 1.1, surf), MakeSteiner(surf));
    addme->ApplyTransform(RotationYMatrix(45*M_PI/180));
    addme->ApplyTransform(TranslationMatrix(Vector3D(6, 0, 0)));
    world.AddObject(addme);

    Object3D *qcyl = MakeQuarticCylinder(surf);
    qcyl->ApplyTransform(RotationYMatrix(45*M_PI/180));
    addme = new CSGIntersection(new Sphere(Vector3D(0), 4, surf), qcyl);
    addme->ApplyTransform(TranslationMatrix(Vector3D(3, 0, 0)));
    world.AddObject(addme);

    Object3D *kummer = MakeKummer(surf);
    kummer->ApplyTransform(RotationYMatrix(25*M_PI/180));
    kummer->ApplyTransform(TranslationMatrix(Vector3D(0, 0, 5)));
    world.AddObject(kummer);
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}

