// ===================================================================
// CSGDEMO.CPP
//	Illustrate constructive solid geometry.
// Copyright (C) 1993 by Nicholas Wilt.  All rights reserved.
// ===================================================================


#include "oort.h"
#include "world.h"
#include "colors.h"
#include "utils.h"

void
PopulateWorld(World& world)
{
    world.SetDepthLimit(3);
    world.SetOutputFile("csgdemo.raw");

    world.SetViewerParameters(Vector3D(0), Vector3D(0, 5, 12), Vector3D(0, 1, 0));
    world.SetScreenWidth(4);
    world.SetScreenHeight(3);
    world.SetScreenDistance(2);

    world.SetBackgroundColor(RGBColor(0));

    world.AddLight(new PointLight(Vector3D(0, 10, 20), RGBColor(1./3), 1, 0, 0));
    world.AddLight(new DirectLight(Vector3D(0, 1, 0), RGBColor(1./3)));
    world.AddLight(new PointLight(Vector3D(0, 5, 12), RGBColor(1./3), 1, 0, 0));

    Texture *ChessBoard = new Checkerboard(5, 5,
					   new PureColor(0, 0, 0.5),
					   new PureColor(0, 0.5, 0.5),
					   RotationXMatrix(-90*M_PI/180));

    HallSurface *surf = new HallSurface;
    surf->SetBumpMap(new TwoSided);
    surf->SetDiffuse(new PureColor(Blue));

    HallSurface *sph2surf = new HallSurface;
    surf->SetBumpMap(new TwoSided);
    sph2surf->SetDiffuse(new PureColor(Red));

    {
	Sphere sph1(Vector3D(0), 2, surf);
	Sphere sph2(Vector3D(1.5, 0, 0), 2, sph2surf); 
	Aggregate *addme = new Aggregate;
	addme->AddObject(sph1);
	addme->AddObject(sph2);
	addme->ApplyTransform(TranslationMatrix(Vector3D(-4, 0, 0)));
        world.AddObject(addme);
	CSGIntersection *inter = new CSGIntersection(sph1.Dup(), sph2.Dup());
	inter->ApplyTransform(TranslationMatrix(Vector3D(4, 0, 0)));
        world.AddObject(inter);
    }

    {
	Object3D *right = new CSGIntersection(new Plane(Vector3D(0, 1, 0), -2, surf), new Plane(Vector3D(0, -1, 0), -2, surf));
	right = new CSGIntersection(right, MakeCylinder(0.5, 0.75, surf));
	right->ApplyTransform(RotationXMatrix(90*M_PI/180) * RotationYMatrix(30*M_PI/180));
	Object3D *left = new Sphere(Vector3D(0), 1.5, surf);

	Aggregate *nodiff = new Aggregate;
	nodiff->AddObject(left->Dup());
	nodiff->AddObject(right->Dup());
	nodiff->ApplyTransform(TranslationMatrix(Vector3D(-5, 5, 0)));
	world.AddObject(nodiff);

	Object3D *diff = new CSGDifference(left, right);
        diff->ApplyTransform(TranslationMatrix(Vector3D(5, 5, 0)));
	world.AddObject(diff);
    }

    {
	Aggregate *addme = new Aggregate;
	addme->AddObject(new Sphere(Vector3D(0), 1, MakeGlass()));
	addme->AddObject(new Sphere(Vector3D(-0.5, 0, 0), 1, MakeGlass()));
	addme->ApplyTransform(TranslationMatrix(Vector3D(-3, 0, 7)));
	world.AddObject(addme);
	Sphere *sph1 = new Sphere(Vector3D(0), 1, MakeGlass());
	Sphere *sph2 = new Sphere(Vector3D(-0.5, 0, 0), 1, MakeGlass());
	CSGUnion *uni = new CSGUnion(sph1, sph2);
        uni->ApplyTransform(TranslationMatrix(Vector3D(3, 0, 7)));
        world.AddObject(uni);
    }

    HallSurface *checks = new HallSurface;
    checks->SetDiffuse(/*new PureColor(CornflowerBlue)*/ChessBoard);
    world.AddObject(new Plane(Vector3D(0, 1, 0), 5.0, checks));
}

int
main(int argc, char *argv[])
{
    World TheWorld;

    // Say hello
    cout << "OORT: The Object-Oriented Ray Tracer  Version 1.0\n";
    cout << "Copyright (C) 1992 by Nicholas Wilt.  All rights reserved.\n\n";

    // Allocate global noise sources.
    if (! GlobalNoise::Noise)
	GlobalNoise::Noise = new PerlinNoise;
    if (! GlobalNoise::Waves)
    	GlobalNoise::Waves = new WaveSource(10, GlobalNoise::Noise);
    if (! GlobalNoise::Freqs)
    	GlobalNoise::Freqs = new FreqSource(10, GlobalNoise::Noise);

    PopulateWorld(TheWorld);

    // Parse the command line; options given on the command line subsume
    // stuff specified in the input file.
    TheWorld.ParseCommandLine(argc, argv);

    // Write RAW output file.
    TheWorld.RayTrace();

    // Report on statistics gathered.
    Report(cout);
    if (Statistics::AppendTo) {
    	ofstream app(Statistics::AppendTo, ios::ate);
    	if (! (! app)) {
    	    Report(app);
    	    app.close();
    	}
    }

    // Delete global noise sources.
    delete GlobalNoise::Noise;
    delete GlobalNoise::Waves;
    delete GlobalNoise::Freqs;

    return 0;
}
