// ===================================================================
// heap.h
//	Template-based implementation of heap-based priority queue
//	class.	Calls a function called Compare to determine
//	how a given pair of T's are ordered.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

template<class T> 
class Heap {
public:
    Heap();
    Heap(const Heap&);
    Heap& operator=(const Heap&);
    ~Heap();

    void Insert(const T& x);
    T Min() const;
    T ExtractMin();
    int N() const;
private:
    T *elms;
    int n;
    int maxsize;

    void SiftUp(int i);
    void SiftDown();
};

template <class T>
Heap<T>::Heap()
{
    elms = new T[2];
    n = 0;
    maxsize = 2;
}

// Copy constructor
template <class T> 
Heap<T>::Heap(const Heap<T>& x)
{
    elms = new T[n = x.n];
    for (int i = 0; i < n; i++)
	elms[i] = x.elms[i];
    maxsize = n;
}

// Copy operator
template <class T> 
Heap<T>&
Heap<T>::operator=(const Heap<T>& x)
{
    if (elms)
	delete[] elms;
    maxsize = n = x.n;
    if (! n)
	maxsize = 2;
    elms = new T[maxsize];
    for (int i = 0; i < x.n; i++)
	elms[i] = x.elms[i];
    return *this;
}

template<class T>
Heap<T>::~Heap()
{
    delete[] elms;
}

template<class T>
void
Heap<T>::Insert(const T& x)
{
    if (++n >= maxsize) {
	T *temp = new T[maxsize <<= 1];
	for (int i = 0; i < n-1; i++)
	    temp[i] = elms[i];
	delete[] elms;
	elms = temp;
    }
    elms[n-1] = x;
    SiftUp(n-1);
}

template<class T>
T
Heap<T>::Min() const
{
    return elms[0];
}

template<class T>
T
Heap<T>::ExtractMin() 
{
    T ret = elms[0];		// Here's the return value
    if (--n > 0) {		// Now restore the heap property
        elms[0] = elms[n];	// Set top to last elm in heap
        SiftDown();		//   and sift it down
    }
    return ret;
}

template<class T>
int
Heap<T>::N() const
{
    return n;
}

template <class T>
void
Heap<T>::SiftUp(int i)
{
    while (i) {
	int p = (i-1) >> 1;
	T temp;
	if (Compare(elms[p], elms[i]) < 0)
	    break;
	temp = elms[i];
	elms[i] = elms[p];
	elms[p] = temp;
	i = p;
    }
}

template <class T>
void
Heap<T>::SiftDown()
{
    int i = 0;
    int c;

    while ( (c = i+i+1) < n) {
	T temp;
	if (c+1 < n)
	    c += Compare(elms[c+1], elms[c]) < 0;
	if (Compare(elms[i], elms[c]) < 0)
	    break;
	temp = elms[i];
	elms[i] = elms[c];
	elms[c] = temp;
	i = c;
    }
}
